/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL ES-CM 1.1 plugin
 *
 * Copyright © 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#ifndef __PGM_GLES_CONTEXT_H__
#define __PGM_GLES_CONTEXT_H__

/* pgmglescontext.h and pgmglesbackend.h/pgmglesviewport.h include eachother */
typedef struct _PgmGlesContext            PgmGlesContext;
typedef struct _PgmGlesContextTask        PgmGlesContextTask;
typedef struct _PgmGlesContextProcAddress PgmGlesContextProcAddress;

#include "pgmglestexture.h"
#include "pgmglesbackend.h"
#include "pgmglesviewport.h"
#include "pgmglesdefs.h"

G_BEGIN_DECLS

/* Context locking */
#define PGM_GLES_CONTEXT_LOCK(context)   (g_mutex_lock (context->mutex))
#define PGM_GLES_CONTEXT_UNLOCK(context) (g_mutex_unlock (context->mutex))

/* Type casting */
#define PGM_GLES_CONTEXT(obj)      ((PgmGlesContext *) (obj))
#define PGM_GLES_CONTEXT_TASK(obj) ((PgmGlesContextTask *) (obj))

/* Task handler */
typedef void (*PgmGlesContextTaskFunc) (PgmGlesContext *context, gpointer data);

/* Task types */
typedef enum {
  /* Various */
  PGM_GLES_CONTEXT_PROJECTION = 0, /* Update viewport and projection matrix */
  PGM_GLES_CONTEXT_VISIBILITY,     /* Update window visibility */
  PGM_GLES_CONTEXT_ALPHA_BLENDING, /* Update alpha blending state */
  PGM_GLES_CONTEXT_READ_PIXELS,    /* Read frame buffer */
  /* Texture related */
  PGM_GLES_CONTEXT_GEN_TEXTURE,    /* Generate texture */
  PGM_GLES_CONTEXT_CLEAN_TEXTURE,  /* Clean up texture */
  PGM_GLES_CONTEXT_UPLOAD_TEXTURE, /* Upload texture buffer */
  PGM_GLES_CONTEXT_UPDATE_TEXTURE, /* Update texture parameters */
  PGM_GLES_CONTEXT_FREE_TEXTURE,   /* Free texture */
  /* Task types count */
  PGM_GLES_CONTEXT_LAST_TASK
} PgmGlesContextTaskType;

/* Context */
struct _PgmGlesContext {
  PgmGlesViewport *glesviewport;

  /* PgmContext structure access mutex */
  GMutex *mutex;

  /* Rendering thread */
  GThread      *render_thread;
  GMainContext *render_context;
  GMainLoop    *render_loop;

  /* Immediate event source */
  gint        immediate_fd[2];
  GIOChannel *immediate_out;
  GIOChannel *immediate_in;
  guint       immediate_tag;

  /* Task queues */
  GList *immediate_task;
  GList *deferred_task;

  /* Auto-update */
  GMutex   *update_mutex;
  gboolean  auto_updated;
  GTimeVal  update_timestamp;
  guint     update_tag;
  guint     requested_fps;

  /* Initialization lock */
  GMutex   *init_mutex;
  GCond    *init_cond;
  gboolean  initialized;

  /* Physical screen informations */
  gint screen_width_mm;
  gint screen_height_mm;

  /* Limits */
  gint max_texture_size;
  gint max_texture_units;

  /* Pixel store modes */
  gint row_length;
  gint skip_rows;
  gint skip_pixels;

  /* Renderer informations */
  const gchar *vendor;
  const gchar *version_string;
  const gchar *renderer;
  const gchar *extensions;
  gfloat       version;

  /* Features */
  gulong feature_mask;

  /* OpenGL ES backend */
  PgmGlesBackend *backend;

  /* OpenGL ES function pointers */
  PgmGlesContextProcAddress *gles;

  /* Frame rate per second */
  GTimeVal fps_tick_time;
  guint    fps;

  /* Task handlers */
  PgmGlesContextTaskFunc task_func[PGM_GLES_CONTEXT_LAST_TASK];
};

/* Task */
struct _PgmGlesContextTask {
  /* Task type */
  PgmGlesContextTaskType type;
  /* Opaque pointer depending on task type */
  gpointer data;
};

/* OpenGL ES 1.1 function pointers, used for extensions loading */
struct _PgmGlesContextProcAddress {
  /* OpenGL ES 1.1 core functions */
  pgm_gles_enable               enable;
  pgm_gles_disable              disable;
  pgm_gles_get_error            get_error;
  pgm_gles_get_string           get_string;
  pgm_gles_enable_client_state  enable_client_state;
  pgm_gles_disable_client_state disable_client_state;
  pgm_gles_vertex_pointer       vertex_pointer;
  pgm_gles_color_pointer        color_pointer;
  pgm_gles_tex_coord_pointer    tex_coord_pointer;
  pgm_gles_draw_arrays          draw_arrays;
  pgm_gles_draw_elements        draw_elements;
  pgm_gles_color_4f             color_4f;
  pgm_gles_blend_func           blend_func;
  pgm_gles_clear                clear;
  pgm_gles_clear_color          clear_color;
  pgm_gles_matrix_mode          matrix_mode;
  pgm_gles_push_matrix          push_matrix;
  pgm_gles_pop_matrix           pop_matrix;
  pgm_gles_load_identity        load_identity;
  pgm_gles_load_matrix_f        load_matrix_f;
  pgm_gles_viewport             viewport;
  pgm_gles_flush                flush;
  pgm_gles_finish               finish;
  pgm_gles_frustum_f            frustum_f;
  pgm_gles_ortho_f              ortho_f;
  pgm_gles_scale_f              scale_f;
  pgm_gles_translate_f          translate_f;
  pgm_gles_rotate_f             rotate_f;
  pgm_gles_hint                 hint;
  pgm_gles_shade_model          shade_model;
  pgm_gles_read_pixels          read_pixels;
  pgm_gles_gen_textures         gen_textures;
  pgm_gles_delete_textures      delete_textures;
  pgm_gles_bind_texture         bind_texture;
  pgm_gles_tex_image_2d         tex_image_2d;
  pgm_gles_tex_sub_image_2d     tex_sub_image_2d;
  pgm_gles_tex_parameter_i      tex_parameter_i;
  pgm_gles_active_texture       active_texture;
  pgm_gles_get_integer_v        get_integer_v;
  pgm_gles_get_float_v          get_float_v;

  /* OpenGL ES 1.1 extension functions */
  /* ... */
};

/* Create a new context */
PgmGlesContext *pgm_gles_context_new          (PgmGlesViewport *glesviewport);

/* Free a context */
void            pgm_gles_context_free         (PgmGlesContext *context);

/* Create a new task */
PgmGlesContextTask *pgm_gles_context_task_new (PgmGlesContextTaskType type,
                                               gpointer data);

/* Free a task */
void pgm_gles_context_task_free               (PgmGlesContextTask *task);

/* Request a rendering update */
void pgm_gles_context_update                  (PgmGlesContext *context);

/* Push a new task on the immediate queue */
void pgm_gles_context_push_immediate_task     (PgmGlesContext *context,
                                               PgmGlesContextTask *task);

/* Push a new task on the deferred queue */
void pgm_gles_context_push_deferred_task      (PgmGlesContext *context,
                                               PgmGlesContextTask *task);

/* Remove tasks with the given data from the immediate and deferred queues */
void pgm_gles_context_remove_tasks_with_data  (PgmGlesContext *context,
                                               gconstpointer data);

G_END_DECLS

#endif /* __PGM_GLES_CONTEXT_H__ */
