/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL plugin
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Mirco Müller <macslow@bangang.de>
 */

/*
 * PgmGlText uses the Pango text rendering pipeline to generate a pixmap with
 * the size of the drawable and then bind it as a texture. Each time a property
 * is changed, the pixmap is updated. When the drawable size is changed, the
 * pixmap is reallocated to this new one and then updated.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <math.h>
#include "pgmgltext.h"

#ifndef HAVE_PANGO_1_16
#ifndef G_OS_WIN32
#include <locale.h>
#endif
#endif

/* This is the offset such that (0xFF << PGM_n_OFFSET) is the mask to access the
 * nth byte of a guint32 */
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
#define BYTE_1_OFFSET 0
#define BYTE_2_OFFSET 8
#define BYTE_3_OFFSET 16
#define BYTE_4_OFFSET 24
#else /* G_BIG_ENDIAN */
#define BYTE_1_OFFSET 24
#define BYTE_2_OFFSET 16
#define BYTE_3_OFFSET 8
#define BYTE_4_OFFSET 0
#endif

/* These are the offsets to shift to access the alpha, red, green and blue bytes
 * in a native endian ARGB value (such as what is given by Cairo) */
#define ARGB_A_OFFSET 24
#define ARGB_R_OFFSET 16
#define ARGB_G_OFFSET 8
#define ARGB_B_OFFSET 0

/* These are the A,R,G,B component byte masks for a native endian ARGB value
 * (such as what is given by Cairo) */
#define ARGB_A_MASK 0xFF << ARGB_A_OFFSET
#define ARGB_R_MASK 0xFF << ARGB_R_OFFSET
#define ARGB_G_MASK 0xFF << ARGB_G_OFFSET
#define ARGB_B_MASK 0xFF << ARGB_B_OFFSET

GST_DEBUG_CATEGORY_STATIC (pgm_gl_text_debug);
#define GST_CAT_DEFAULT pgm_gl_text_debug

static PgmGlDrawableClass *parent_class = NULL;

/* Private methods */

/* Gets the projected height of the drawable on the viewport at z=0 */
static gfloat
get_projected_height (PgmGlDrawable *gldrawable,
                      gfloat canvas_height)
{
  PgmGlViewport *glviewport = gldrawable->glviewport;
  PgmViewport *viewport = PGM_VIEWPORT (glviewport);
  gfloat height;

  GST_OBJECT_LOCK (viewport);
  height = (canvas_height * glviewport->projected_h) / glviewport->canvas_h;
  GST_OBJECT_UNLOCK (viewport);

  return height;
}

/* Gets the projected width of the drawable on the viewport at z=0 */
static gfloat
get_projected_width (PgmGlDrawable *gldrawable,
                     gfloat canvas_width)
{
  PgmGlViewport *glviewport = gldrawable->glviewport;
  PgmViewport *viewport = PGM_VIEWPORT (glviewport);
  gfloat width;

  GST_OBJECT_LOCK (viewport);
  width = (canvas_width * glviewport->projected_w) / glviewport->canvas_w;
  GST_OBJECT_UNLOCK (viewport);

  return width;
}

/* Sets the position of the text taking care of aligning the vertices to the
 * pixel boundaries. It ensures clean text rendering without any artifact
 * implied by the sub-pixels rasterization process. */
static void
set_position (PgmGlDrawable *gldrawable)
{
  PgmGlViewport *glviewport = gldrawable->glviewport;
  PgmGlText *gltext = PGM_GL_TEXT (gldrawable);
  gfloat offset_x = glviewport->pixel_offset_x;
  gfloat offset_y = glviewport->pixel_offset_y;
  gfloat bound_width = floorf (gldrawable->width / offset_x) * offset_x;
  gfloat bound_height = floorf (gldrawable->height / offset_y) * offset_y;
  gfloat bound_x = floorf (gldrawable->x / offset_x) * offset_x;
  gfloat bound_y = floorf (gldrawable->y / offset_y) * offset_y;

  gltext->vertex[0] = bound_x;
  gltext->vertex[1] = bound_y;
  gltext->vertex[2] = gldrawable->z;
  gltext->vertex[3] = bound_x + bound_width;
  gltext->vertex[4] = bound_y;
  gltext->vertex[5] = gldrawable->z;
  gltext->vertex[6] = gltext->vertex[3];
  gltext->vertex[7] = bound_y + bound_height;
  gltext->vertex[8] = gldrawable->z;
  gltext->vertex[9] = bound_x;
  gltext->vertex[10] = gltext->vertex[7];
  gltext->vertex[11] = gldrawable->z;
}

/* Sets the size of the text taking care of aligning the vertices to the
 * pixel boundaries. It ensures clean text rendering without any artifact
 * implied by the sub-pixels rasterization process. */
static void
set_size (PgmGlDrawable *gldrawable)
{
  PgmGlViewport *glviewport = gldrawable->glviewport;
  PgmGlText *gltext = PGM_GL_TEXT (gldrawable);
  gfloat offset_x = glviewport->pixel_offset_x;
  gfloat offset_y = glviewport->pixel_offset_y;
  gfloat bound_width = floorf (gldrawable->width / offset_x) * offset_x;
  gfloat bound_height = floorf (gldrawable->height / offset_y) * offset_y;

  gltext->vertex[3] = gldrawable->bg_vertex[0] + bound_width;
  gltext->vertex[6] = gltext->vertex[3];
  gltext->vertex[7] = gldrawable->bg_vertex[1] + bound_height;
  gltext->vertex[10] = gltext->vertex[7];
}

/* Defines texture coordinates */
static void
set_coordinates (PgmGlText *gltext)
{
  PgmTexture *texture = gltext->texture;

  gltext->coord[2] = (PgmGlFloat) texture->width / texture->width_pot;
  gltext->coord[4] = gltext->coord[2];
  gltext->coord[5] = (PgmGlFloat) texture->height / texture->height_pot;
  gltext->coord[7] = gltext->coord[5];
}

/* Stores outline width depending on PgmText outline width converting it
 * from canvas coordinates to viewport coordinates */
static void
set_outline_width (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  gltext->outline_width = get_projected_height (gldrawable, text->outline_width);
}

/* Stores outline color depending on PgmText color normalizing components to
 * the range [0.0; 1.0] */
static void
set_outline_color (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  GST_OBJECT_LOCK (text);
  gltext->outline_color[0] = text->outline_r * INV_255;
  gltext->outline_color[1] = text->outline_g * INV_255;
  gltext->outline_color[2] = text->outline_b * INV_255;
  gltext->outline_color[3] = text->outline_a * INV_255;
  GST_OBJECT_UNLOCK (text);
}

/* Defines Pango spacing depending on PgmText line spacing converting it
 * from canvas coordinates to viewport coordinates */
static void
set_line_spacing (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);
  gfloat line_spacing;

  line_spacing = get_projected_height (gldrawable, text->line_spacing);
  pango_layout_set_spacing (gltext->layout, (gint)
                            (line_spacing * PANGO_SCALE));
}

/* Calculates the maximum total height of a line of text for the current font
 * and font height set in gltext. Returns the value in device unit. */
static gfloat
get_max_height (PgmGlText *gltext)
{
  /* FIXME: we use the language of the current locale, but the text we display
   *        may be in another language. */
  PangoLanguage *language = NULL;
  PangoFontMetrics *metrics = NULL;
  gint ascent, descent;
  gchar *locale = NULL;

#ifdef HAVE_PANGO_1_16
  language = pango_language_get_default ();
#else
#ifdef G_OS_WIN32
  locale = g_win32_getlocale ();
#else
  locale = g_strdup (setlocale (LC_CTYPE, NULL));
#endif
  language = pango_language_from_string (locale);
#endif

  metrics = pango_context_get_metrics (gltext->pango_ctx,
                                       gltext->desc, language);
  ascent = pango_font_metrics_get_ascent (metrics);
  descent = pango_font_metrics_get_descent (metrics);

  pango_font_metrics_unref (metrics);

  g_free (locale);

  return ((ascent + descent) / ((gfloat) PANGO_SCALE));
}

/* Defines Pango font size depending on PgmText height converting it from
 * canvas coordinates to viewport coordinates. This function depends on
 * various parameters, including the font set for gltext. When setting many
 * parameters, it is cautious to set this one last. */
static void
set_font_height (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  /* All the heights here are in device unit. */
  gfloat requested_height, wanted_height, max_height, font_ratio;

  requested_height = get_projected_height (gldrawable, text->height);
  requested_height *= gldrawable->height;

  /* The "pango height" is something like the height of an M, but we want to set
   * the maximum height of a line, so we need to calculate the ratio first. */
  pango_font_description_set_absolute_size (gltext->desc,
                                            requested_height * PANGO_SCALE);
  max_height = get_max_height (gltext);
  font_ratio = requested_height / max_height;

  wanted_height = requested_height * font_ratio;
  pango_font_description_set_absolute_size (gltext->desc,
                                            wanted_height * PANGO_SCALE);
  pango_layout_set_font_description (gltext->layout, gltext->desc);
}

/* Defines Pango font family depending on PgmText font family */
static void
set_font_family (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  GST_OBJECT_LOCK (text);
  pango_font_description_set_family (gltext->desc, text->font_family);
  pango_layout_set_font_description (gltext->layout, gltext->desc);
  GST_OBJECT_UNLOCK (text);
}

/* Defines Pango text/markup depending on PgmText label */
static void
set_label (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  /* FIXME: defining a markup and right after a label keeps the
   *        previous markups on the new label. */
  GST_OBJECT_LOCK (text);
  if (text->use_markup)
    pango_layout_set_markup (gltext->layout, text->label, -1);
  else
    pango_layout_set_text (gltext->layout, text->label, -1);
  GST_OBJECT_UNLOCK (text);
}

/* Defines Pango width depending on PgmText width */
static void
set_width (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmViewport *viewport = PGM_VIEWPORT (gldrawable->glviewport);
  gfloat non_square_ratio;
  gfloat width;

  width = get_projected_width (gldrawable, gldrawable->width);

  GST_OBJECT_LOCK (viewport);
  non_square_ratio = 1.0f / viewport->pixel_aspect_ratio;
  GST_OBJECT_UNLOCK (viewport);

  pango_layout_set_width (gltext->layout, (guint)
                          (width * PANGO_SCALE * non_square_ratio));
}

/* Defines Pango justification depending on PgmText justification */
static void
set_justify (PgmGlText *gltext)
{
#ifdef HAVE_PANGO_1_16
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  GST_OBJECT_LOCK (text);
  pango_layout_set_justify (gltext->layout, text->justify);
  GST_OBJECT_UNLOCK (text);
#endif /* HAVE_PANGO_1_16 */
}

/* Defines Pango weight depending on PgmText weight */
static void
set_weight (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  switch (text->weight)
    {
    case PGM_TEXT_WEIGHT_LIGHT:
      pango_font_description_set_weight (gltext->desc, PANGO_WEIGHT_LIGHT);
      break;

    case PGM_TEXT_WEIGHT_NORMAL:
      pango_font_description_set_weight (gltext->desc, PANGO_WEIGHT_NORMAL);
      break;

    case PGM_TEXT_WEIGHT_BOLD:
      pango_font_description_set_weight (gltext->desc, PANGO_WEIGHT_BOLD);
      break;

    default:
      break;
    }

  pango_layout_set_font_description (gltext->layout, gltext->desc);
}

/* Defines Pango stretching depending on PgmText stretching */
static void
set_stretch (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  switch (text->stretch)
    {
    case PGM_TEXT_STRETCH_CONDENSED:
      pango_font_description_set_stretch (gltext->desc,
                                          PANGO_STRETCH_CONDENSED);
      break;

    case PGM_TEXT_STRETCH_NORMAL:
      pango_font_description_set_stretch (gltext->desc, PANGO_STRETCH_NORMAL);
      break;

    case PGM_TEXT_STRETCH_EXPANDED:
      pango_font_description_set_stretch (gltext->desc,
                                          PANGO_STRETCH_EXPANDED);
      break;

    default:
      break;
    }

  pango_layout_set_font_description (gltext->layout, gltext->desc);
}

/* Defines Pango alignment depending on PgmText alignment */
static void
set_alignment (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  switch (text->alignment)
    {
    case PGM_TEXT_ALIGN_LEFT:
      pango_layout_set_alignment (gltext->layout, PANGO_ALIGN_LEFT);
      break;

    case PGM_TEXT_ALIGN_CENTER:
      pango_layout_set_alignment (gltext->layout, PANGO_ALIGN_CENTER);
      break;

    case PGM_TEXT_ALIGN_RIGHT:
      pango_layout_set_alignment (gltext->layout, PANGO_ALIGN_RIGHT);
      break;

    default:
      break;
    }
}

/* Defines Pango ellipsizing depending on PgmText ellipsizing */
static void
set_ellipsize (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  switch (text->ellipsize)
    {
    case PGM_TEXT_ELLIPSIZE_NONE:
      pango_layout_set_ellipsize (gltext->layout, PANGO_ELLIPSIZE_NONE);
      break;

    case PGM_TEXT_ELLIPSIZE_START:
      pango_layout_set_ellipsize (gltext->layout, PANGO_ELLIPSIZE_START);
      break;

    case PGM_TEXT_ELLIPSIZE_MIDDLE:
      pango_layout_set_ellipsize (gltext->layout, PANGO_ELLIPSIZE_MIDDLE);
      break;

    case PGM_TEXT_ELLIPSIZE_END:
      pango_layout_set_ellipsize (gltext->layout, PANGO_ELLIPSIZE_END);
      break;

    default:
      break;
    }
}

/* Defines Pango wrapping depending on PgmText wrapping */
static void
set_wrap (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  switch (text->wrap)
    {
    case PGM_TEXT_WRAP_WORD:
      pango_layout_set_wrap (gltext->layout, PANGO_WRAP_WORD);
      break;

    case PGM_TEXT_WRAP_CHAR:
      pango_layout_set_wrap (gltext->layout, PANGO_WRAP_CHAR);
      break;

    case PGM_TEXT_WRAP_WORD_CHAR:
      pango_layout_set_wrap (gltext->layout, PANGO_WRAP_WORD_CHAR);
      break;

    default:
      break;
    }
}

/* Defines Pango gravity depending on PgmText gravity */
static void
set_gravity (PgmGlText *gltext)
{
#ifdef HAVE_PANGO_1_16
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  switch (text->gravity)
    {
    case PGM_TEXT_GRAVITY_SOUTH:
      pango_context_set_base_gravity (gltext->pango_ctx, PANGO_GRAVITY_SOUTH);
      break;

    case PGM_TEXT_GRAVITY_EAST:
      pango_context_set_base_gravity (gltext->pango_ctx, PANGO_GRAVITY_EAST);
      break;

    case PGM_TEXT_GRAVITY_NORTH:
      pango_context_set_base_gravity (gltext->pango_ctx, PANGO_GRAVITY_NORTH);
      break;

    case PGM_TEXT_GRAVITY_WEST:
      pango_context_set_base_gravity (gltext->pango_ctx, PANGO_GRAVITY_WEST);
      break;

    case PGM_TEXT_GRAVITY_AUTO:
      pango_context_set_base_gravity (gltext->pango_ctx, PANGO_GRAVITY_AUTO);
      break;

    default:
      break;
    }
#endif /* HAVE_PANGO_1_16 */
}

/* Defines Pango style depending on PgmText style */
static void
set_style (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  switch (text->style)
    {
    case PGM_TEXT_STYLE_NORMAL:
      pango_font_description_set_style (gltext->desc, PANGO_STYLE_NORMAL);
      break;

    case PGM_TEXT_STYLE_OBLIQUE:
      pango_font_description_set_style (gltext->desc, PANGO_STYLE_OBLIQUE);
      break;

    case PGM_TEXT_STYLE_ITALIC:
      pango_font_description_set_style (gltext->desc, PANGO_STYLE_ITALIC);
      break;

    default:
      break;
    }

  pango_layout_set_font_description (gltext->layout, gltext->desc);
}

/* Defines Pango variant depending on PgmText variant */
static void
set_variant (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmText *text = PGM_TEXT (gldrawable->drawable);

  switch (text->variant)
    {
    case PGM_TEXT_VARIANT_NORMAL:
      pango_font_description_set_variant (gltext->desc, PANGO_VARIANT_NORMAL);
      break;

    case PGM_TEXT_VARIANT_SMALL_CAPS:
      pango_font_description_set_variant (gltext->desc,
                                          PANGO_VARIANT_SMALL_CAPS);
      break;

    default:
      break;
    }

  pango_layout_set_font_description (gltext->layout, gltext->desc);
}

/* Update all the properties of the text */
static void
update_properties (PgmGlText *gltext)
{
  set_font_family (gltext);
  set_weight (gltext);
  set_stretch (gltext);
  set_style (gltext);
  set_variant (gltext);
  set_alignment (gltext);
  set_ellipsize (gltext);
  set_justify (gltext);
  set_line_spacing (gltext);
  set_width (gltext);
  set_wrap (gltext);
  set_gravity (gltext);
  set_label (gltext);
  set_outline_width (gltext);
  set_outline_color (gltext);
  set_font_height (gltext);
}

/* Cairo uses premultiplied-alpha to represent pixels (mostly for compositing).
 * This function unpremultiplies the alpha of cairo's native endian ARGB buffer
 * and converts it to BGRA in memory order. */
static void
unpremultiply_alpha (guint32 *buffer,
                     guint width,
                     guint height)
{
  guint size = width * height;
  guint8 a, r, g, b;
  guint i;

  for (i = 0; i < size; i++)
    {
      a = (buffer[i] & ARGB_A_MASK) >> ARGB_A_OFFSET;
      if (a == 0) /* optimisation for most common case */
        {
          continue;
        }
      else if (a == 255)
        {
          /* If the pixel is full alpha, we don't have to change the rgb values,
           * but we have to convert the pixel from native endian ARGB to memory
           * BGRA, which is equivalent to little endian ARGB. */
          buffer[i] = GUINT32_TO_LE (buffer[i]);
        }
      else
        {
          r = (((buffer[i] & ARGB_R_MASK) >> ARGB_R_OFFSET)
               * 255 + a / 2) / a;
          g = (((buffer[i] & ARGB_G_MASK) >> ARGB_G_OFFSET)
               * 255 + a / 2) / a;
          b = (((buffer[i] & ARGB_B_MASK) >> ARGB_B_OFFSET)
               * 255 + a / 2) / a;
          /* we write the buffer in BGRA memory order (B in low address, A in
           * high address) in one memory access */
          buffer[i] = b << BYTE_1_OFFSET
              | g << BYTE_2_OFFSET
              | r << BYTE_3_OFFSET
              | a << BYTE_4_OFFSET;
        }
    }
}

/* Allocates the pixmap, the Pango pipeline using a Cairo backend and
 * generates the texture */
static void
create_pixmap (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmViewport *viewport = PGM_VIEWPORT (gldrawable->glviewport);
  PgmContextTask *task;
  gfloat sx, sy;

  gltext->width =
    MAX (1, (guint) get_projected_width (gldrawable, gldrawable->width));
  gltext->height =
    MAX (1, (guint) get_projected_height (gldrawable, gldrawable->height));

  GST_OBJECT_LOCK (viewport);
  if (viewport->rotation == PGM_VIEWPORT_ROTATION_NONE
      || viewport->rotation == PGM_VIEWPORT_ROTATION_180)
    {
      sx = viewport->pixel_aspect_ratio;
      sy = 1.0f;
    }
  else
    {
      sx = 1.0f;
      sy = viewport->pixel_aspect_ratio;
    }
  GST_OBJECT_UNLOCK (viewport);

  gltext->size = gltext->width * gltext->height * 4;
  gltext->buffer = (guchar *) g_slice_alloc0 (gltext->size);
  gltext->surface = cairo_image_surface_create_for_data (gltext->buffer,
                                                         CAIRO_FORMAT_ARGB32,
                                                         gltext->width,
                                                         gltext->height,
                                                         gltext->width * 4);

  gltext->cairo_ctx = cairo_create (gltext->surface);
  cairo_set_font_options (gltext->cairo_ctx, gltext->font_options);
  cairo_scale (gltext->cairo_ctx, sx, sy);
  gltext->layout = pango_cairo_create_layout (gltext->cairo_ctx);
  gltext->pango_ctx = pango_layout_get_context (gltext->layout);

  pgm_texture_set_buffer (gltext->texture, gltext->buffer, PGM_IMAGE_BGRA,
                          gltext->width, gltext->height, gltext->size, 0);
  set_coordinates (gltext);

  /* And request a texture generation */
  task = pgm_context_task_new (PGM_CONTEXT_GEN_TEXTURE, gltext->texture);
  pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
}

/* Frees the pixmap and the Cairo backend */
static void
free_pixmap (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmContextTask *task;

  g_object_unref (gltext->layout);
  gltext->layout = NULL;
  cairo_destroy (gltext->cairo_ctx);
  gltext->cairo_ctx = NULL;
  cairo_surface_destroy (gltext->surface);
  gltext->surface = NULL;
  if (gltext->buffer)
    {
      g_slice_free1 (gltext->size, (gpointer) gltext->buffer);
      gltext->buffer = NULL;
    }

  /* Request texture clean up */
  task = pgm_context_task_new (PGM_CONTEXT_CLEAN_TEXTURE, gltext->texture);
  pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
}

/* Updates text properties and draw it on the pixmap */
static void
update_pixmap (PgmGlText *gltext)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (gltext);
  PgmContextTask *task;

  /* Make context fully transparent */
  cairo_set_operator (gltext->cairo_ctx, CAIRO_OPERATOR_SOURCE);
  cairo_set_source_rgba (gltext->cairo_ctx, 0.0f, 0.0f, 0.0f, 0.0f);
  cairo_paint (gltext->cairo_ctx);

  /* Then draw the text */
  cairo_set_source_rgba (gltext->cairo_ctx, 1.0f, 1.0f, 1.0f, 1.0f);
  pango_cairo_show_layout (gltext->cairo_ctx, gltext->layout);

  /* And the stroke */
  if (gltext->outline_width > 0.0f)
    {
      pango_cairo_layout_path (gltext->cairo_ctx, gltext->layout);
      cairo_set_source_rgba (gltext->cairo_ctx,
                             gltext->outline_color[0],
                             gltext->outline_color[1],
                             gltext->outline_color[2],
                             gltext->outline_color[3]);
      cairo_set_line_width (gltext->cairo_ctx, gltext->outline_width);
      cairo_stroke (gltext->cairo_ctx);
    }

#if 0
  /* Debugging */
#ifndef WIN32
  cairo_surface_write_to_png (gltext->surface, "/tmp/pgmgltext.png");
#else
  cairo_surface_write_to_png (gltext->surface, "C:\\pgmgltext.png");
#endif /* WIN32 */
#endif

  /* Get rid of the premultiplied alpha */
  unpremultiply_alpha ((guint32*) gltext->buffer, gltext->width, gltext->height);

  /* Update texture */
  pgm_texture_set_buffer (gltext->texture, gltext->buffer, PGM_IMAGE_BGRA,
                          gltext->width, gltext->height, gltext->size, 0);

  /* Request texture upload */
  task = pgm_context_task_new (PGM_CONTEXT_UPLOAD_TEXTURE, gltext->texture);
  pgm_context_push_deferred_task (gldrawable->glviewport->context, task);
}

/* Do a complete regeneration of a text */
static void
regenerate_text (PgmGlText *gltext)
{
  free_pixmap (gltext);
  create_pixmap (gltext);
  update_properties (gltext);
  update_pixmap (gltext);
  set_position (PGM_GL_DRAWABLE (gltext));
}

/* Update a text by simply generating the new one in the current allocated
 * memory if the size has not changed. If it has changed, the whole text is
 * regenerated. */
static void
update_text (PgmGlText *gltext)
{
  if (!gltext->size_updated)
    update_pixmap (gltext);
  else
    {
      gltext->size_updated = FALSE;
      regenerate_text (gltext);
    }
}

/* PgmGlDrawable methods */

static void
pgm_gl_text_draw (PgmGlDrawable *gldrawable)
{
  PgmGlText *gltext = PGM_GL_TEXT (gldrawable);

  GST_LOG_OBJECT (gldrawable, "draw");

  /* Only draw if it's visible */
  if (gltext->fg_color[3] != 0.0f)
    {
      PgmContextProcAddress *gl = gldrawable->glviewport->context->gl;

      pgm_texture_bind (gltext->texture);
      gl->enable_client_state (PGM_GL_VERTEX_ARRAY);
      gl->enable_client_state (PGM_GL_TEXTURE_COORD_ARRAY);
      gl->vertex_pointer (3, PGM_GL_FLOAT, 0, gltext->vertex);
      gl->tex_coord_pointer (2, PGM_GL_FLOAT, 0, gltext->coord);
      gl->color_4fv (gltext->fg_color);
      gl->draw_arrays (PGM_GL_QUADS, 0, 4);
      gl->disable_client_state (PGM_GL_VERTEX_ARRAY);
      gl->disable_client_state (PGM_GL_TEXTURE_COORD_ARRAY);
      pgm_texture_unbind (gltext->texture);
    }
}

static void
pgm_gl_text_regenerate (PgmGlDrawable *gldrawable)
{
  PgmGlText *gltext = PGM_GL_TEXT (gldrawable);

  GST_DEBUG_OBJECT (gldrawable, "regenerate");

  if (gltext->size_updated)
    {
      regenerate_text (gltext);
      gltext->size_updated = FALSE;
    }
}

static void
pgm_gl_text_update_projection (PgmGlDrawable *gldrawable)
{
  PgmGlText *gltext = PGM_GL_TEXT (gldrawable);

  GST_DEBUG_OBJECT (gldrawable, "update_projection");

  gltext->size_updated = TRUE;
}

static void
pgm_gl_text_set_size (PgmGlDrawable *gldrawable)
{
  PgmGlText *gltext = PGM_GL_TEXT (gldrawable);

  GST_DEBUG_OBJECT (gldrawable, "set_size");

  set_size (gldrawable);

  gltext->size_updated = TRUE;
}

static void
pgm_gl_text_set_position (PgmGlDrawable *gldrawable)
{
  GST_DEBUG_OBJECT (gldrawable, "set_position");

  set_position (gldrawable);
}

static void
pgm_gl_text_set_fg_color (PgmGlDrawable *gldrawable)
{
  PgmGlText *gltext = PGM_GL_TEXT (gldrawable);
  PgmDrawable *drawable;

  GST_DEBUG_OBJECT (gldrawable, "set_fg_color");

  drawable = gldrawable->drawable;

  GST_OBJECT_LOCK (drawable);
  gltext->fg_color[0] = drawable->fg_r * INV_255;
  gltext->fg_color[1] = drawable->fg_g * INV_255;
  gltext->fg_color[2] = drawable->fg_b * INV_255;
  gltext->fg_color[3] = drawable->fg_a * drawable->opacity * SQR_INV_255;
  GST_OBJECT_UNLOCK (drawable);
}

static void
pgm_gl_text_set_opacity (PgmGlDrawable *gldrawable)
{
  PgmGlText *gltext = PGM_GL_TEXT (gldrawable);
  PgmDrawable *drawable;

  GST_DEBUG_OBJECT (gldrawable, "set_opacity");

  drawable = gldrawable->drawable;

  GST_OBJECT_LOCK (drawable);
  gltext->fg_color[3] = drawable->fg_a * drawable->opacity * SQR_INV_255;
  GST_OBJECT_UNLOCK (drawable);
}

static void
pgm_gl_text_sync (PgmGlDrawable *gldrawable)
{
  PgmGlText *gltext = PGM_GL_TEXT (gldrawable);

  GST_DEBUG_OBJECT (gldrawable, "sync");

  create_pixmap (gltext);
  update_properties (gltext);

  /* Synchronize various other properties */
  pgm_gl_text_set_position (gldrawable);
  pgm_gl_text_set_fg_color (gldrawable);

  update_pixmap (gltext);
}

/* GObject stuff */

PGM_DEFINE_DYNAMIC_TYPE (PgmGlText, pgm_gl_text, PGM_TYPE_GL_DRAWABLE);

void
pgm_gl_text_register (GTypeModule *module)
{
  pgm_gl_text_register_type (module);
}

static void
pgm_gl_text_dispose (GObject *object)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (object);
  PgmGlText *gltext = PGM_GL_TEXT (object);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (gltext, "dispose");

  free_pixmap (gltext);

  pango_font_description_free (gltext->desc);
  gltext->desc = NULL;

  cairo_font_options_destroy (gltext->font_options);
  gltext->font_options = NULL;

  gst_object_unref (gldrawable->drawable);
  pgm_context_remove_tasks_with_data (gldrawable->glviewport->context,
                                      gltext->texture);
  task = pgm_context_task_new (PGM_CONTEXT_FREE_TEXTURE, gltext->texture);
  pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
  gltext->texture = NULL;

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_gl_text_class_init (PgmGlTextClass *klass)
{
  GObjectClass *gobject_class;
  PgmGlDrawableClass *drawable_class;

  GST_DEBUG_CATEGORY_INIT (pgm_gl_text_debug, "pgm_gl_text", 0,
                           "OpenGL plugin: PgmGlText");

  parent_class = g_type_class_peek_parent (klass);

  gobject_class = G_OBJECT_CLASS (klass);
  drawable_class = PGM_GL_DRAWABLE_CLASS (klass);

  /* GObject virtual table */
  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_gl_text_dispose);

  /* PgmGlDrawable virtual table */
  drawable_class->sync = GST_DEBUG_FUNCPTR (pgm_gl_text_sync);
  drawable_class->draw = GST_DEBUG_FUNCPTR (pgm_gl_text_draw);
  drawable_class->regenerate = GST_DEBUG_FUNCPTR (pgm_gl_text_regenerate);
  drawable_class->update_projection =
    GST_DEBUG_FUNCPTR (pgm_gl_text_update_projection);
  drawable_class->set_size = GST_DEBUG_FUNCPTR (pgm_gl_text_set_size);
  drawable_class->set_position = GST_DEBUG_FUNCPTR (pgm_gl_text_set_position);
  drawable_class->set_fg_color = GST_DEBUG_FUNCPTR (pgm_gl_text_set_fg_color);
  drawable_class->set_opacity = GST_DEBUG_FUNCPTR (pgm_gl_text_set_opacity);
}

static void
pgm_gl_text_class_finalize (PgmGlTextClass *klass)
{
  return;
}

static void
pgm_gl_text_init (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "init");

  gltext->font_options = cairo_font_options_create ();
  cairo_font_options_set_antialias (gltext->font_options,
                                    CAIRO_ANTIALIAS_SUBPIXEL);

  gltext->desc = pango_font_description_new ();

  gltext->size_updated = FALSE;
}

/* Public methods */

PgmGlDrawable *
pgm_gl_text_new (PgmDrawable *drawable,
                 PgmGlViewport *glviewport)
{
  PgmGlDrawable *gldrawable;
  PgmGlText *gltext;

  gltext = g_object_new (PGM_TYPE_GL_TEXT, NULL);

  GST_DEBUG_OBJECT (gltext, "created new gltext");

  gltext->texture = pgm_texture_new (glviewport->context);
  gltext->texture->filter = PGM_GL_NEAREST;

  gldrawable = PGM_GL_DRAWABLE (gltext);
  gldrawable->drawable = gst_object_ref (drawable);
  gldrawable->glviewport = glviewport;
  pgm_gl_viewport_connect_changed_callback (glviewport, gldrawable);
  pgm_gl_drawable_sync (gldrawable);

  return gldrawable;
}

void
pgm_gl_text_set_label (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_label");

  set_label (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_markup (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_markup");

  set_label (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_font_family (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_font_family");

  set_font_family (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_font_height (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_font_height");

  set_font_height (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_ellipsize (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_ellipsize");

  set_ellipsize (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_justify (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_justify");

  set_justify (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_alignment (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_alignment");

  set_alignment (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_wrap (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_wrap");

  set_wrap (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_gravity (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_gravity");

  set_gravity (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_stretch (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_stretch");

  set_stretch (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_style (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_style");

  set_style (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_variant (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_variant");

  set_variant (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_weight (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_weight");

  set_weight (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_line_spacing (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_line_spacing");

  set_line_spacing (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_outline_color (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_outline_color");

  set_outline_color (gltext);
  update_text (gltext);
}

void
pgm_gl_text_set_outline_width (PgmGlText *gltext)
{
  GST_DEBUG_OBJECT (gltext, "set_outline_width");

  set_outline_width (gltext);
  update_text (gltext);
}
