#!/usr/bin/perl
# Last.fm Info Plugin for Pidgin / libpurple
# Copyright (c) 2008 by Dominik George <pidgin-lastfm@naturalnet.de>
#
# Yippie, my first perl script :-D
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 3
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

use Purple;

# Global variables
our $song_joker;
our $ago_joker;
our $time_joker;

# Store plugin reference globally
our $plugin;

# Information on plugin for libpurple
%PLUGIN_INFO = (
 perl_api_version => 2,
 name => "Last.fm Info Plugin",
 version => "0.3b",
 summary => "Display Last.fm info in status message",
 description => "Plugin to display information from your Last.fm profile in Pidgin",
 author => "Dominik George <pidgin-lastfm\@naturalnet.de>",
 url => "http://pidgin-lastfm.naturalnet.de",
 
 # Subs that libpurple has to know
 load => "plugin_load",
 unload => "plugin_unload",
 prefs_info => "prefs_info_cb",
 plugin_action_sub => "plugin_actions_cb"
);

# Called upon plugin initialization
sub plugin_init {
 return %PLUGIN_INFO;
}

# Get everything ready upon loading
sub plugin_load {
 my $plugin = shift;
 Purple::Debug::info("lastfm", "Last.fm Plugin loading ...\n");

 # Set the current version
 our $actversion = "0.3b";
 
 # Store plugin reference
 our $plugin = $plugin;

 # Load or create preferences
 Purple::Prefs::add_none("/plugins/core/lastfm");
 Purple::Prefs::add_string("/plugins/core/lastfm/username", "");
 Purple::Prefs::add_string("/plugins/core/lastfm/content", "recenttracks");
 Purple::Prefs::add_int("/plugins/core/lastfm/timeout", 60);

 # Song info cache from preferences
 Purple::Prefs::add_string("/plugins/core/lastfm/song_joker", "%s");
 Purple::Prefs::add_string("/plugins/core/lastfm/ago_joker", "%a");
 Purple::Prefs::add_string("/plugins/core/lastfm/time_joker", "%t");

 # Setup timeout for refreshing info
 Purple::timeout_add($plugin, 1, \&loadinfo_cb, $plugin);

 Purple::Debug::info("lastfm", "Last.FM Plugin loaded.\n");
}

# Cleanly unload plugin
sub plugin_unload {
 my $plugin = shift;

 # Reset status message so we find the standard placeholder next time
 # set_status("%s", "%a", "%t");

 Purple::Debug::info("lastfm", "Last.fm Plugin removed.\n");
}

# Preferences dialog box
sub prefs_info_cb {
 my $frame = Purple::PluginPref::Frame->new();
 
 # Audioscrobbler username
 my $ppref = Purple::PluginPref->new_with_name_and_label("/plugins/core/lastfm/username", "Username:");
 $ppref->set_type(2);
 $ppref->set_max_length(16);
 $frame->add($ppref);

 # Desired content
 $ppref = Purple::PluginPref->new_with_name_and_label("/plugins/core/lastfm/content", "Content:");
 $ppref->set_type(1);
 $ppref->add_choice("Most recent track", "recenttracks");
 $frame->add($ppref);

 # Timeout for refreshing content
 $ppref = Purple::PluginPref->new_with_name_and_label("/plugins/core/lastfm/timeout", "Refresh (sec):");
 $ppref->set_type(3);
 $ppref->set_bounds(30, 300);
 $frame->add($ppref);

 return $frame;
}

# Actions that user can run from UI menu
sub plugin_actions_cb {
 my @actions = ("Refresh info", "Reset placeholders", "Check for updates", "Credits");
}

%plugin_actions = (
 "Refresh info" => \&loadinfo_cb,
 "Reset placeholders" => \&reset_placeholders,
 "Check for updates" => \&checkupdate_cb,
 "Credits" => \&credits_cb
);

# Helper function to escape strings for regex compatibility
sub regex_escape {
 my $string = $_[0];

 $string =~ s/\\/\\\\/g;
 $string =~ s/\//\\\//g;
 $string =~ s/\./\\./g;
 $string =~ s/\+/\\+/g;
 $string =~ s/\?/\\?/g;
 $string =~ s/\^/\\^/g;
 $string =~ s/\$/\\\$/g;
 $string =~ s/\|/\\|/g;
 $string =~ s/\(/\\(/g;
 $string =~ s/\)/\\)/g;
 $string =~ s/\[/\\]/g;
 $string =~ s/\]/\\]/g;
 $string =~ s/\{/\\{/g;
 $string =~ s/\}/\\}/g;

 return $string;
}

# Load placeholders from preferences
sub load_placeholders {
 our $song_joker = Purple::Prefs::get_string("/plugins/core/lastfm/song_joker");
 our $ago_joker = Purple::Prefs::get_string("/plugins/core/lastfm/ago_joker");
 our $time_joker = Purple::Prefs::get_string("/plugins/core/lastfm/time_joker");
}

# Save current placeholders to the preferences
# I hope we will survive SEGFAULTs that way :-)
sub save_placeholders {
 Purple::Prefs::set_string("/plugins/core/lastfm/song_joker", $song_joker);
 Purple::Prefs::set_string("/plugins/core/lastfm/ago_joker", $ago_joker);
 Purple::Prefs::set_string("/plugins/core/lastfm/time_joker", $time_joker);
}

# Reset placeholders to the defaults
sub reset_placeholders {
 load_placeholders();

 our $song_joker = "%s";
 our $ago_joker = "%a";
 our $time_joker = "%t";

 save_placeholders();
}

sub credits_cb {
 my $plugin = shift;

 Purple::Notify::message($plugin, 2, "Last.fm Plugin Credits", "The following people helped a lot at developing the plugin:",
"Miljan Karadzic - For indirect help with setting the status message\n\n
Savar and eggy (debianforum.de) - For some basic perl help\n\n
hawkeye (debianforum.de) - For playing with the time ;-)\n\n
deryni, datallah, resiak and stu (Pidgin IRC Channel) - For psychological treatment in times of trouble\n\n
Vincent (Last.fm Support) - For the logo artwork",
NULL, NULL);
}

sub checkupdate_cb {
 my $plugin = shift;

 # URL with text file containing version information
 my $url = "http://pidgin-lastfm.naturalnet.de/res/version.txt";

 # Fetch version information
 Purple::Util::fetch_url($plugin, $url, TRUE, "Mozilla/5.0", TRUE, "parse_update_cb");
}

sub parse_update_cb {
 my $version = shift;

 my $linebreak = index($version, "\n");
 $version = substr($version, 0, $linebreak);

 Purple::Debug::info("lastfm", "The running version is " . $actversion . ".\n");
 Purple::Debug::info("lastfm", "The most recent version is " . $version . ".\n");

 if ($version > $actversion) {
  Purple::Notify::message($plugin, 2, "Last.fm Plugin", "New version available!", "A new version of the Pidgin Last.FM plugin (version " . $version . ") is available for download.", NULL, NULL);
 } else {
  if ($actversion > $version) {
   Purple::Notify::message($plugin, 2, "Last.fm Plugin", "Your version is from the future!", "You are running a more recent version than the server knows.\n\nI conclude: You are either a developer or a magician :-D ...", NULL, NULL);
  } else {
   Purple::Notify::message($plugin, 2, "Last.fm Plugin", "No new version available!", "You are running the most recent version of the plugin.", NULL, NULL);
  }
 }
}

sub set_status {
 my $song = $_[0];
 my $ago  = $_[1];
 my $time = $_[2];

 # Parse status message and set info
 my $status  = Purple::SavedStatus::get_current();
 my $message = Purple::SavedStatus::get_message($status);
 my $oldmsg  = $message;

 load_placeholders();

 my $song_regex = regex_escape($song_joker);
 my $ago_regex = regex_escape($ago_joker);
 my $time_regex = regex_escape($time_joker);

 Purple::Debug::info("lastfm", "Looking for song joker \"" . $song_joker . "\".\n");

 if ($message =~ /$song_regex/i) {
  Purple::Debug::info("lastfm", "Old message was \"" . $message . "\".\n");
 
  # Now do the regex replacements

  if (($message =~ /$song_regex/i) && ($song ne "") && ($song_joker ne $song)) {
   $message =~ s/$song_regex/$song/i;
   our $song_joker = $song;
  }
 
  if (($message =~ /$ago_regex/i) && ($ago ne "") && ($ago_joker ne $ago)) {
   $message =~ s/$ago_regex/$ago/i;
   our $ago_joker = $ago;
  }
 
  if (($message =~ /$time_regex/i) && ($time ne "") && ($time_joker ne $time)) {
   $message =~ s/$time_regex/$time/i;
   our $time_joker = $time;
  }
 
  save_placeholders();
 
  # Only update if something has changed
  if ($message ne $oldmsg) {
   Purple::Debug::info("lastfm", "Changing to \"" . $message . "\".\n");
   Purple::SavedStatus::set_message($status, $message);
   Purple::SavedStatus::activate($status);
  } else {
   Purple::Debug::info("lastfm", "Status message unchanged.\n");
  }
 } else {
  Purple::Debug::info("lastfm", "Song joker not found ...\n");
  Purple::Debug::info("lastfm", "If you think this is wrong, try resetting the song jokers!\n");
 }

 # Reset timeout so refresh is called regularly 
 my $timeout = Purple::Prefs::get_int("/plugins/core/lastfm/timeout");
 Purple::timeout_add($plugin, $timeout, \&loadinfo_cb, $plugin);
 Purple::Debug::info("lastfm", "Reset timeout.\n");
}

# Parse data from Audioscrobbler HTTP request
sub parse_data_cb {
 my $data = shift;
 Purple::Debug::info("lastfm", "Callback function for HTTP request called.\n");

 if ($data ne "") {
  # Extract first song title and artist
  
  my $colon     = index($data, ",");
  my $linebreak = index($data, "\n");
  my $startpos  = $colon + 1;
  my $length    = $linebreak - $startpos;
 
  my $song      = substr($data, $startpos, $length);
  my $timestamp = substr($data, 0, $colon);
 
  # Play with the time :)
  my $ago = "";
 
  my $time = time;
  my $diff = $time - $timestamp;
  Purple::Debug::info("lastfm", "Current time: " . $time . "\n");
  Purple::Debug::info("lastfm", "Srobbled at : " . $timestamp . "\n");
  Purple::Debug::info("lastfm", "Difference  : " . $diff . "\n");
 
  my $days = int($diff / 86400);
  $diff %= 86400;
 
  my $hours = int($diff / 3600);
  $diff %= 3600;
 
  my $minutes = int($diff / 60);
 
  if ($days > 0) {
   $ago .= $days . " day";
 
   if ($days > 1) {
    $ago .= "s";
   }
  }
 
  if ($hours > 0) {
   if ($days > 0) {
    $ago .= ", ";
   }
 
   $ago .= $hours . " hour";
 
   if ($hours > 1) {
    $ago .= "s";
   }
  }
 
  if ($minutes > 0) {
   if (($hours > 0) || ($days > 0)) {
    $ago .= ", ";
   }
 
   $ago .= $minutes . " minute";
 
   if ($minutes > 1) {
    $ago .= "s";
   }
  }
 
  $ago .= " ago";
 
  $timestring = localtime($timestamp);
 
  Purple::Debug::info("lastfm", "Song is \"" . $song . "\"\n");
  Purple::Debug::info("lastfm", "Scrobbled " . $ago. ".\n");
  Purple::Debug::info("lastfm", "This was at " . $timestring . ".\n");
 
  # Call method that will hopefully set our new status message
  set_status($song, $ago, $timestring);
 } else {
  Purple::Debug::error("lastfm", "Huh? We got an empty dataset from the server ...\n");
  Purple::Debug::error("lastfm", "Skipping status message changing.\n");

  my $username = Purple::Prefs::get_string("/plugins/core/lastfm/username");
  Purple::Notify::message($plugin, 1, "Last.fm Plugin", "No data from server!", "It was not possible to retrieve data from the Audioscrobbler web services for the user \"" . $username . "\".\n\nThe plugin will stop trying to do its work for now. Please verify that your preferences are correct and the web services are online. After doing so, you need to select \"Refresh info\" from the tools menu in order to re-enable the timeout!\n\nHint: Maybe you just didn't listen to music recently ;-) ?", NULL, NULL);
  Purple::Debug::warning("lastfm", "No more timeout based updates will be done.\n");
 }
}

# Routine that is called in any case that triggers a refresh
sub loadinfo_cb {
 my $plugin = shift;
 Purple::Debug::info("lastfm", "User or timeout requested to refresh info.\n");

 # Find out what we must fetch
 my $url = build_url();

 # Let libpurple do the work
 Purple::Debug::info("lastfm", "Telling libpurple to fetch data.\n");
 Purple::Debug::info("lastfm", "URL to be fetched: " . $url . "\n");
 Purple::Util::fetch_url($plugin, $url, TRUE, "Mozilla/5.0", TRUE, "parse_data_cb");
}

# Build URL to retrieve from Audioscrobbler
sub build_url {
 my $protocol = "http://";
 my $hostname = "ws.audioscrobbler.com";
 my $username = Purple::Prefs::get_string("/plugins/core/lastfm/username");
 my $content = Purple::Prefs::get_string("/plugins/core/lastfm/content");
 my $uri = "/1.0/user/" . $username . "/" . $content . ".txt";
 my $url = $protocol . $hostname . $uri;

 return $url;
}

