<?

/**************************************************************************
 * Functions to create an HTML form.  All functions take an optional
 * argument (the last argument) called $extraopts.  This is either an
 * associative array of var="arg" pairs (var => arg) or a string of
 * arguments which will be appended to the HTML tag before the closing
 * bracket.  If this argument is not supplied or is NULL, no extra data
 * will be printed.
 *
 * In most commands, there are extra optional arguments.  If no argument
 * is given (ie not that many arguments are supplied) or the argument value
 * is precisely NULL, the listed default will be used.
 *
 * Most arguments are capable of determining the default user-visible value
 * from the data passed to the form from the client browser.  Other
 * arguments' defaults are given in the function description.
 *
 * Copyright (C) 2001, Matthew Palmer.  Released under the GPL version 2.
 * You should have received a copy of this licence with this software, if not
 * see http://www.fsf.org/copyleft/gpl.html for a full copy of the licence.
 */

// $Id: form.inc,v 1.6 2002/07/12 08:54:35 mjp16 Exp $

require_once("phtml/misc.inc");

/* Prints the HTML corresponding to a form header.  $action is the URL which
 * will be sent the data associated with this form (the argument to the
 * action= field in the HTML), and $method is the method which will be used
 * to submit this form (POST being the default).
 */
function form_header($action, $method=NULL, $extraopts = NULL)
{
	if ($method === NULL) {
		$method = "post";
	}
	$extras = misc_extraopts($extraopts);
	$retval = "<form action=\"$action\" enctype=\"multipart/form-data\" method=\"$method\"$extras>\n";
	if (function_exists("hidden_vars")) {
		$retval .= hidden_vars();
	}

	return $retval;
}

/* An HTML form footer.
 */
function form_footer($extraopts = NULL)
{
	$extras = misc_extraopts($extraopts);
	return "</form$extras>\n";
}

/* A single-line text entry box.  $variable will be the name under which the
 * text entered will be returned to server.  $value is the initial text, and
 * the default is whatever is in $HTTP_POST_VARS under the name of $variable.
 * $maxlen is the maximum possible length of text data input, while $boxlen
 * is the number of characters visible simultaneously.  If neither is
 * specified, the browser defaults will be used, while if only $maxlen is
 * given, $boxlen will be that value.  $ex is the extra options as specified
 * above.
 */
function form_textbox($variable, $value = NULL, $maxlen = NULL, $boxlen = NULL, $ex = NULL)
{
	$ex = misc_extraopts($ex);
	$VALUES = form_values();
	
	$retval = "<input type=\"text\" ";
	if ($maxlen != NULL) {
		if ($boxlen === NULL) {
			$boxlen = $maxlen;
		}
		$retval = $retval."maxlength=\"$maxlen\" size=\"$boxlen\" ";
	}
	
	if ($value === NULL) {
		$value = $VALUES[$variable];
	}
	return $retval." name=\"$variable\" value=\"$value\"$ex>\n";
}

/* A multi-line text entry box.  $variable is the name of the data element,
 * $value is the default contents of the textbox.  $cols is the number of
 * columns and $rows is the number of rows in the box.  $ex is the extra
 * options.
 */
function form_textarea($variable, $value = NULL, $cols = 20, $rows = 3, $ex = NULL)
{
	$ex = misc_extraopts($ex);
	$VALUES = form_values();
	
	if ($value === NULL) {
		$value = $VALUES[$variable];
	}
	$retval = "<textarea name=\"$variable\" rows=\"$rows\" cols=\"$cols\"$ex>$value</textarea>";
	
	return $retval;
}

/* A hidden variable.  Sets the variable with the name of $var to the given
 * value, or the value currently in $HTTP_POST_VARS if none is given.  $ex
 * is any extra options.
 */
function form_hidden($var, $value = NULL, $ex = NULL)
{
	$ex = misc_extraopts($ex);
	$VALUES = form_values();
	
	if ($value === NULL) {
		$value = $VALUES[$var];
	}

	return "<input type=\"hidden\" name=\"$var\", value=\"$value\">\n";
}

/* A password box.  The variable this will be stored in is given in $variable,
 * while any default value (which will be presented as asterisks on screen)
 * is given in $value, and any extra options are given in $ex.
 */
function form_password($variable, $value = NULL, $ex = NULL)
{
	$ex = misc_extraopts($ex);
	$VALUES = form_values();
	if ($value === NULL) {
		$value = $VALUES[$variable];
	}
	
	return "<input type=\"password\" name=\"$variable\" value=\"$value\"$ex>\n";
}

/* A select item.  $var is the variable name in which the data will be
 * returned, while $list is an associative array of the items which should
 * be presented for selection.  Each key will be the value stored in the
 *variable if it is selected, while the value associated with the key will
 * be the text the user sees in the select list.  $value, if given, should
 * be the key of the pre-selected item.  $size is the number of items will be
 * shown simultaneously, and $ex is the extra options in the main
 * select tag.
 */
function form_select($var, $list, $value = NULL, $size = 1, $ex = NULL)
{
	$ex = misc_extraopts($ex);
	$VALUES = form_values();
	
	if ($size == -1) {
		$size = count($list);
	}
	
	if ($value === NULL) {
		$value = $VALUES[$var];
	}

	$retstr = "<select name=\"".$var."\" size=\"$size\"$ex>\n";

	foreach ($list as $val => $display) {
		$retstr .= "<option value=\"$val\"";
		if ($value == $val) {
			$retstr .= " selected";
		}
		$retstr .= ">$display</option>\n";
	}
	
	$retstr .= "</select>\n";
	
	return $retstr;
}

/* This function produces a similar effect to the form_select() above, but
 * allows multiple items to be select simultaneously.
 */
function form_multiselect($var, $list, $values = NULL, $size = 1, $ex = NULL)
{
	$ex = misc_extraopts($ex);
	$VALUES = form_values();

	if ($values === NULL) {
		$values = $VALUES[$var];
	}
	if (!$values) {
		$values = array();
	}

	if ($size == -1) {
		$size = count($list);
	}
	
	$retstr = "<select name=\"".$var."[]\" size=\"$size\" multiple>\n";

	foreach ($list as $val => $display) {
		$retstr .= "<option value=\"$val\"";
		if (array_intersect($values, array($val))) {
			$retstr .= " selected";
		}
		$retstr .= ">$display</option>\n";
	}
	
	$retstr = $retstr."</select>\n";

	return $retstr;
}

/* Produces a checkbox with the given variable name in $var.  If $checked
 * evaluates to TRUE, the checkbox will be set.  If exactly NULL,
 * $HTTP_POST_VARS will be used to determine initial status.
 */
function form_checkbox($var, $checked = NULL, $ex = NULL)
{
	$ex = misc_extraopts($ex);
	$VALUES = form_values();
	if ($checked === NULL) {
		$checked = $VALUES[$var];
	}
	
	$retval = "<input type=\"checkbox\" name=\"$var\"";
	if ($checked) {
		$retval .= " checked";
	}
	
	$retval .= "$ex>\n";
	
	return $retval;
}

/* Produce a set of boxes for date entry ($prefix_day, $prefix_month, $prefix_year).
 * $date is the optional current date to be shown, in ISO-standard format
 * $(YYYY-MM-DD).
 */
function form_date($prefix, $date = NULL)
{
	if ($date) {
		debug("Producing date box with $date");
		$year = substr($date, 0, 4);
		$month = substr($date, 5, 2);
		$day = substr($date, 8, 2);
	} else {
		$year = $month = $day = NULL;
	}
	
	$retval = form_textbox($prefix."_day", $day, 2);
	$retval .= "/";
	$retval .= form_textbox($prefix."_month", $month, 2);
	$retval .= "/";
	$retval .= form_textbox($prefix."_year", $year, 4);

	return $retval;
}

/* A file entry box.  $var is the name of the variable which will contain
 * the file data (via $HTTP_POST_FILES), while $value will be the default
 * file name.  $ex as per usual.
 */
function form_file($var, $value = NULL, $ex = NULL)
{
	$ex = misc_extraopts($ex);
	$VALUES = form_values();
	if ($value === NULL) {
		$value = $VALUES[$var];
	}
	$retval = "<input type=\"file\" name=\"$var\" value=\"$value\"$ex>\n";
	return $retval;
}

/* A clickable button.  It will not do anything by default - it requires
 * something from the $ex portion to give it something to do.
 */
function form_button($label = "Click Me!", $ex = NULL)
{
	$ex = misc_extraopts($ex);
	return "<input type=\"button\" value=\"$label\"$ex>\n";
}

/* A submit button.  $label is the visual text which will be displayed on
 * the button, or 'Submit' if none is given.  Causes the form data
 * encapsulated within the form_header() and form_footer() to be submitted
 * to the server when it is clicked.
 */
function form_submit($label = "Submit", $ex = NULL)
{
	$ex = misc_extraopts($ex);
	return "<input type=\"submit\" value=\"$label\"$ex>\n";
}

/* A form reset button.
 */
function form_reset($label = "Reset", $ex = NULL)
{
	$ex = misc_extraopts($ex);
	return "<input type=\"reset\" value=\"$label\"$ex>\n";
}

/* Returns all the possible data from which values for form elements can be
 * sourced.
 */
function form_values()
{
	global $HTTP_POST_VARS, $HTTP_GET_VARS;
	return array_merge($HTTP_POST_VARS, $HTTP_GET_VARS);
}

?>
