/*
 *  Copyright (C) 2004 - 2005 Gareth Ardron <gareth@fission.org.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "php.h"
#include "php_ini.h"
#include "ext/standard/info.h"
#include "php_clam.h"
#include <clamav.h>
#include <dirent.h>

/* If you declare any globals in php_clam.h uncomment this: */
ZEND_DECLARE_MODULE_GLOBALS(clam)


/* True global resources - no need for thread safety here */
static int le_clam;
struct cl_stat clam_dbstat;
struct cl_limits clam_limits;
struct cl_node *root = NULL;
int retl, retb, ht, no = 0;
const char * cl_error = NULL;


/* {{{ clam_functions[] */
function_entry clam_functions[] = {
        PHP_FE(clam_scan_buffer,	NULL)
	PHP_FE(clam_scan_file,		NULL)
	PHP_FE(clam_get_version,	NULL)
	PHP_FE(clam_test,		NULL)
	{NULL, NULL, NULL}
};
/* }}} */


/* {{{ clam_module_entry */
zend_module_entry clam_module_entry = {
#if ZEND_MODULE_API_NO >= 20010901
	STANDARD_MODULE_HEADER,
#endif
	"clam",
	clam_functions,
	PHP_MINIT(clam),
	PHP_MSHUTDOWN(clam),
	PHP_RINIT(clam),
	PHP_RSHUTDOWN(clam),
	PHP_MINFO(clam),
#if ZEND_MODULE_API_NO >= 20010901
	"0.3",
#endif
	STANDARD_MODULE_PROPERTIES
};
/* }}} */

#ifdef COMPILE_DL_CLAM
ZEND_GET_MODULE(clam)
#endif

/* {{{ PHP_INI */
PHP_INI_BEGIN()
/*	if statement needed for api change between php4 and 5 - same functionally, and Int version
	still uses long's internally, but the name changed to remind developers to use Long over
	Int for 64bit compatibility reasons.
*/
#if ZEND_MODULE_API_NO > 20030304
#define _ini_int_var OnUpdateLong
#else
#define _ini_int_var OnUpdateInt
#endif
	STD_PHP_INI_ENTRY("clam.virus_dbdir", "/var/lib/clamav/", PHP_INI_ALL, OnUpdateString, virus_dbdir, zend_clam_globals, clam_globals)
	STD_PHP_INI_ENTRY("clam.recursion_level", "5", PHP_INI_ALL, _ini_int_var, recursion_level, zend_clam_globals, clam_globals)
	STD_PHP_INI_ENTRY("clam.max_files", "1000", PHP_INI_ALL, _ini_int_var, max_files, zend_clam_globals, clam_globals)
	STD_PHP_INI_BOOLEAN("clam.memory_limit", "1", PHP_INI_ALL, _ini_int_var, memory_limit, zend_clam_globals, clam_globals)
PHP_INI_END()
/* }}} */


/* {{{ php_clam_init_globals */
static void php_clam_init_globals(zend_clam_globals *clam_globals)
{
	clam_globals->virus_dbdir = NULL;
	clam_globals->recursion_level = 5;
	clam_globals->max_files = 1000;
	clam_globals->memory_limit = 1;
}
/* }}} */


/* {{{ PHP_RINIT_FUNCTION
 */
PHP_RINIT_FUNCTION(clam)
{
	/* scan for database updates */
	if(cl_statchkdir(&clam_dbstat) == 1) {
		/* reload db */
		cl_free(root);
		cl_loaddbdir(cl_retdbdir(), &root, &no);
		cl_build(root);
		/* reload watcher */
		cl_statfree(&clam_dbstat);
		cl_statinidir(cl_retdbdir(), &clam_dbstat);
	}
	
	/* generate clam_limits */
	clam_limits.maxfiles = CLAM_G(max_files);
	clam_limits.maxreclevel = CLAM_G(recursion_level);
	clam_limits.archivememlim = CLAM_G(memory_limit);
	
	/* retval */
	return SUCCESS;
}
/* }}} */


/* {{{ PHP_RSHUTDOWN_FUNCTION
 */
PHP_RSHUTDOWN_FUNCTION(clam)
{
	return SUCCESS;
}
/* }}} */


/* {{{ PHP_MINIT_FUNCTION
 */
PHP_MINIT_FUNCTION(clam)
{
	ZEND_INIT_MODULE_GLOBALS(clam, php_clam_init_globals, NULL);
	REGISTER_INI_ENTRIES();
	
	/* load db */
	retl = cl_loaddbdir(cl_retdbdir(), &root, &no);
	if (retl != 0) {
		cl_error = cl_strerror(retl);
	} else {
		retb = cl_build(root);
		if (retb != 0) {
			cl_error = cl_strerror(retb);
		}
	}
	
	/* start scanner for dbdir updates */
	memset(&clam_dbstat, 0, sizeof(struct cl_stat));
	cl_statinidir(cl_retdbdir(), &clam_dbstat);
	
	/* register long constants for options */
	REGISTER_LONG_CONSTANT( "CL_SCAN_MAIL", CL_SCAN_MAIL, CONST_CS | CONST_PERSISTENT ) ;
	REGISTER_LONG_CONSTANT( "CL_SCAN_ARCHIVE", CL_SCAN_ARCHIVE, CONST_CS | CONST_PERSISTENT ) ;
	REGISTER_LONG_CONSTANT( "CL_SCAN_OLE2", CL_SCAN_OLE2, CONST_CS | CONST_PERSISTENT ) ;
	REGISTER_LONG_CONSTANT( "CL_SCAN_BLOCKENCRYPTED", CL_SCAN_BLOCKENCRYPTED, CONST_CS | CONST_PERSISTENT ) ;
	REGISTER_LONG_CONSTANT( "CL_SCAN_HTML", CL_SCAN_HTML, CONST_CS | CONST_PERSISTENT ) ;
	REGISTER_LONG_CONSTANT( "CL_SCAN_RAW", CL_SCAN_RAW, CONST_CS | CONST_PERSISTENT ) ;
	REGISTER_LONG_CONSTANT( "CL_SCAN_PE", CL_SCAN_PE, CONST_CS | CONST_PERSISTENT ) ;
	REGISTER_LONG_CONSTANT( "CL_SCAN_DISABLERAR", CL_SCAN_DISABLERAR, CONST_CS | CONST_PERSISTENT ) ;
	REGISTER_LONG_CONSTANT( "CL_SCAN_BLOCKBROKEN", CL_SCAN_BLOCKBROKEN, CONST_CS | CONST_PERSISTENT ) ;
	REGISTER_LONG_CONSTANT( "CL_SCAN_MAILURL", CL_SCAN_MAILURL, CONST_CS | CONST_PERSISTENT ) ;
	REGISTER_LONG_CONSTANT( "CL_SCAN_BLOCKMAX", CL_SCAN_BLOCKMAX, CONST_CS | CONST_PERSISTENT ) ;
	REGISTER_LONG_CONSTANT( "CL_SCAN_STDOPT", CL_SCAN_STDOPT, CONST_CS | CONST_PERSISTENT ) ;
	
	/* return/close */
	return SUCCESS;
}
/* }}} */


/* {{{ PHP_MSHUTDOWN_FUNCTION
 */
PHP_MSHUTDOWN_FUNCTION(clam)
{
	UNREGISTER_INI_ENTRIES();
	
	/* unload database */
	if (root != NULL) {
		cl_free(root);
	}
	
	/* clear dbdir scanner */
	cl_statfree(&clam_dbstat);
	
	return SUCCESS;
}
/* }}} */


/* {{{ PHP_MINFO_FUNCTION
 */
PHP_MINFO_FUNCTION(clam)
{
	const char *clam_v = cl_retver();
	php_info_print_table_start();
	php_info_print_table_row(2, "ClamAV Support", "Enabled");
	php_info_print_table_row(2, "ClamAV Version", clam_v);
	php_info_print_table_row(2, "PHP-ClamAV Version", "0.3");
	php_info_print_table_end();
	DISPLAY_INI_ENTRIES();
}
/* }}} */


/* {{{ proto string clam_scan_buffer(string $buffer)
   Scan a given buffer for viruses. Returns false if no virus present */
PHP_FUNCTION(clam_scan_buffer) {
	zval **buffer;
	char *real_buffer = NULL;
	long arg_len, ret, argc = ZEND_NUM_ARGS( );
	const char *virname;
	char *result;
	
	/* make sure virus dbdir exists - prevents a segfault */
	if (cl_error) {
		php_error( E_WARNING, "Virus database directory (%s) does not exist",cl_retdbdir()) ;
		RETURN_FALSE;
	}
	
	/* parse zend input */
	if (ZEND_NUM_ARGS() != 1 | zend_get_parameters_ex(1, &buffer) == FAILURE) {
		WRONG_PARAM_COUNT;
	}
	
	/* convert zval input to real strings */
	real_buffer = Z_STRVAL_PP(buffer);
	
	/* scan buffer */
	ret = cl_scanbuff(real_buffer ,strlen(real_buffer), &virname, root);
	if (ret == CL_VIRUS) {
		RETURN_STRINGL((char *)virname,strlen(virname),1);
	} else if (ret == CL_CLEAN) {
		RETURN_FALSE;
	} else {
		php_error(E_WARNING,"error: %s", cl_strerror(ret));
		RETURN_FALSE;
	}
}
/* }}} */


/* {{{ proto string clam_scan_file(string $file)
   Scan a given file for viruses. Returns false if no virus present */
PHP_FUNCTION(clam_scan_file) {
	zval **file;
	char *real_file = NULL;
	long options = CL_SCAN_STDOPT;
	long arg_len, ret, argc = ZEND_NUM_ARGS( );
	const char *virname, *result = NULL;

	/* make sure virus dbdir exists - prevents a segfault */
	if (cl_error) {
		php_error( E_WARNING, "Virus database directory (%s) does not exist",cl_retdbdir()) ;
		RETURN_FALSE;
	}
	
	/* parse zend input */
	if (ZEND_NUM_ARGS() != 1 | zend_get_parameters_ex(1, &file) == FAILURE) {
		WRONG_PARAM_COUNT;
	}
	
	/* convert zval input to real strings */
	real_file = Z_STRVAL_PP(file);
	
	/* scan file */
	/*RETURN_STRINGL(real_file,strlen(real_file),1);*/
	ret = cl_scanfile(real_file, &virname, NULL, root, &clam_limits, options);
	/* return result */
	if (ret == CL_VIRUS) {
		RETURN_STRINGL((char *)virname,strlen(virname),1);
	} else if (ret == CL_CLEAN) {
		RETURN_FALSE;
	} else {
		php_error(E_WARNING,"error: %s", cl_strerror(ret));
		RETURN_FALSE;
	}
}
/* }}} */


PHP_FUNCTION(clam_get_version) {
	const char *clam_v = cl_retver();
	RETURN_STRINGL((char *)clam_v,strlen(clam_v),1);
}

PHP_FUNCTION(clam_test) {
	/*const char *clam_v = cl_retver();*/
	
	/* returns "3"
	out = cl_retflevel();
	RETURN_LONG(out);
	*/
	
}

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim600: noet sw=4 ts=4 fdm=marker
 * vim<600: noet sw=4 ts=4
 */
