/**
 * @brief 
 * 
 * This file is a part of PFSTOOLS package.
 * ---------------------------------------------------------------------- 
 * Copyright (C) 2003,2004 Rafal Mantiuk and Grzegorz Krawczyk
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * ---------------------------------------------------------------------- 
 * 
 * @author Rafal Mantiuk, <mantiuk@mpi-sb.mpg.de>
 *
 * $Id: ppmio.cpp,v 1.2 2007/01/23 13:02:18 gkrawczyk Exp $
 */

#include "ppmio.h"

extern "C" {
 #include <ppm.h>
}

#include <assert.h>

struct PPMData
{
    pixval maxPV;
    int formatP;
};

static inline float clamp( const float v, const float minV, const float maxV )
{
    if( v < minV ) return minV;
    if( v > maxV ) return maxV;
    return v;
}

PPMReader::PPMReader( FILE *fh ) : fh(fh)
{
    data = new PPMData;

    ppm_readppminit(  fh, &width, &height, &data->maxPV, &data->formatP );
}

bool PPMReader::is8bitdata()
{
    return ( data->maxPV<=255 ) ? true : false; 
}

void PPMReader::readImage( pfs::Array2D *X, pfs::Array2D *Y, pfs::Array2D *Z )
{
    pixel *ppmRow;
    ppmRow = ppm_allocrow( width );
    assert( ppmRow != NULL );

    bool data8bit = is8bitdata();

    for( int y = 0; y < height; y++ ) { // For each row of the image

        ppm_readppmrow( fh, ppmRow, width, data->maxPV, data->formatP );

        //!! 8bit data should be normalized to 0..1 range
        // but 16bit data should be output without change.
        // (Grzesiek)
        if( data8bit )
          for( int x = 0; x < width; x++ ) {
            (*X)(x,y) = (float)PPM_GETR(ppmRow[x]) / 255.f;
            (*Y)(x,y) = (float)PPM_GETG(ppmRow[x]) / 255.f;
            (*Z)(x,y) = (float)PPM_GETB(ppmRow[x]) / 255.f;
          }
        else
          for( int x = 0; x < width; x++ ) {
            (*X)(x,y) = (float)PPM_GETR(ppmRow[x]);
            (*Y)(x,y) = (float)PPM_GETG(ppmRow[x]);
            (*Z)(x,y) = (float)PPM_GETB(ppmRow[x]);
          }

    }
    ppm_freerow( ppmRow );
}


PPMReader::~PPMReader()
{
    delete data;
}



void PPMWriter::writeImage( pfs::Array2D *X, pfs::Array2D *Y, pfs::Array2D *Z )
{
    pixel *ppmRow;
    int width = X->getCols();
    int height =  X->getRows();
    ppmRow = ppm_allocrow( width );
    assert( ppmRow != NULL );

    ppm_writeppminit( fh, width, height, 255, false );

    for( int y = 0; y < height; y++ ) { // For each row of the image

        for( int x = 0; x < width; x++ ) {
            PPM_ASSIGN( ppmRow[x],
                (pixval)( clamp( (*X)( x, y )*255.f, 0.f, 255.f) ),
                (pixval)( clamp( (*Y)( x, y )*255.f, 0.f, 255.f) ),
                (pixval)( clamp( (*Z)( x, y )*255.f, 0.f, 255.f) ) );
        }
        ppm_writeppmrow( fh, ppmRow, width, 255, false );

    }
    ppm_freerow( ppmRow );

}
