/*
    scconfig - detection of POSIX library features: process specific calls
    Copyright (C) 2018  Tibor Palinkas
    Copyright (C) 2018  Aron Barath

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

		Project page: http://repo.hu/projects/scconfig
		Contact via email: scconfig [at] igor2.repo.hu
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include "libs.h"
#include "log.h"
#include "db.h"
#include "dep.h"

int find_posix_getsid(const char *name, int logdepth, int fatal)
{
	const char *key = "libs/posix/proc/getsid";
	const char *test_c =
		NL "#include <stdio.h>"
		NL "int main()"
		NL "{"
		NL "	if (getsid((pid_t)0) != ((pid_t)(-1)))"
		NL "		puts(\"OK\");"
		NL "	return 0;"
		NL "}"
		NL;

	require("cc/cc", logdepth, fatal);

	report("Checking for getsid()... ");
	logprintf(logdepth, "find_posix_getsid: trying to find getsid()...\n");
	logdepth++;

	if (try_icl(logdepth, key, test_c, "#include <unistd.h>", NULL, NULL))
		return 0;
	if (try_icl(logdepth, key, test_c, "#include <sys/types.h>\n#include <unistd.h>", NULL, NULL))
		return 0;

	return try_fail(logdepth, key);
}

int find_posix_getpid(const char *name, int logdepth, int fatal)
{
	const char *key = "libs/posix/proc/getpid";
	const char *test_c =
		NL "#include <stdio.h>"
		NL "int main()"
		NL "{"
		NL "	if (getpid() != ((pid_t)(-1)))"
		NL "		puts(\"OK\");"
		NL "	return 0;"
		NL "}"
		NL;

	require("cc/cc", logdepth, fatal);

	report("Checking for getpid()... ");
	logprintf(logdepth, "find_posix_getpid: trying to find getpid()...\n");
	logdepth++;

	if (try_icl(logdepth, key, test_c, "#include <unistd.h>", NULL, NULL))
		return 0;
	if (try_icl(logdepth, key, test_c, "#include <sys/types.h>\n#include <unistd.h>", NULL, NULL))
		return 0;

	return try_fail(logdepth, key);
}

int find_posix_getppid(const char *name, int logdepth, int fatal)
{
	const char *key = "libs/posix/proc/getppid";
	const char *test_c =
		NL "#include <stdio.h>"
		NL "int main()"
		NL "{"
		NL "	if (getppid() != ((pid_t)(-1)))"
		NL "		puts(\"OK\");"
		NL "	return 0;"
		NL "}"
		NL;

	require("cc/cc", logdepth, fatal);

	report("Checking for getppid()... ");
	logprintf(logdepth, "find_posix_getppid: trying to find getppid()...\n");
	logdepth++;

	if (try_icl(logdepth, key, test_c, "#include <unistd.h>", NULL, NULL))
		return 0;
	if (try_icl(logdepth, key, test_c, "#include <sys/types.h>\n#include <unistd.h>", NULL, NULL))
		return 0;

	return try_fail(logdepth, key);
}

int find_posix_getpgid(const char *name, int logdepth, int fatal)
{
	const char *key = "libs/posix/proc/getpgid";
	const char *test_c =
		NL "#include <stdio.h>"
		NL "int main()"
		NL "{"
		NL "	if (getpgid(getpid()) != ((pid_t)(-1)))"
		NL "		puts(\"OK\");"
		NL "	return 0;"
		NL "}"
		NL;

	require("cc/cc", logdepth, fatal);
	if (require("libs/posix/proc/getpid", logdepth, fatal) != 0)
		return try_fail(logdepth, key);

	report("Checking for getpgid()... ");
	logprintf(logdepth, "find_posix_getpgid: trying to find getpgid()...\n");
	logdepth++;

	if (try_icl(logdepth, key, test_c, "#include <unistd.h>", NULL, NULL))
		return 0;
	if (try_icl(logdepth, key, test_c, "#include <sys/types.h>\n#include <unistd.h>", NULL, NULL))
		return 0;

	return try_fail(logdepth, key);
}
