/*
    libparted - a library for manipulating disk partitions
    Copyright (C) 1998, 1999, 2000 Free Software Foundation, Inc.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef PED_DISK_H_INCLUDED
#define PED_DISK_H_INCLUDED

typedef struct _PedGeometry		PedGeometry;
typedef struct _PedDisk			PedDisk;
typedef struct _PedPartition		PedPartition;
typedef const struct _PedDiskOps	PedDiskOps;
typedef struct _PedDiskType		PedDiskType;

#include <parted/device.h>
#include <parted/filesys.h>
#include <parted/natmath.h>

struct _PedGeometry {
	PedDisk*		disk;
	PedSector		start;
	PedSector		length;
	PedSector		end;
};

typedef enum {
	PED_PARTITION_PRIMARY		= 0x00,
	PED_PARTITION_LOGICAL		= 0x01,
	PED_PARTITION_EXTENDED		= 0x02,
	PED_PARTITION_FREESPACE		= 0x04,
	PED_PARTITION_METADATA		= 0x08
} PedPartitionType;

typedef enum {
	PED_PARTITION_BOOT=1,
	PED_PARTITION_ROOT=2,
	PED_PARTITION_SWAP=3,
	PED_PARTITION_HIDDEN=4,
	PED_PARTITION_RAID=5,
	PED_PARTITION_LVM=6,
	PED_PARTITION_HPSERVICE=7,
	PED_PARTITION_LBA=8
} PedPartitionFlag;
#define PED_PARTITION_FIRST_FLAG	PED_PARTITION_BOOT
#define PED_PARTITION_LAST_FLAG		PED_PARTITION_LBA

typedef enum {
	PED_DISK_TYPE_EXTENDED=1,
	PED_DISK_TYPE_PARTITION_NAME=2
} PedDiskTypeFeature;
#define PED_DISK_TYPE_FIRST_FEATURE	PED_DISK_TYPE_EXTENDED
#define PED_DISK_TYPE_LAST_FEATURE	PED_DISK_TYPE_PARTITION_NAME

struct _PedPartition {
	PedPartition*		prev;
	PedPartition*		next;

	PedGeometry		geom;
	int			num;

	PedPartitionType	type;
	const PedFileSystemType* fs_type;
	PedPartition*		part_list;	/* for extended partitions */

	void*			disk_specific;
};

struct _PedDisk {
	PedDevice*		dev;
	PedDiskType*		type;
	PedPartition*		part_list;

	void*			disk_specific;

/* office use only ;-) */
	int			update_mode;
};

struct _PedDiskOps {
	/* disk label operations */
	int (*probe) (const PedDevice *dev);
	PedDisk* (*open) (PedDevice* dev);
	PedDisk* (*create) (PedDevice* dev);
	int (*clobber) (PedDevice* dev);
	int (*close) (PedDisk* disk);
	int (*read) (PedDisk* disk);
	int (*write) (PedDisk* disk);

	/* partition operations */
	PedPartition* (*partition_new) (
		const PedDisk* disk,
	       	PedPartitionType part_type,
		const PedFileSystemType* fs_type,
		PedSector start,
		PedSector end);
	void (*partition_destroy) (PedPartition* part);
	int (*partition_set_flag) (
		PedPartition* part,
	       	PedPartitionFlag flag,
		int state);
	int (*partition_get_flag) (
		const PedPartition* part,
		PedPartitionFlag flag);
	int (*partition_is_flag_available) (
		const PedPartition* part,
		PedPartitionFlag flag);
	void (*partition_set_name) (PedPartition* part, const char* name);
	const char* (*partition_get_name) (const PedPartition* part);
	int (*partition_align) (PedPartition* part,
				const PedConstraint* constraint);
	int (*partition_enumerate) (PedPartition* part);
	int (*partition_set_extended_system) (PedPartition* part);

	/* other */
	int (*alloc_metadata) (PedDisk* disk);
	int (*get_max_primary_partition_count) (const PedDisk* disk);
};

struct _PedDiskType {
	PedDiskType*		next;
	const char*		name;
	PedDiskOps* const	ops;

	PedDiskTypeFeature	features;
};

extern void ped_register_disk_type (PedDiskType* type);
extern void ped_unregister_disk_type (PedDiskType* type);
extern PedDiskType* ped_disk_type_get_next (PedDiskType* type);
extern PedDiskType* ped_disk_type_get (const char* name);
extern PedDiskType* ped_disk_probe (const PedDevice* dev);
extern int ped_disk_type_check_feature (const PedDiskType* disk_type,
	       				PedDiskTypeFeature feature);

extern PedDisk* ped_disk_open (PedDevice* dev);
extern PedDisk* ped_disk_create (PedDevice* dev, PedDiskType* type);
extern int ped_disk_clobber (PedDisk* disk);
extern int ped_disk_close (PedDisk* disk);
extern int ped_disk_read (PedDisk* disk);
extern int ped_disk_write (PedDisk* disk);
extern int ped_disk_is_busy (const PedDisk* disk);

int ped_disk_get_primary_partition_count (PedDisk* disk);
int ped_disk_get_last_partition_num (PedDisk* disk);
extern int ped_disk_get_max_primary_partition_count (const PedDisk* disk);

extern PedDisk* ped_disk_alloc (PedDevice* dev, PedDiskType* type);
extern void ped_disk_free (PedDisk* disk);

extern int ped_geometry_init (PedGeometry* geom, const PedDisk* disk,
			      PedSector start, PedSector length);
extern PedGeometry* ped_geometry_new (const PedDisk* disk, PedSector start,
				      PedSector length);
extern PedGeometry* ped_geometry_duplicate (const PedGeometry* geom);
extern PedGeometry* ped_geometry_intersect (const PedGeometry* a,
	       				    const PedGeometry* b);
extern void ped_geometry_destroy (PedGeometry* geom);
extern int ped_geometry_set (PedGeometry* geom, PedSector start,
			     PedSector length);
extern int ped_geometry_set_start (PedGeometry* geom, PedSector start);
extern int ped_geometry_set_end (PedGeometry* geom, PedSector end);
extern int ped_geometry_test_overlap (const PedGeometry* a,
	       			      const PedGeometry* b);
extern int ped_geometry_test_inside (const PedGeometry* a,
				     const PedGeometry* b);
extern int ped_geometry_test_equal (const PedGeometry* a, const PedGeometry* b);
extern int ped_geometry_test_sector_inside (const PedGeometry* geom,
	       				    PedSector sect);

extern int ped_geometry_read (const PedGeometry* geom, void* buffer,
			      PedSector offset, PedSector count);
extern int ped_geometry_write (PedGeometry* geom, const void* buffer,
			       PedSector offset, PedSector count);
extern PedSector ped_geometry_check (PedGeometry* geom, void* buffer,
				     PedSector buffer_size, PedSector offset,
				     PedSector granularity, PedSector count);
extern int ped_geometry_sync (PedGeometry* geom);

/* returns -1 if "sector" is not within dest's space. */
extern PedSector ped_geometry_map (const PedGeometry* dst,
				   const PedGeometry* src,
				   PedSector sector);

extern PedPartition* ped_partition_alloc (const PedDisk* disk,
					  PedPartitionType type,
					  const PedFileSystemType* fs_type,
					  PedSector start,
					  PedSector end);
extern PedPartition* ped_partition_new (const PedDisk* disk,
					PedPartitionType type,
					const PedFileSystemType* fs_type,
					PedSector start,
					PedSector end);
extern void ped_partition_destroy (PedPartition* part);
extern int ped_partition_is_active (const PedPartition* part);
extern int ped_partition_set_flag (PedPartition* part, PedPartitionFlag flag,
	       			   int state);
extern int ped_partition_get_flag (const PedPartition* part,
	       			   PedPartitionFlag flag);
extern int ped_partition_is_flag_available (const PedPartition* part,
	       			      	    PedPartitionFlag flag);
extern int ped_partition_set_system (PedPartition* part,
				     const PedFileSystemType* fs_type);
extern int ped_partition_set_name (PedPartition* part, const char* name);
extern const char* ped_partition_get_name (const PedPartition* part);
extern int ped_partition_is_busy (const PedPartition* part);

extern const char* ped_partition_type_get_name (PedPartitionType part_type);
extern const char* ped_partition_flag_get_name (PedPartitionFlag flag);
extern PedPartitionFlag ped_partition_flag_get_by_name (const char* name);
extern PedPartitionFlag ped_partition_flag_next (PedPartitionFlag flag);

extern int ped_disk_add_partition (PedDisk* disk, PedPartition* part,
				   const PedConstraint* constraint);
extern int ped_disk_remove_partition (PedDisk* disk, PedPartition* part);
extern int ped_disk_delete_partition (PedDisk* disk, PedPartition* part);
extern int ped_disk_delete_all (PedDisk* disk);
extern int ped_disk_set_partition_geom (PedDisk* disk, PedPartition* part,
					const PedConstraint* constraint,
					PedSector start, PedSector end);
extern int ped_disk_maximize_partition (PedDisk* disk, PedPartition* part,
					const PedConstraint* constraint);
extern PedGeometry* ped_disk_get_max_partition_geometry (PedDisk* disk,
		PedPartition* part, const PedConstraint* constraint);
extern int ped_disk_minimize_extended_partition (PedDisk* disk);

extern PedPartition* ped_disk_next_partition (const PedDisk* disk,
					      const PedPartition* part);
extern PedPartition* ped_disk_get_partition (const PedDisk* disk, int num);
extern PedPartition* ped_disk_get_partition_by_sector (const PedDisk* disk,
						       PedSector sect);
extern PedPartition* ped_disk_extended_partition (const PedDisk* disk);

#endif /* PED_DISK_H_INCLUDED */

