(*---------------------------------------------------------------------------*
  IMPLEMENTATION  cf_nameinfo.ml

  Copyright (c) 2003-2005, James H. Woodyatt
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

    Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
  FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
  COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
  INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
  OF THE POSSIBILITY OF SUCH DAMAGE. 
 *---------------------------------------------------------------------------*)

type unresolved_t =
    EAI_ADDRFAMILY | EAI_AGAIN | EAI_BADFLAGS | EAI_FAIL | EAI_FAMILY |
    EAI_MEMORY | EAI_NODATA | EAI_NONAME | EAI_SERVICE | EAI_SOCKTYPE |
    EAI_BADHINTS | EAI_PROTOCOL | EAI_UNKNOWN of int

exception Unresolved of unresolved_t

let _ = Callback.register_exception
    "Cf_nameinfo.Unresolved" (Unresolved (EAI_UNKNOWN (-1)))

external error_message: unresolved_t -> string = "cf_nameinfo_error_message"

;;
external init_: unit -> unit = "cf_nameinfo_init";;
init_ ();;

module AF = struct
    type tag_t = unit
    type address_t = unit Cf_socket.sockaddr_t
    
    external domain_:
        unit -> unit Cf_socket.domain_t = "cf_nameinfo_domain"

    let domain = domain_ ()
    
    let to_sockaddr = Obj.magic
    let of_sockaddr = Obj.magic
    
    external unspecified_: unit -> address_t = "cf_nameinfo_unspecified"
    
    let unspecified = unspecified_ ()
end

external specialize_sockaddr:
    unit Cf_socket.sockaddr_t -> 'af Cf_socket.domain_t ->
    'af Cf_socket.sockaddr_t option = "cf_nameinfo_specialize_sockaddr"

type of_address_flags_t = {
    ni_nofqdn: bool;
    ni_numerichost: bool;
    ni_namereqd: bool;
    ni_numericserv: bool;
    ni_dgram: bool;
}

let of_address_default_flags = {
    ni_nofqdn = false;
    ni_numerichost = false;
    ni_namereqd = false;
    ni_numericserv = false;
    ni_dgram = false;
}

external of_address:
    ?host:int -> ?serv:int -> ?flags:of_address_flags_t ->
    'a Cf_socket.sockaddr_t -> string * string = "cf_nameinfo_of_address"

type to_address_flags_t = {
    ai_passive: bool;
    ai_canonname: bool;
    ai_numerichost: bool;
}

let to_address_default_flags = {
    ai_passive = false;
    ai_canonname = false;
    ai_numerichost = false;
}

type to_address_arg_t =
    | A_nodename of string
    | A_servicename of string
    | A_bothnames of string * string

type addrinfo_t = {
    ai_flags: to_address_flags_t;
    ai_family: unit Cf_socket.domain_t;
    ai_socktype: unit Cf_socket.socktype_t;
    ai_protocol: Cf_socket.protocol_t;
    ai_cname: string option;
    ai_addr: AF.address_t;
}

let nzero_ = Nativeint.of_int 0

external default_ai_flags_:
    unit -> to_address_flags_t = "cf_nameinfo_default_ai_flags"

let addrinfo_default_hint = {
    ai_flags = default_ai_flags_ ();
    ai_family = AF.domain;
    ai_socktype = Obj.magic nzero_;
    ai_protocol = Obj.magic nzero_;
    ai_cname = None;
    ai_addr = AF.unspecified;
}

external to_address:
    ?hint:addrinfo_t -> to_address_arg_t -> addrinfo_t list =
    "cf_nameinfo_to_address"

(*--- End of File [ cf_nameinfo.ml ] ---*)
