//
// C++ Implementation: aptfrontpackagedb
//
// Description: 
//
//
// Author: Benjamin Mesing <bensmail@gmx.net>, (C) 2005
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "aptfrontpackagedb.h"
#include "aggregator.h"
#include <ept/apt/apt.h>
#include <ept/apt/packagerecord.h>
#include <ept/textsearch/textsearch.h>
#include <xapian.h>

#include <QStringList>

// NPlugin
#include <iprovider.h>
#include <packagenotfoundexception.h>

#include <helpers.h>


namespace NApt {

AptFrontPackageDB::AptFrontPackageDB(NPlugin::IProvider* pProvider) :
	_pProvider(pProvider),
	_currentPackage(_pProvider->aptFrontCache(), std::string())
{
}


AptFrontPackageDB::~AptFrontPackageDB()
{
}


const AptFrontPackage& AptFrontPackageDB::getPackageRecord(const QString& pkg) const
{
	return getPackageRecord(toString(pkg));
}


const AptFrontPackage& AptFrontPackageDB::getPackageRecord(const string& package) const
{
	_currentPackage = AptFrontPackage
	(
		_pProvider->aptFrontCache(), _pProvider->aptFrontCache().apt().validate(package)
	);
	if (!_currentPackage.isValid())
		throw NPlugin::PackageNotFoundException(package);
	return _currentPackage;
}

const QString AptFrontPackageDB::getShortDescription(const string& package) const
{
	return getPackageRecord(package).shortDescription();
}

IPackage::InstalledState AptFrontPackageDB::getState(const string& package) const
{
	return getPackageRecord(package).installedState();
}

void AptFrontPackageDB::reloadPackageInformation(NUtil::IProgressObserver* pObserver)
{
	_pProvider->reloadAptFrontCache();
}

int AptFrontPackageDB::getDescriptionCount(const string& package, const QString& pattern, bool caseSensitive) const
{
	return 1;
}
bool AptFrontPackageDB::matchesName(const string& package, const QString& pattern, bool caseSensitive) const
{
}



/////////////////////////////////////////////////////
// IAptSearch Interface
/////////////////////////////////////////////////////

bool AptFrontPackageDB::search(std::set<string>& result,
		const QString& pattern, bool searchDescr, bool caseSensitive) const
{
	typedef ept::apt::Apt Apt;
	typedef ept::textsearch::TextSearch TextSearch;
	Apt& packages = _pProvider->aptFrontCache().apt();
	TextSearch& textsearch = _pProvider->aptFrontCache().textsearch();
	if (searchDescr)
	{
		if (textsearch.hasData())
		{
			Xapian::Enquire enq(textsearch.db());
			enq.set_query(textsearch.makeORQuery(pattern.ascii()));
// 			textsearch.search(enq, pattern.ascii());
			// Get the 50 best results out of Xapian
			Xapian::MSet matches = enq.get_mset(0, 50);
			for (Xapian::MSetIterator i = matches.begin(); i != matches.end(); ++i)
			{
				// Filter out results that apt doesn't know
				if (!packages.isValid(i.get_document().get_data()))
					continue;
				result.insert(i.get_document().get_data());
				// For reference, this would be the Xapian relevance
				// i.get_percent();
			}
		} else {
			ept::apt::PackageRecord rec;
			for ( Apt::record_iterator it = packages.recordBegin(); it != packages.recordEnd(); ++it )
			{
				rec.scan(*it);
				if ( toQString(rec.package()).contains(pattern, caseSensitive) || 
					toQString(rec.description()).contains(pattern, caseSensitive))
				{
					result.insert(rec.package());
				}
			}
		}
	} else {
		// Much faster version if we only search names
		for ( Apt::iterator it = packages.begin(); it != packages.end(); ++it )
			if ( toQString(*it).contains(pattern, caseSensitive))
				result.insert(*it);
	}
	return result.empty();
}


bool AptFrontPackageDB::search(std::set<string>& result, const QStringList& includePatterns, 
	const QStringList& excludePatterns, bool searchDescr, bool caseSensitive) const
{
	typedef ept::apt::Apt Apt;
	typedef ept::textsearch::TextSearch TextSearch;
	Apt& packages = _pProvider->aptFrontCache().apt();
	TextSearch& textsearch = _pProvider->aptFrontCache().textsearch();
	if (searchDescr)
	{
		ept::apt::PackageRecord rec;
		if (textsearch.hasData())
		{
			//try {
			Xapian::Enquire enq(textsearch.db());
			vector<string> patterns;
			// Get the weighted OR include query
			for (QStringList::const_iterator jt = includePatterns.begin(); jt != includePatterns.end(); ++jt)
				patterns.push_back(jt->ascii());
			Xapian::Query inc = textsearch.makeORQuery(patterns.begin(), patterns.end());
			if (excludePatterns.empty())
			{
				// Build the normal OR query
				enq.set_query(inc);
			} else {
				// Get the weighted OR exclude query
				patterns.clear();
				for (QStringList::const_iterator jt = excludePatterns.begin(); jt != excludePatterns.end(); ++jt)
					patterns.push_back(jt->ascii());
				Xapian::Query exc = textsearch.makeORQuery(patterns.begin(), patterns.end());
				// Build the combined AND NOT query
				enq.set_query(Xapian::Query(Xapian::Query::OP_AND_NOT, inc, exc));
			}

			// Get the 50 best results out of Xapian
			Xapian::MSet matches = enq.get_mset(0, 50);
			for (Xapian::MSetIterator i = matches.begin(); i != matches.end(); ++i)
			{
				// Filter out results that apt doesn't know
				if (!packages.isValid(i.get_document().get_data()))
					continue;
				result.insert(i.get_document().get_data());
				// For reference, this would be the Xapian relevance
				// i.get_percent();
			}
			/*
			// FIXME: Xapian has its own exceptions that you may
			// want to catch somewhere
			} catch (Xapian::Error& e) {
				fprintf(stderr, "%s: %s (%s)\n",
					e.get_type().c_str(),
					e.get_msg().c_str(),
					e.get_context().c_str());
			}
			*/
		} else {
			for ( Apt::record_iterator it = packages.recordBegin(); it != packages.recordEnd(); ++it )
			{
				rec.scan(*it);
				QString name = toQString(rec.package());
				QString desc = toQString(rec.description());
				// holds if the search matches the package
				bool included = true;
				// check if each included pattern occurs in the package
				for (QStringList::const_iterator jt = includePatterns.begin(); jt != includePatterns.end(); ++jt)
				{
					if ( ! ( name.contains(*jt, caseSensitive) ||
						    desc.contains(*jt, caseSensitive) ) )
					{
						included = false;
						break;
					}
				}
				if (!included)
					continue;
				// check if each excluded pattern does not occur in the package
				for (QStringList::const_iterator jt = excludePatterns.begin(); jt != excludePatterns.end(); ++jt)
				{
					if ( name.contains(*jt, caseSensitive) ||
						  desc.contains(*jt, caseSensitive))
					{
						included = false;
						break;
					}
				}
				if (included)
					result.insert(rec.package());
			}
		}
	} else {
		for ( Apt::iterator it = packages.begin(); it != packages.end(); ++it )
		{
			QString name = toQString(*it);
			// holds if the search matches the package
			bool included = true;
			// check if each included pattern occurs in the package
			for (QStringList::const_iterator jt = includePatterns.begin(); jt != includePatterns.end(); ++jt)
			{
				if ( !name.contains(*jt, caseSensitive) )
				{
					included = false;
					break;
				}
			}
			if (!included)
				continue;
			// check if each excluded pattern does not occur in the package
			for (QStringList::const_iterator jt = excludePatterns.begin(); jt != excludePatterns.end(); ++jt)
			{
				if ( name.contains(*jt, caseSensitive) )
				{
					included = false;
					break;
				}
			}
			if (included)
				result.insert(*it);
		}
	}
	return result.empty();
}

}

#undef emit
//#include <ept/cache/apt/index.tcc>
//#include <ept/cache/apt/state.tcc>
