//
// C++ Interface: %{MODULE}
//
// Description: 
//
//
// Author: %{AUTHOR} <%{EMAIL}>, (C) %{YEAR}
//
// Copyright: See COPYING file that comes with this distribution
//
//
#ifndef __NWIDGET_CUSTOMIZABLELISTVIEW_H_2005_04_20
#define __NWIDGET_CUSTOMIZABLELISTVIEW_H_2005_04_20

#include <map>
#include <list>

#include <qobject.h>
#include <q3listview.h>

#include <xmldata.h>
#include <ixmlstorable.h>

using namespace std;

namespace NWidget {

/** @brief Class that handles a ListView and allows to customize its columns.
  *
  * This class is capable of loading and saving the state including column ordering,
  * width and display status (shown or hidden).<br>
  * The column captions must be unique.
  *
  * This class would better subclass QListView, but as this works very poorly with 
  * the QT Designer approach I decided against this.
  * @author Benjamin Mesing
  */
class CustomizableListView : public QObject, public NXml::IXmlStorable
{
Q_OBJECT
protected:
	/** The state for one column. */
	struct Column
	{
		Column(QString caption, int width) :
			_caption(caption),
			_width(width)
		{
		}
		QString _caption;
		int _width;
	};
	
	/** List of columns with some convenience functions. */
	class ColumnList : public list<Column>
	{
	public:
		/** @brief Finds the column with the given caption. */
		list<Column>::const_iterator find(QString caption) const
		{
			for (const_iterator it = begin(); it != end(); ++it)
			{
				if (it->_caption == caption)
					return it;
			}
			return end();
		}	
		/** @brief Finds the column with the given caption (non-const version). */
		list<Column>::iterator find(QString caption)
		{
			for (iterator it = begin(); it != end(); ++it)
			{
				if (it->_caption == caption)
					return it;
			}
			return end();
		}
		
		using list<Column>::insert;
		/** @brief Inserts the handed column at position <i>index</i>.
		  *
		  * If i > size() the column will be attached.
		  */
		void insert(int index, const Column& c)
		{
			int i=0;
			for (iterator it = begin(); it != end(); ++it, ++i)
			{
				if (i == index)
				{
					insert(it, c);
					return;
				}
			}
			// if index was greater size()
			insert(end(), c);
		}
		
		/** @brief Returns a list of the captions of all columns.
		  *
		  * The order corresponds to the order of the columns.
		  **/
		QStringList getCaptions() const
		{
			QStringList list;
			for (const_iterator it = begin(); it != end(); ++it)
			{
				list.push_back(it->_caption);
			}
			return list;
		}
	};
	
private:
	/** The list view controlled by this class. */
	Q3ListView* _pLv;
	
	/** @brief List of the shown columns. 
	  * 
	  * This does not always reflect the order of the corresponding columns.
	  */
	ColumnList _shownColumns;
	/** @brief List the hidden columns.  */
	ColumnList _hiddenColumns;

	/** @brief Holds the column by which to sort. 
	  *
	  * It will be saved and loaded. When no column of the given name does
	  * exists, the list will be sorted by the first column.
	  * The state if to sort acending or descending will be hold by the list-
	  * views entry for the column.
	  *
	  * By default the list is not sorted, but the user can select a column for sorting.
	  *
	  * This value is always in sync with the listview state.
	  */
	QString _sortColumn;
	/** @brief Holds the sort order of the column to be sorted.
	  *
	  * Works like #_sortColumn. 
	  * <b>Default value:</b> Qt::Ascending
	  */
	Qt::SortOrder _sortOrder;
protected slots:
	/** @brief This refreshes #_shownColumns according to the currently displayed state. 
	  *
	  * Call this whenever the user changes the column ordering or size.
	  */
	void refreshShown();
	/** @brief This refreshes the column sorting.
	  *
	  * Call this whenever the user changes the column sorting.
	  */
	void refreshSorting();
signals:
	/** Emitted whenever the set of shown columns was changed. */
	void shownColumnsChanged();
protected:
	/** @brief Updates the list view according to the items in #_shownColumns. */
	void updateListView();
	/** @brief This gets the width of the column with the given caption. 
	  *
	  * It searches the hidden and shown column.
	  * @returns the width of the column or -1 if no such column was found. */
	int getColumnWidth(QString caption) const;
	
	/** @brief This gets the Column instance for the given caption. 
	  *
	  * It searches the shown and the hidden columns.
	  * @returns the width of the column or #_hiddenColumns.end() if no such column was found. */
	ColumnList::iterator getColumn(QString caption);
public slots:
	/** @brief Shows a control dialog to the user. 
		*
		* The function returns when the dialog is closed.
		*/
	void showControlDialog();

public:
	/**
	  * @param pLv the list view that shall be controlled by this object. 
	  */
	CustomizableListView(Q3ListView* pLv, QObject *parent = 0, const char *name = 0);
	~CustomizableListView();
	
	/** @brief Sets the #_sortColumnProperty.
	  *
	  * If the column is not shown, sorting will not be changed.
	  */
	void setSorting(QString column, Qt::SortOrder sortOrder);
	/** @brief This returns the name of the column by which we currently sort.
	  *
	  * This returns always the correct value. If the list is not sorted, it
	  * returns an empty string.
	  */
	QString sortColumn() const;
	/** @brief Returns if a column with the handed caption is either hidden or shown. */
	bool isExisting(QString caption) const;
	/** @brief Returns if the column with the handed caption is hidden. */
	bool isHidden(QString caption) const;
	/** @brief Adds a new column to the shown columns.
	  *
	  * If the column is already shown or hidden, it will not be inserted.
	  * @param caption caption to be used for the new column
	  * @param width width of the column, if -1 the column is sized automatically.
	  * This parameter will be ignored if a width was loaded from the settings.
	  * @pre name should not be in #_hiddenColumns or #_shownColumns
	  */
	void addColumn(QString caption, int width = -1);
	/** @brief Adds a new column to the shown columns inserting at position <i>position</i>.
	  *
	  * @param caption caption to be used for the new column
	  * @param position the position were to insert the new column 
	  * @param width width of the column, if -1 the column is sized automatically.
	  * This parameter will be ignored if a width was loaded from the settings.
	  * @pre name should not be in #_hiddenColumns or #_shownColumns
	  */
	void addColumn(QString caption, int column, int width);
	/** @brief Removes the given column.
	  *
	  * The column is removed from the #_shownColumns or #_hiddenColumns (only
	  * one entry should be available). No setting for this column will be saved 
	  * any more
	  */
	void removeColumn(QString caption);
	/** @brief This removes all columns from #_shownColumns and #_hiddenColumns which
	  * are not in the handed list. 
	  *
	  * If some of the columns not in the list were shown before, they will also be removed
	  * from the listView.
	  */
	void removeColumnsNotInList(const QStringList columns);
	/** @brief Sets the shown columns explicitly.
	  * 
	  * Normally this should not be needed.
	  * @warning not yet implemented
	  */
	void setShownColumns(QStringList shown) {};
	/** @brief Sets the hidden columns explicitly.
	  * 
	  * Normally this should not be needed.
	  * @warning not yet implemented
	  */
	void setHiddenColumns(QStringList hidden) {};
	virtual void saveSettings(NXml::XmlData& outData, QDomElement parent) const;
	virtual QDomElement loadSettings(QDomElement source);
	/** @brief Returns a list of the columns shown. 
	  *
	  * This list is in the order of how columns shown (i.e. from index 0 to n). 
	  */
	QStringList shownColumns() const	{ return _shownColumns.getCaptions(); }
	/** @brief Maps the given index to the section.
	  * 
	  * The index is the position where the column is currently shown while the 
	  * section changes only if the columns available are changed.
	  *
	  * This function does have meaning for the _shownColumns only. Actually it calls
	  * _pLv->header()->mapToSection(index).
	  */
	int mapToSection(int index) const;
};

};

#endif	//  __NWIDGET_CUSTOMIZABLELISTVIEW_H_2005_04_20
