/*
 *
 * Copyright (C) 2007 Loic Dachary <loic@dachary.org>
 * Copyright (C) 2005, 2006 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * Authors:
 *  Cedric Pinson <cpinson@freesheep.org>
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif // HAVE_CONFIG_H

#ifdef USE_NPROFILE
#include <nprofile/profile.h>
#else  // USE_NPROFILE
#define NPROFILE_SAMPLE(a)
#endif

#include "osgcal.h"
#include <assert.h>
#include "osgviewer.h"
//#include <unistd.h>
#include <cstdio>

#include <string>
#include <vector>
#include <iostream>

#include <limits.h>
#include <stdio.h>
#include <errno.h>
#include <cstdlib>
#include <fcntl.h>

#include <glib.h>

#include <osgUtil/SceneView>
#include <osg/Timer>
#include <osg/Version>
#include <osgDB/ReadFile>
#include <osgDB/FileUtils>
#include <osgText/Text>
#include <osg/CoordinateSystemNode>
#include <osg/MatrixTransform>
#include <osg/Projection>

#include <osgCal/Model>


std::string read_file(const std::string& path) 
{
  gchar* outfit_bytes = NULL;
  GError* error = NULL;
  if(!g_file_get_contents(path.c_str(), &outfit_bytes, NULL, &error)) {
    osg::notify(osg::FATAL) << path << (error ? error->message : "unknown error");
    return "";
  }
  std::string result = outfit_bytes;
  g_free(outfit_bytes);
  return result;
}

Manipulator::Manipulator(osgViewer& viewer, osgCal::Model *model, bool fixcam, bool vertexProgram) : mViewer(viewer) 
{
  mFixcam_flag = fixcam;
	mVertexProgram = vertexProgram;
  mModel=model;
  mNMov=0; 
  mStep = 1000;
  mTarget = osg::Vec3(0,0,0);
  mPosition = osg::Vec3(0,0,0);
  mMouseX = mMouseY = 0;
  mLeftButton = mRightButton = false;
  mViewer.GetSceneView()->setComputeNearFarMode(osgUtil::CullVisitor::DO_NOT_COMPUTE_NEAR_FAR);
}

void Manipulator::InitCamera()
{

  if (mFixcam_flag) {

    mTarget = osg::Vec3(0,0,100);
    mPosition = osg::Vec3(0,-500,0);

    mViewer.GetSceneView()->setViewMatrixAsLookAt(mTarget, mPosition, osg::Vec3(0,0,1));

  }
  else {

    const osg::BoundingSphere& boundingSphere= mModel->getBound();      

    mPosition = boundingSphere._center+osg::Vec3( 0.0,-3.5f * boundingSphere._radius,0.0f);
    mTarget = boundingSphere._center;

    mViewer.GetSceneView()->setViewMatrixAsLookAt(mPosition, mTarget, osg::Vec3(0,0,1));  
  }
}

void Manipulator::RecomputeCamera()
{
  mViewer.GetSceneView()->setViewMatrixAsLookAt(mPosition, mTarget, osg::Vec3(0,0,1));
}

void Manipulator::Update() {

  osg::Vec3 dirTmp = mTarget - mPosition;
  
  int x,y;
  SDL_GetMouseState(&x, &y);
  float xrel = (x - mMouseX);
  float yrel = (y - mMouseY);

  mMouseX = x;
  mMouseY = y;

  // rotate around mTarget
  if (mLeftButton) {

    mPosition = mPosition * osg::Matrix::rotate(-yrel*.01, osg::Vec3(1,0,0),
                                                0, osg::Vec3(0,1,0),
                                                -xrel*.01, osg::Vec3(0,0,1));

  }

  // slide along dirTmp
  if (mRightButton) {

    dirTmp.normalize();
    mPosition = mPosition + ( dirTmp*(yrel*.5) );
  }

}

/** handle events, return true if handled, false otherwise.*/
void Manipulator::Handle(EventList& events) 
{
  for (EventList::iterator it = events.begin(); it != events.end(); it++) {
	  const SDL_Event& evt = *it;
    if (evt.type == SDL_KEYDOWN) {
	    if ( evt.key.keysym.sym == SDLK_m) {

        /*
          mModel->getCalModel()->getMixer()->clearCycle(mNMov, 3);
          mNMov = ++mNMov%6;
          printf("Movement No.%d (Blend time 3sec)\n", mNMov);
          mModel->getCalModel()->getMixer()->blendCycle(mNMov, 1, 3);
          return;
        */
        launchAnim();

      } else if (evt.key.keysym.sym == SDLK_n) {

        applyNextOutfit();
        mComment->setText(mModel->getOutfit()->_comment);
					
        InitCamera();
      } else if (evt.key.keysym.sym == SDLK_p) {

		    mCurrentOutfit--;
        if(mCurrentOutfit < 0) mCurrentOutfit = mOutfits.size() - 1;
        mModel->installOutfitFromXMLString(read_file(mOutfits[mCurrentOutfit]));
        mComment->setText(mModel->getOutfit()->_comment);

        InitCamera();
      } else if (evt.key.keysym.sym == SDLK_SPACE) {
        InitCamera();
      }


    } else if (evt.type == SDL_MOUSEBUTTONDOWN) {

      if (evt.button.button == SDL_BUTTON_LEFT) {
        mLeftButton = true;
      } else if (evt.button.button == SDL_BUTTON_RIGHT) {
        mRightButton = true;
      }
    } else if (evt.type == SDL_MOUSEBUTTONUP) {

      if (evt.button.button == SDL_BUTTON_LEFT) {
        mLeftButton = false;
      } else if (evt.button.button == SDL_BUTTON_RIGHT) {
        mRightButton = false;
      }
	  }
  }
}


void Manipulator::setOutfits(const Outfits& outfits) {
  mOutfits = outfits;
  mCurrentOutfit = 0;
}

int Manipulator::getNbOutfit() {
  return mOutfits.size();
}

void Manipulator::applyNextOutfit() {
  mCurrentOutfit++;
  if(mCurrentOutfit >= (int)mOutfits.size()) mCurrentOutfit = 0;

	if (mVertexProgram == false) {
		mModel->installOutfitFromXMLString(read_file(mOutfits[mCurrentOutfit]));
	}
	else {
		osg::ref_ptr<osgCal::CoreModel> coreModel = mModel->getCoreModel();
		osg::Group *fxGroup = mModel->getFXGroup();
		osg::State *fxState = mModel->getFXState();
		osg::Group *root = (osg::Group*) mViewer.GetRoot()->getChild(0);
		root->removeChild(mModel.get());
		mModel = NULL;
		mModel = new osgCal::Model();
		mModel->setCoreModel(coreModel.get());
		mModel->setUseVertexProgram(true);
		mModel->setFXGroup(fxGroup);
		mModel->setFXState(fxState);
		bool res = mModel->initOutfitFromXMLString(read_file(mOutfits[mCurrentOutfit]));
		if (res == true) {
			if (!mModel->create()) {
				std::cerr << "create: " << CalError::getLastErrorDescription().c_str() << std::endl;
				return;
			}

			if(!mModel->applyParameterFromOutfitDescription())
				printf("applyParameterFromOutfitDescription failed, continuing...\n");

			root->addChild(mModel.get());
		}
	}
}

void Manipulator::SetComment(osgText::Text* comment) { mComment = comment; }

void Manipulator::launchAnim() {
  mModel->getCalModel()->getMixer()->clearCycle(mNMov, 3);
  mModel->getCalModel()->getMixer()->blendCycle(0, 1, 3);
}


bool ParseDirectory(const std::string& dir, std::vector<std::string>& result)
{
  GError* error = 0;
  GDir* dir_descriptor = g_dir_open(dir.c_str(), 0, &error);
  if  (!dir_descriptor) {
    std::cout << "Directory " << dir << " not found\n";
    return false;
  }

  const gchar* file_char;
  while((file_char = g_dir_read_name(dir_descriptor))) {

    std::string file(file_char);
    if(file == "." || file == "..")
      continue;

    result.push_back(dir+G_DIR_SEPARATOR_S+file);
  }

  g_dir_close(dir_descriptor);
  return true;
}



bool GetNumber(int& result,char* number)
{
  char* endptr;
  errno = 0;
  long val = strtol(number,&endptr,10);
  if ((errno == ERANGE && (val == LONG_MAX || val == LONG_MIN))
      || (errno != 0 && val == 0)) {
    return false;
  }

  if (endptr == number) {
    return false;
  }
  result = val;
  return true;
}


std::string GetExtension(const std::string& file)
{
  std::string::size_type dot = file.rfind('.');
  std::string lcpath = file;
  std::string suffix;
  std::transform(lcpath.begin(), lcpath.end(), lcpath.begin(), tolower);
  if(dot != std::string::npos) {
    suffix = lcpath.substr(dot + 1);
  }
  return suffix;
}

ParseArguments::ParseArguments(int ac, char** av) 
{
  mArgc = ac;
  mArgv = av;

  mNumloops = 1;
  mNumframes = 500;
  mVerbose_flag = false;
  mVertexprogram_flag = false;
  mBenchmark_flag = false;
  mNumframes_flag = false;
  mDirectory_flag = false;
  mFile = -1;
  mError_occurred = false;
  mFixcam_flag = false;
  mDirectory = "";
}

bool ParseArguments::Parse() 
{
  std::stringstream error;

  for (int i = 1; i < mArgc; i++) {
    if (!strcmp(mArgv[i],"--benchmark")) {
      mBenchmark_flag = 1;
    } else if (!strcmp(mArgv[i],"--help")) {
      mVerbose_flag = 1;
      mError_occurred = true;
      std::cout << mArgv[0] << " [--benchmark] [--numberloops loops - this option work only with --benchmark] [--numberframe frames] [--vertexprogram] [--fixcam] [--help] cal3d.xfg files.xfg \n";
      std::cout << "or\n";
      std::cout << mArgv[0] << " [--benchmark] [--numberloops loops - this option work only with --benchmark] [--numberframe frames] [--vertexprogram] [--fixcam] [--directory dir] [--help] cal3d.xfg\n";
      return false;
    } else if (!strcmp(mArgv[i],"--vertexprogram")) {
      mVertexprogram_flag = 1;
    } else if (!strcmp(mArgv[i],"--fixcam")) {
      mFixcam_flag = true;
    } else if (!strcmp(mArgv[i],"--directory")) {
      if (i+1 >= mArgc) {
        mError_occurred = true;
        error << "--directory needs one argument";
        break;
      } else {
        mDirectory_flag = true;
        mDirectory = mArgv[i+1];
        i++;
      }
    } else if (!strcmp(mArgv[i],"--numberloops")) {
      if (i+1 < mArgc) {
        bool rc = GetNumber(mNumloops,mArgv[i+1]);
        if (!rc) {
          mError_occurred = true;
          error << "--numberloops needs a numeric argument and have " << std::string(mArgv[i+1]);
          break;
        }
        i++;
      } else {
        mError_occurred = true;
        error << "--numberloops needs one numeric argument";
      }
    } else if (!strcmp(mArgv[i],"--numberframe")) {
      if (i+1 < mArgc) {
        bool rc = GetNumber(mNumframes,mArgv[i+1]);
        if (!rc) {
          mError_occurred = true;
          error << "--numberframe needs a numeric argument and have " << std::string(mArgv[i+1]);
          break;
        }
        mNumframes_flag = true;
        i++;
      } else {
        mError_occurred = true;
        error << "--numberframe needs one numeric argument";
      }
    } else {
      if (mFile == -1)
        mFile = i;
    }
  }

  if (!mError_occurred) {
    if (mNumloops >1 && !mBenchmark_flag) {
      mError_occurred = true;
      error << "you can't use --numberloops without --benchmark";
    }
  }

  if (!mError_occurred) {
    if (mFile != -1) {
      while (mFile < mArgc) {
        mPositional_arguments.push_back(std::string(mArgv[mFile++]));
      }
      
      if (mDirectory_flag) {
        if (mPositional_arguments.size() < 1) {
          mError_occurred = true;
          error << "bad argument with --directory expect \"--directory \"your directory\" cal3d.xfg\" and you have --directory " << std::string(mDirectory) << " and no cal3d.xfg";
        } else {
          std::string file_xfg = mPositional_arguments[0];
          mPositional_arguments.clear();
          mPositional_arguments.push_back(file_xfg);
          if (!ParseDirectory(mDirectory, mPositional_arguments)) {
            mError_occurred = true;
            error << "error while paring directory " << mDirectory;
          }
          // filter all valid files
          for (int i = mPositional_arguments.size()-1; i>=0; i--) {
            std::string suff = GetExtension(mPositional_arguments[i]);
            if(! (suff == "cfg" || suff == "xfg")) {
              mPositional_arguments.erase(mPositional_arguments.begin()+i);
            }
          }
        }
      }
    }
  }

  mPath = "";
  if (!mError_occurred) { 
    if (!mPositional_arguments.empty()) {
      mPath = mPositional_arguments.front();
      mPositional_arguments.erase(mPositional_arguments.begin());
    } else {
      mError_occurred = true;
      error << "arguments missing you need at least a cal3d.xfg file and an outfit.xfg file";
    }
  }

  if (!mError_occurred) { 
    mSuffix = GetExtension(mPath);
    if(! (mSuffix == "cfg" || mSuffix == "xfg")) {
      mError_occurred = true;
      error << "expected a .cfg or .xfg file but got " << mPath << " instead";
    }
  }

  if(mError_occurred) {
    std::cout << error.str() << ", see usage with --help" << std::endl;
    return false;
  }
  return true;
}
