/*
*
* Copyright (C) 2004, 2005 Mekensleep
*
* Mekensleep
* 24 rue vieille du temple
* 75004 Paris
*       licensing@mekensleep.com
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301, USA.
*
* Authors:
*  Igor Kravtchenko <igor@ozos.net>
*
*/

#ifdef WIN32
#include <windows.h>
#include <winbase.h>
#endif

#include <osgCal/TextureLayersFlatten>

#include <osg/BlendFunc>
#include <osg/Group>
#include <osg/Image>
#include <osg/MatrixTransform>
#include <osg/Projection>
#include <osg/TexEnvCombine>
#include <osg/Texture2D>

#include <osgUtil/CullVisitor>
#include <osgUtil/RenderBin>

#ifndef MIN
#define MIN(a, b)  (((a) < (b)) ? (a) : (b))
#endif

#ifndef MAX
#define MAX(a, b)  (((a) > (b)) ? (a) : (b))
#endif

#define SET_VIEWPORT_FROM_TEXTURE_SIZE(a)   vp_w = (a)->getTextureWidth(); vp_h = (a)->getTextureHeight(); \
                                            if (!vp_w) { vp_w = (a)->getImage()->s(); vp_h = (a)->getImage()->t(); }

#define START_INDEX_RBIN -5000000

namespace osgCal {

static bool g_bInit = false;
static osg::ref_ptr<osg::Texture2D> g_tmpTexture[3];
static int g_iRBin = START_INDEX_RBIN;

static std::vector< osg::ref_ptr<osg::Texture2D> > g_filterTexture;

#ifdef SUPPORT_GIF_HLS
std::map< osg::ref_ptr<osg::Texture2D>, osg::ref_ptr<GIFImage> > TextureLayersFlatten::texture2GIFimage_;
#endif

TextureLayersFlatten::TextureLayersFlatten()
{
  if (!g_bInit) {
    g_bInit = true;

    for (int i = 0; i < 3; i++) {
      osg::Texture2D *tex = new osg::Texture2D();
      g_tmpTexture[i] = tex;

      tex->setInternalFormat(GL_RGB);
      tex->setFilter(osg::Texture::MIN_FILTER, osg::Texture::NEAREST);
      tex->setFilter(osg::Texture::MAG_FILTER, osg::Texture::NEAREST);
      tex->setTextureSize(512, 512);
    }
  }
}

void TextureLayersFlatten::destroy()
{
  if (g_bInit) {
    for (int i = 0; i < 3; i++) {
      g_tmpTexture[i] = NULL;
    }
    g_bInit = false;
  }
}

TextureLayersFlatten::~TextureLayersFlatten()
{
  int nbTechnics = technics_.size();
  for (int i = 0; i < nbTechnics; i++) {
    BaseRenderTechnic *technic = technics_[i];
    delete technic;
  }

  int nbFilterTextures = g_filterTexture.size();
  for (int i = 0; i < nbFilterTextures; i++) {
    osg::Texture2D *tex = g_filterTexture[i].get();
    if (tex == maskOn_.get() || tex == outputTexture_.get() ) {
      g_filterTexture.erase( g_filterTexture.begin() + i);
      break;
    }
  }

  if (modelview_abs_.valid() && modelview_abs_->getNumParents() > 0)
    modelview_abs_->getParent(0)->removeChild(modelview_abs_.get());

  if (projection_.valid() && projection_->getNumParents() > 0)
    projection_->getParent(0)->removeChild(projection_.get());

  modelview_abs_ = NULL;
  projection_ = NULL;
  outputTexture_ = NULL;
  colorMask_ = NULL;
  maskOn_ = NULL;
}

void TextureLayersFlatten::resetBinCounter()
{
  g_iRBin = START_INDEX_RBIN;
}

void TextureLayersFlatten::init(int _width, int _height,
                                std::vector<Layer> &_layers,
                                osg::Group *_on,
                                osg::Texture2D *_myOutputTexture,
                                osg::Texture2D *_colorMask,
                                osg::Texture2D *_maskOn,
                                bool _bUseOldLayersParams)
{
  int i;
        //        int j;

  _on->setCullingActive(false);

  if (_layers.size() == 0)
    return;

  if (!_myOutputTexture) {
    outputTexture_ = new osg::Texture2D();
    outputTexture_->setInternalFormat(GL_RGB);
    outputTexture_->setFilter(osg::Texture::MIN_FILTER, osg::Texture::NEAREST);
    outputTexture_->setFilter(osg::Texture::MAG_FILTER, osg::Texture::NEAREST);
  }
  else {
    outputTexture_ = _myOutputTexture;
  }

  float opacity[20];
#ifdef SUPPORT_GIF_HLS
  osg::Vec3f HSL[20];
#endif
  osg::Vec3f color[20];

  int nbOldTechnics = technics_.size();
  for (i = 0; i < nbOldTechnics; i++) {
    BaseRenderTechnic *technic = technics_[i];
    if (_bUseOldLayersParams) {
      opacity[i] = technic->getOpacity();
#ifdef SUPPORT_GIF_HLS
      HSL[i] = technic->getHSL();
#endif
      color[i] = technic->getColorFactor();
    }
    delete technic;
  }
  technics_.clear();

  if (modelview_abs_.valid()) {
    modelview_abs_->removeChild(projection_.get());
    modelview_abs_->getParent(0)->removeChild(modelview_abs_.get());
  }

  modelview_abs_ = NULL;
  projection_ = NULL;

  colorMask_ = _colorMask;
  maskOn_ = _maskOn;

  modelview_abs_ = new osg::MatrixTransform;
  modelview_abs_->setReferenceFrame(osg::Transform::ABSOLUTE_RF);
  modelview_abs_->setMatrix(osg::Matrix::identity());
  projection_ = new osg::Projection;
  modelview_abs_->addChild(projection_.get());
  _on->addChild(modelview_abs_.get());

  int nbLayers = _layers.size();
  orgLayers_.clear();
  for (i = 0; i < nbLayers; i++) {
    orgLayers_.push_back( _layers[i] );
  }

  Layer &layer = orgLayers_[0];

  osg::Texture2D *textureA = layer.texture_.get();
  textureA->setFilter(osg::Texture::MIN_FILTER, osg::Texture::NEAREST);
  textureA->setFilter(osg::Texture::MAG_FILTER, osg::Texture::NEAREST);

  BaseRenderTechnic *technic = NULL;
  technic = new DecalRenderTechnic( this, textureA, outputTexture_.get() );
  technic->setup(projection_.get());

  if (_bUseOldLayersParams && nbOldTechnics > 1) {
    technic->setColorFactor( color[0] );
#ifdef SUPPORT_GIF_HLS
    technic->setHSL( HSL[0] );
#endif
    technic->setOpacity( opacity[0] );
  }

  technics_.push_back(technic);
  technic->createdFromlayer_ = &layer;

  for (i = 1; i < nbLayers; i++) {

    Layer &layer = orgLayers_[i];

    osg::Texture2D *textureA;
    osg::Texture2D *textureB;

    textureA = outputTexture_.get();
    textureB = layer.texture_.get();

    textureA->setFilter(osg::Texture::MIN_FILTER, osg::Texture::NEAREST);
    textureA->setFilter(osg::Texture::MAG_FILTER, osg::Texture::NEAREST);

    textureB->setFilter(osg::Texture::MIN_FILTER, osg::Texture::NEAREST);
    textureB->setFilter(osg::Texture::MAG_FILTER, osg::Texture::NEAREST);

    BaseRenderTechnic *technic = NULL;

    int pixop = layer.pixelOp_;
    switch(pixop) {

      case PIXELOP_NORMAL:
        technic = new NormalRenderTechnic( this, textureA, textureB, outputTexture_.get() );
        break;

      case PIXELOP_OVERLAY:
        technic = new OverlayRenderTechnic(this, textureA, textureB, outputTexture_.get() );
        break;

      case PIXELOP_LINEARDODGE:
        technic = new LinearDodgeRenderTechnic( this, textureA, textureB, outputTexture_.get() );
        break;

      case PIXELOP_MULTIPLY:
        technic = new MultiplyRenderTechnic( this, textureA, textureB, outputTexture_.get() );
        break;

      default:
        break;
    }

    technic->createdFromlayer_ = &layer;
    technic->setup(projection_.get());

    if (_bUseOldLayersParams && i < nbOldTechnics) {
      technic->setColorFactor( color[i] );
#ifdef SUPPORT_GIF_HLS
      technic->setHSL( HSL[i] );
#endif
      technic->setOpacity( opacity[i] );
    }

    technics_.push_back(technic);
  }

  if (colorMask_.valid()) {

    g_filterTexture.push_back( maskOn_.get() );

    _colorMask->setFilter(osg::Texture::MIN_FILTER, osg::Texture::NEAREST);
    _colorMask->setFilter(osg::Texture::MAG_FILTER, osg::Texture::NEAREST);

    setupLastColorMaskPass(projection_.get());
  }
  else {
    g_filterTexture.push_back( outputTexture_.get() );
  }
}

void TextureLayersFlatten::setupLastColorMaskPass(osg::Group *_on)
{
  osg::TexEnvCombine *combiner;
  osg::BlendFunc *bf;
  osg::StateSet *ss;
  MyGeometry *geom;
  QuadParams quadParam;

  quadParam.minPt = osg::Vec2f(-1, -1);
  quadParam.maxPt = osg::Vec2f(1, 1);
  quadParam.bInvertUV = false;

  osg::Group *grp = new osg::Group();
  grp->setName("TextureLayersFlatten_ColorMask");
  _on->addChild(grp);

  int vp_w = maskOn_.get()->getTextureWidth();
  int vp_h = maskOn_.get()->getTextureHeight();
  if (!vp_w) {
    vp_w = maskOn_.get()->getImage()->s();
    vp_h = maskOn_.get()->getImage()->t();
  }

  // PASS1, OUTPUT TEXTURE * COLOR MASK
  geom = new MyGeometry( NULL, false, g_iRBin);
  quadParam.geomToUse = geom;
  quads_[0] = new Quad(quadParam);
  ss = geom->getOrCreateStateSet();
  ss->setRenderBinDetails(g_iRBin++, "RenderBin");

  geom->setTexCoordArray(1, quads_[0]->getUVArray() );

  bf = new osg::BlendFunc();
  bf->setFunction(GL_ONE, GL_ZERO);
  ss->setAttributeAndModes(bf);

  SET_VIEWPORT_FROM_TEXTURE_SIZE( maskOn_.get() );
  ss->setAttributeAndModes(new osg::Viewport(0, 0, vp_w, vp_h));

  ss->setTextureAttributeAndModes(0, colorMask_.get() );
  ss->setTextureAttributeAndModes(1, outputTexture_.get() );

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_REPLACE);
  combiner->setCombine_Alpha(GL_REPLACE);

  combiner->setSource0_RGB(GL_TEXTURE);
  combiner->setSource0_Alpha(GL_TEXTURE);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(0, combiner);

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_MODULATE);

  combiner->setSource0_RGB(GL_PREVIOUS_ARB);
  combiner->setSource0_Alpha(GL_PREVIOUS_ARB);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(1, combiner);

  grp->addChild(quads_[0]->getGeode());

  // PASS2, IN ADDITIVE MODE: MASKON * (1 - COLORMASK)
  geom = new MyGeometry( maskOn_.get(), true, g_iRBin);
  quadParam.geomToUse = geom;
  quads_[1] = new Quad(quadParam);
  ss = geom->getOrCreateStateSet();
  ss->setRenderBinDetails(g_iRBin++, "RenderBin");

  geom->setTexCoordArray(1, quads_[1]->getUVArray() );

  bf = new osg::BlendFunc();
  bf->setFunction(GL_ONE, GL_ONE);
  ss->setAttributeAndModes(bf);
  ss->setAttributeAndModes(new osg::Viewport(0, 0, vp_w, vp_h));

  ss->setTextureAttributeAndModes(0, colorMask_.get() );
  ss->setTextureAttributeAndModes(1, maskOn_.get() );

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_REPLACE);
  combiner->setCombine_Alpha(GL_REPLACE);

  combiner->setSource0_RGB(GL_TEXTURE);
  combiner->setSource0_Alpha(GL_TEXTURE);
  combiner->setOperand0_RGB(GL_ONE_MINUS_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(0, combiner);

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_MODULATE);

  combiner->setSource0_RGB(GL_PREVIOUS_ARB);
  combiner->setSource0_Alpha(GL_PREVIOUS_ARB);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(1, combiner);

  grp->addChild(quads_[1]->getGeode());
}

void TextureLayersFlatten::flushTextureCacheForAllBRT()
{
  int size = technics_.size();
  for (int i = 0; i < size; i++) {
    BaseRenderTechnic *technic = technics_[i];
    technic->flushTextureCache();
  }

  if (quads_[0].valid())
    ((MyGeometry*)quads_[0]->getGeometry())->bRendered_ = false;
  if (quads_[1].valid())
    ((MyGeometry*)quads_[1]->getGeometry())->bRendered_ = false;
}


// BASE RENDER TECHNIC

TextureLayersFlatten::BaseRenderTechnic::~BaseRenderTechnic()
{
  if (parentGroup_.valid())
    parentGroup_->removeChild(0, 1);
}

void TextureLayersFlatten::BaseRenderTechnic::setOpacity(float _opacity)
{
  opacity_ = _opacity;
  updateCombiners();
}

void TextureLayersFlatten::BaseRenderTechnic::setColorFactor(const osg::Vec3f &_colorFactor)
{
  colorFactor_ = _colorFactor;
  updateCombiners();
}

#ifdef SUPPORT_GIF_HLS
void TextureLayersFlatten::BaseRenderTechnic::updateGIFTextureForHSL(osg::State &_state)
{
  int i;

  if (createdFromlayer_->orgGIFforHLS_.valid()) {

    GIFImage *orgImg = createdFromlayer_->orgGIFforHLS_.get();
    int w = orgImg->s();
    int h = orgImg->t();

    // work on the palette

    GIFImage::Color *orgCols = orgImg->getPalette();
    int nbColors = orgImg->getNbColors();
    GIFImage::Color changedPalette[256];
    for (i = 0; i < nbColors; i++) {
      GIFImage::Color &orgCol = orgCols[i];

      float flt_r = orgCol.r / 255.0f;  //180 / 255.0f;
      float flt_g = orgCol.g / 255.0f; //50 / 255.0f;
      float flt_b = orgCol.b / 255.0f; //120 / 255.0f;

      float fh, fs, fl;
      RGBtoHSL(flt_r, flt_g, flt_b, fh, fs, fl);
      //hsl_ = osg::Vec3f(-13/360.f, 0.75f, 0.0f);

      fh += hsl_._v[0];
      fs += hsl_._v[1];
      if (hsl_._v[2] >= 0)
        fl = fl * (1 - hsl_._v[2]) + hsl_._v[2];
      else
        fl = fl * (1 + hsl_._v[2]);

      if (fh < 0) fh += 1;
      else if (fh > 1) fh -= 1;

      if (fs < 0) fs = 0;
      else if (fs > 1) fs = 1;

      float fr, fg, fb;
      HSLtoRGB(fh, fs, fl, fr, fg, fb);

      if (fr < 0) fr = 0;
      else if (fr > 1) fr = 1;
      if (fg < 0) fg = 0;
      else if (fg > 1) fg = 1;
      if (fb < 0) fb = 0;
      else if (fb > 1) fb = 1;

      changedPalette[i].r = int(fr * 255.0f) & 0xFF;
      changedPalette[i].g = int(fg * 255.0f) & 0xFF;
      changedPalette[i].b = int(fb * 255.0f) & 0xFF;
    }

    unsigned char *orgData = orgImg->data();
    osg::Texture *texture = createdFromlayer_->texture_.get();
    int trans = orgImg->getTransparentColor();

//    char str[200];
  //  sprintf(str, "Trans is %ld\n", trans);
//    OutputDebugString(str);

    unsigned char *rgb_image = new unsigned char[w*h*4];
    osg::ref_ptr<osg::Image> realRGBImage = new osg::Image();
    realRGBImage->setImage(w, h, 1, 4, GL_RGBA, GL_UNSIGNED_BYTE, rgb_image, osg::Image::USE_NEW_DELETE);

    for (i = 0; i < w*h; i++) {
      unsigned char index = orgData[i];

      if (trans != -1 && index == trans) {
        *rgb_image++ = 0;
        *rgb_image++ = 0;
        *rgb_image++ = 0;
        *rgb_image++ = 0;
      }
      else {
        unsigned char r = changedPalette[index].r;
        unsigned char g = changedPalette[index].g;
        unsigned char b = changedPalette[index].b;

        *rgb_image++ = r;
        *rgb_image++ = g;
        *rgb_image++ = b;
        *rgb_image++ = 0xff;
      }
    }

    texture->dirtyTextureObject();
    texture->setImage(0, realRGBImage.get() );
    texture->apply(_state);
    texture->setImage(0, NULL);
    realRGBImage = NULL;
  }
}
#endif

void TextureLayersFlatten::BaseRenderTechnic::updateCombiners()
{
}

void TextureLayersFlatten::BaseRenderTechnic::flushTextureCache()
{
  int nbQuads = quads_.size();
  for (int i = 0; i < nbQuads; i++) {
    if (quads_[i].valid())
      ((MyGeometry*)quads_[i]->getGeometry())->bRendered_ = false;
  }
}


// DECAL RENDER TECHNIC

void TextureLayersFlatten::DecalRenderTechnic::updateCombiners()
{
  Parent::updateCombiners();
  combiner_->setConstantColor(osg::Vec4f(colorFactor_._v[0], colorFactor_._v[1], colorFactor_._v[2], 1));
}

void TextureLayersFlatten::DecalRenderTechnic::flushTextureCache()
{
  Parent::flushTextureCache();
}

void TextureLayersFlatten::DecalRenderTechnic::setup(osg::Group *_on)
{
  osg::TexEnvCombine *combiner;
  osg::BlendFunc *bf;
  osg::StateSet *ss;
  MyGeometry *geom;
  QuadParams quadParam;
  Quad *quad;
  int vp_w;
  int vp_h;

  quadParam.minPt = osg::Vec2f(-1, -1);
  quadParam.maxPt = osg::Vec2f(1, 1);
  quadParam.bInvertUV = false;

  osg::Group *grp = new osg::Group();
  grp->setName("TextureLayersFlatten_DecalGroup");
  _on->addChild(grp);
  parentGroup_ = grp;

  // PASS1, COPY TEXTURE A
  geom = new MyGeometry( textureOutput_.get(), true, g_iRBin);
  quadParam.geomToUse = geom;
  quad = new Quad(quadParam);
  quads_.push_back(quad);
  ss = geom->getOrCreateStateSet();
  ss->setRenderBinDetails(g_iRBin++, "RenderBin");

  bf = new osg::BlendFunc();
  bf->setFunction(GL_ONE, GL_ZERO);
  ss->setAttributeAndModes(bf);

  SET_VIEWPORT_FROM_TEXTURE_SIZE( textureOutput_.get() );
  ss->setAttributeAndModes(new osg::Viewport(0, 0, vp_w, vp_h));

  ss->setTextureAttributeAndModes(0, textureA_.get() );

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_REPLACE);

  combiner->setSource0_RGB(GL_TEXTURE);
  combiner->setSource0_Alpha(GL_TEXTURE);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_CONSTANT_ARB);
  combiner->setOperand1_RGB(GL_SRC_COLOR);

  combiner->setConstantColor(osg::Vec4f(1, 1, 1, 1) );
  combiner_ = combiner;

  ss->setTextureAttributeAndModes(0, combiner);

  grp->addChild(quad->getGeode());
}

// NORMAL RENDER TECHNIC - 1 PASS
// Description: The background is completely covered by the blend layer, so you should have some transparency if you use this mode.
// col = B

void TextureLayersFlatten::NormalRenderTechnic::updateCombiners()
{
  Parent::updateCombiners();

  combiners_[0]->setConstantColor(osg::Vec4f(colorFactor_._v[0]*opacity_, colorFactor_._v[1]*opacity_, colorFactor_._v[2]*opacity_, opacity_));
  combiners_[1]->setConstantColor(osg::Vec4f(1.0f, 1.0f, 1.0f, opacity_));
}

void TextureLayersFlatten::NormalRenderTechnic::flushTextureCache()
{
  Parent::flushTextureCache();
}

void TextureLayersFlatten::NormalRenderTechnic::setup(osg::Group *_on)
{
  osg::TexEnvCombine *combiner;
  osg::BlendFunc *bf;
  osg::StateSet *ss;
  MyGeometry *geom;
  QuadParams quadParam;
  Quad *quad;
  int vp_w, vp_h;

  quadParam.minPt = osg::Vec2f(-1, -1);
  quadParam.maxPt = osg::Vec2f(1, 1);
  quadParam.bInvertUV = false;

  osg::Group *grp = new osg::Group();
  grp->setName("TextureLayersFlatten_NormalGroup");
  _on->addChild(grp);
  parentGroup_ = grp;

  // PASS1, BLEND B
  geom = new MyGeometry( NULL, false, g_iRBin);
  quadParam.geomToUse = geom;
  quad = new Quad(quadParam);
  quads_.push_back(quad);
  ss = geom->getOrCreateStateSet();
  ss->setRenderBinDetails(g_iRBin++, "RenderBin");

  geom->setTexCoordArray(1, quad->getUVArray() );

  bf = new osg::BlendFunc();
  bf->setFunction(GL_ONE, GL_ZERO);
  ss->setAttributeAndModes(bf);

  SET_VIEWPORT_FROM_TEXTURE_SIZE( textureOutput_.get() );
  ss->setAttributeAndModes(new osg::Viewport(0, 0, vp_w, vp_h));

  ss->setTextureAttributeAndModes(0, textureB_.get() );
  ss->setTextureAttributeAndModes(1, textureB_.get() );

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_MODULATE);

  combiner->setSource0_RGB(GL_CONSTANT_ARB);
  combiner->setSource0_Alpha(GL_CONSTANT_ARB);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_ALPHA);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  combiner->setConstantColor(osg::Vec4f(1.0f, 1.0f, 1.0f, 1.0f));
  combiners_[0] = combiner;
  
  ss->setTextureAttributeAndModes(0, combiner);

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_REPLACE);

  combiner->setSource0_RGB(GL_PREVIOUS_ARB);
  combiner->setSource0_Alpha(GL_PREVIOUS_ARB);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(1, combiner);

  grp->addChild(quad->getGeode());

  // PASS2, BLEND A
  geom = new MyGeometry( textureOutput_.get(), true, g_iRBin);
  quadParam.geomToUse = geom;
  quad = new Quad(quadParam);
  quads_.push_back(quad);
  ss = geom->getOrCreateStateSet();
  ss->setRenderBinDetails(g_iRBin++, "RenderBin");

  geom->setTexCoordArray(1, quad->getUVArray() );

  bf = new osg::BlendFunc();
  bf->setFunction(GL_ONE, GL_ONE);
  ss->setAttributeAndModes(bf);
  ss->setAttributeAndModes(new osg::Viewport(0, 0, vp_w, vp_h));

  ss->setTextureAttributeAndModes(0, textureB_.get() );
  ss->setTextureAttributeAndModes(1, textureA_.get() );

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_MODULATE);

  combiner->setSource0_RGB(GL_CONSTANT_ARB);
  combiner->setSource0_Alpha(GL_CONSTANT_ARB);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  combiner->setConstantColor(osg::Vec4f(1.0f, 1.0f, 1.0f, 1.0f));
  combiners_[1] = combiner;

  ss->setTextureAttributeAndModes(0, combiner);

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_REPLACE);

  combiner->setSource0_RGB(GL_PREVIOUS_ARB);
  combiner->setSource0_Alpha(GL_PREVIOUS_ARB);
  combiner->setOperand0_RGB(GL_ONE_MINUS_SRC_ALPHA);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(1, combiner);

  grp->addChild(quad->getGeode());
}


// OVERLAY RENDER TECHNIC - 10 PASS :(
// Description: A combination of screen and multiply mode, depending on the base color.
// if (A < 0.5) col = 2 * A * B
// else col = 1 - 2 * (1 - A) * (1 - B)

void TextureLayersFlatten::OverlayRenderTechnic::updateCombiners()
{
  Parent::updateCombiners();
  combiners_[0]->setConstantColor(osg::Vec4f(opacity_, opacity_, opacity_, opacity_));
  combiners_[1]->setConstantColor(osg::Vec4f(1.0f, 1.0f, 1.0f, opacity_));
}

void TextureLayersFlatten::OverlayRenderTechnic::flushTextureCache()
{
  Parent::flushTextureCache();
}

void TextureLayersFlatten::OverlayRenderTechnic::setup(osg::Group *_on)
{
  osg::TexEnvCombine *combiner;
  osg::BlendFunc *bf;
  osg::StateSet *ss;
  MyGeometry *geom;
  QuadParams quadParam;
  Quad *quad;
  int vp_w;
  int vp_h;

  quadParam.minPt = osg::Vec2f(-1, -1);
  quadParam.maxPt = osg::Vec2f(1, 1);
  quadParam.bInvertUV = false;

  osg::Group *grp = new osg::Group();
  grp->setName("TextureLayersFlatten_OverlayGroup");
  _on->addChild(grp);
  parentGroup_ = grp;

  // PASS1 : 2 * A * B
  geom = new MyGeometry( g_tmpTexture[0].get(), false, g_iRBin);
  quadParam.geomToUse = geom;
  quad = new Quad(quadParam);
  quads_.push_back(quad);
  ss = geom->getOrCreateStateSet();
  ss->setRenderBinDetails(g_iRBin++, "RenderBin");

  geom->setTexCoordArray(1, quad->getUVArray());

  bf = new osg::BlendFunc();
  bf->setFunction(GL_ONE, GL_ZERO);
  ss->setAttributeAndModes(bf);

  SET_VIEWPORT_FROM_TEXTURE_SIZE(g_tmpTexture[0].get() );
  ss->setAttributeAndModes(new osg::Viewport(0, 0, vp_w, vp_h));

  ss->setTextureAttributeAndModes(0, textureA_.get() );
  ss->setTextureAttributeAndModes(1, textureB_.get() );

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_REPLACE);
  combiner->setCombine_Alpha(GL_REPLACE);

  combiner->setSource0_RGB(GL_TEXTURE);
  combiner->setSource0_Alpha(GL_TEXTURE);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(0, combiner);

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_MODULATE);

  combiner->setSource0_RGB(GL_PREVIOUS_ARB);
  combiner->setSource0_Alpha(GL_PREVIOUS_ARB);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  combiner->setScale_RGB(2.0f);
  ss->setTextureAttributeAndModes(1, combiner);

  grp->addChild(quad->getGeode());


  // PASS2 : 1 - 2 * (1 - A) * (1 - B)
  geom = new MyGeometry( g_tmpTexture[1].get(), false, g_iRBin);
  quadParam.geomToUse = geom;
  quad = new Quad(quadParam);
  quads_.push_back(quad);
  ss = geom->getOrCreateStateSet();
  ss->setRenderBinDetails(g_iRBin++, "RenderBin");

  geom->setTexCoordArray(1, quad->getUVArray());

  bf = new osg::BlendFunc();
  bf->setFunction(GL_ONE, GL_ZERO);
  ss->setAttributeAndModes(bf);

  SET_VIEWPORT_FROM_TEXTURE_SIZE(g_tmpTexture[1].get() );
  ss->setAttributeAndModes(new osg::Viewport(0, 0, vp_w, vp_h));

  ss->setTextureAttributeAndModes(0, textureA_.get() );
  ss->setTextureAttributeAndModes(1, textureB_.get() );

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_REPLACE);
  combiner->setCombine_Alpha(GL_REPLACE);

  combiner->setSource0_RGB(GL_TEXTURE);
  combiner->setSource0_Alpha(GL_TEXTURE);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(0, combiner);

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_MODULATE);

  combiner->setSource0_RGB(GL_PREVIOUS_ARB);
  combiner->setSource0_Alpha(GL_PREVIOUS_ARB);
  combiner->setOperand0_RGB(GL_ONE_MINUS_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_ONE_MINUS_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  combiner->setScale_RGB(2.0f);
  ss->setTextureAttributeAndModes(1, combiner);

  grp->addChild(quad->getGeode());


  // PASS 3,4,5,6 : IF (A < 0.5) A = 0 ELSE A = 1
  for (int i = 2; i < 6; i++) {
    geom = new MyGeometry( g_tmpTexture[2].get(), false, g_iRBin);
    quadParam.geomToUse = geom;

    quad = new Quad(quadParam);
    quads_.push_back(quad);
    ss = geom->getOrCreateStateSet();
    ss->setRenderBinDetails(g_iRBin++, "RenderBin");

    bf = new osg::BlendFunc();
    bf->setFunction(GL_ONE, GL_ZERO);
    ss->setAttributeAndModes(bf);

    SET_VIEWPORT_FROM_TEXTURE_SIZE(g_tmpTexture[2].get() );
    ss->setAttributeAndModes(new osg::Viewport(0, 0, vp_w, vp_h));

    if (i == 2)
      ss->setTextureAttributeAndModes(0, textureA_.get() );
    else
      ss->setTextureAttributeAndModes(0, g_tmpTexture[2].get() );

    combiner = new osg::TexEnvCombine();

    if (i == 2) {
      combiner->setCombine_RGB(GL_SUBTRACT_ARB);
      combiner->setCombine_Alpha(GL_REPLACE);

      combiner->setSource0_RGB(GL_TEXTURE);
      combiner->setSource0_Alpha(GL_TEXTURE);
      combiner->setOperand0_RGB(GL_SRC_COLOR);
      combiner->setOperand0_Alpha(GL_SRC_ALPHA);

      combiner->setSource1_RGB(GL_CONSTANT_ARB);
      combiner->setSource1_Alpha(GL_CONSTANT_ARB);
      combiner->setOperand1_RGB(GL_SRC_COLOR);
      combiner->setOperand1_Alpha(GL_SRC_ALPHA);

      combiner->setConstantColor( osg::Vec4f(0.5f, 0.5f, 0.5f, 0) );
    }
    else {
      combiner->setCombine_RGB(GL_REPLACE);
      combiner->setCombine_Alpha(GL_REPLACE);

      combiner->setSource0_RGB(GL_TEXTURE);
      combiner->setSource0_Alpha(GL_TEXTURE);
      combiner->setOperand0_RGB(GL_SRC_COLOR);
      combiner->setOperand0_Alpha(GL_SRC_ALPHA);
    }

    combiner->setScale_RGB(4.0f);

    ss->setTextureAttributeAndModes(0, combiner);

    grp->addChild(quad->getGeode());
  }


  // PASS7, LOW PART for A < 0.5
  geom = new MyGeometry( g_tmpTexture[0].get(), false, g_iRBin);
  quadParam.geomToUse = geom;
  quad = new Quad(quadParam);
  quads_.push_back(quad);
  ss = geom->getOrCreateStateSet();
  ss->setRenderBinDetails(g_iRBin++, "RenderBin");

  geom->setTexCoordArray(1, quad->getUVArray());

  bf = new osg::BlendFunc();
  bf->setFunction(GL_ONE, GL_ZERO);
  ss->setAttributeAndModes(bf);

  SET_VIEWPORT_FROM_TEXTURE_SIZE(g_tmpTexture[0].get() );
  ss->setAttributeAndModes(new osg::Viewport(0, 0, vp_w, vp_h));

  ss->setTextureAttributeAndModes(0, g_tmpTexture[0].get() );
  ss->setTextureAttributeAndModes(1, g_tmpTexture[2].get() );

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_REPLACE);
  combiner->setCombine_Alpha(GL_REPLACE);

  combiner->setSource0_RGB(GL_TEXTURE);
  combiner->setSource0_Alpha(GL_TEXTURE);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(0, combiner);

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_MODULATE);

  combiner->setSource0_RGB(GL_PREVIOUS_ARB);
  combiner->setSource0_Alpha(GL_PREVIOUS_ARB);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_ONE_MINUS_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(1, combiner);

  grp->addChild(quad->getGeode());


  // PASS8, HIGH PART for A > 0.5
  geom = new MyGeometry( g_tmpTexture[1].get(), false, g_iRBin);
  quadParam.geomToUse = geom;
  quad = new Quad(quadParam);
  quads_.push_back(quad);
  ss = geom->getOrCreateStateSet();
  ss->setRenderBinDetails(g_iRBin++, "RenderBin");

  geom->setTexCoordArray(1, quad->getUVArray());

  bf = new osg::BlendFunc();
  bf->setFunction(GL_ONE, GL_ZERO);
  ss->setAttributeAndModes(bf);

  SET_VIEWPORT_FROM_TEXTURE_SIZE(g_tmpTexture[1].get() );
  ss->setAttributeAndModes(new osg::Viewport(0, 0, vp_w, vp_h));

  ss->setTextureAttributeAndModes(0, g_tmpTexture[1].get() );
  ss->setTextureAttributeAndModes(1, g_tmpTexture[2].get() );

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_REPLACE);
  combiner->setCombine_Alpha(GL_REPLACE);

  combiner->setSource0_RGB(GL_TEXTURE);
  combiner->setSource0_Alpha(GL_TEXTURE);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(0, combiner);

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_MODULATE);

  combiner->setSource0_RGB(GL_PREVIOUS_ARB);
  combiner->setSource0_Alpha(GL_PREVIOUS_ARB);
  combiner->setOperand0_RGB(GL_ONE_MINUS_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(1, combiner);

  grp->addChild(quad->getGeode());


  // PASS9, LOW PART + HIGH PART
  geom = new MyGeometry( g_tmpTexture[2].get(), false, g_iRBin);
  quadParam.geomToUse = geom;
  quad = new Quad(quadParam);
  quads_.push_back(quad);
  ss = geom->getOrCreateStateSet();
  ss->setRenderBinDetails(g_iRBin++, "RenderBin");

  geom->setTexCoordArray(1, quad->getUVArray());

  bf = new osg::BlendFunc();
  bf->setFunction(GL_ONE, GL_ZERO);
  ss->setAttributeAndModes(bf);

  SET_VIEWPORT_FROM_TEXTURE_SIZE(g_tmpTexture[2].get() );
  ss->setAttributeAndModes(new osg::Viewport(0, 0, vp_w, vp_h));

  ss->setTextureAttributeAndModes(0, g_tmpTexture[0].get() );
  ss->setTextureAttributeAndModes(1, g_tmpTexture[1].get() );

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_REPLACE);
  combiner->setCombine_Alpha(GL_REPLACE);

  combiner->setSource0_RGB(GL_TEXTURE);
  combiner->setSource0_Alpha(GL_TEXTURE);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(0, combiner);

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_ADD);
  combiner->setCombine_Alpha(GL_ADD);

  combiner->setSource0_RGB(GL_PREVIOUS_ARB);
  combiner->setSource0_Alpha(GL_PREVIOUS_ARB);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(1, combiner);

  grp->addChild(quad->getGeode());


  // PASS10, BLEND B
  geom = new MyGeometry( NULL, false, g_iRBin);
  quadParam.geomToUse = geom;
  quad = new Quad(quadParam);
  quads_.push_back(quad);
  ss = geom->getOrCreateStateSet();
  ss->setRenderBinDetails(g_iRBin++, "RenderBin");

  geom->setTexCoordArray(1, quad->getUVArray() );

  bf = new osg::BlendFunc();
  bf->setFunction(GL_ONE, GL_ZERO);
  ss->setAttributeAndModes(bf);

  SET_VIEWPORT_FROM_TEXTURE_SIZE( textureOutput_.get() );
  ss->setAttributeAndModes(new osg::Viewport(0, 0, vp_w, vp_h));

  ss->setTextureAttributeAndModes(0, textureB_.get() );
  ss->setTextureAttributeAndModes(1, g_tmpTexture[2].get() );

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_MODULATE);

  combiner->setSource0_RGB(GL_CONSTANT_ARB);
  combiner->setSource0_Alpha(GL_CONSTANT_ARB);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_ALPHA);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  combiner->setConstantColor(osg::Vec4f(1.0f, 1.0f, 1.0f, 1.0f));
  combiners_[0] = combiner;

  ss->setTextureAttributeAndModes(0, combiner);

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_REPLACE);

  combiner->setSource0_RGB(GL_PREVIOUS_ARB);
  combiner->setSource0_Alpha(GL_PREVIOUS_ARB);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(1, combiner);

  grp->addChild(quad->getGeode());

  // PASS11, BLEND A
  geom = new MyGeometry( textureOutput_.get(), true, g_iRBin);
  quadParam.geomToUse = geom;
  quad = new Quad(quadParam);
  quads_.push_back(quad);
  ss = geom->getOrCreateStateSet();
  ss->setRenderBinDetails(g_iRBin++, "RenderBin");

  geom->setTexCoordArray(1, quad->getUVArray() );

  bf = new osg::BlendFunc();
  bf->setFunction(GL_ONE, GL_ONE);
  ss->setAttributeAndModes(bf);
  ss->setAttributeAndModes(new osg::Viewport(0, 0, vp_w, vp_h));

  ss->setTextureAttributeAndModes(0, textureB_.get() );
  ss->setTextureAttributeAndModes(1, textureA_.get() );

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_MODULATE);

  combiner->setSource0_RGB(GL_CONSTANT_ARB);
  combiner->setSource0_Alpha(GL_CONSTANT_ARB);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  combiner->setConstantColor(osg::Vec4f(1.0f, 1.0f, 1.0f, 1.0f));
  combiners_[1] = combiner;

  ss->setTextureAttributeAndModes(0, combiner);

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_REPLACE);

  combiner->setSource0_RGB(GL_PREVIOUS_ARB);
  combiner->setSource0_Alpha(GL_PREVIOUS_ARB);
  combiner->setOperand0_RGB(GL_ONE_MINUS_SRC_ALPHA);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(1, combiner);

  grp->addChild(quad->getGeode());
}


// LINEAR DODGE RENDER TECHNIC - 2 PASS
// Description: A very basic blend mode.
// A + B

void TextureLayersFlatten::LinearDodgeRenderTechnic::updateCombiners()
{
  Parent::updateCombiners();
  combiners_[0]->setConstantColor(osg::Vec4f(colorFactor_._v[0]*opacity_, colorFactor_._v[1]*opacity_, colorFactor_._v[2]*opacity_, opacity_));
  combiners_[1]->setConstantColor(osg::Vec4f(1.0f, 1.0f, 1.0f, opacity_));
}

void TextureLayersFlatten::LinearDodgeRenderTechnic::flushTextureCache()
{
  Parent::flushTextureCache();
}

void TextureLayersFlatten::LinearDodgeRenderTechnic::setup(osg::Group *_on)
{
  osg::TexEnvCombine *combiner;
  osg::BlendFunc *bf;
  osg::StateSet *ss;
  MyGeometry *geom;
  QuadParams quadParam;
  Quad *quad;
  int vp_w, vp_h;

  quadParam.minPt = osg::Vec2f(-1, -1);
  quadParam.maxPt = osg::Vec2f(1, 1);
  quadParam.bInvertUV = false;

  osg::Group *grp = new osg::Group();
  grp->setName("TextureLayersFlatten_LinearDodgeGroup");
  _on->addChild(grp);
  parentGroup_ = grp;

  // PASS1, A + B
  geom = new MyGeometry( g_tmpTexture[0].get(), false, g_iRBin);
  quadParam.geomToUse = geom;
  quad = new Quad(quadParam);
  quads_.push_back(quad);
  ss = geom->getOrCreateStateSet();
  ss->setRenderBinDetails(g_iRBin++, "RenderBin");

  geom->setTexCoordArray(1, quad->getUVArray());

  bf = new osg::BlendFunc();
  bf->setFunction(GL_ONE, GL_ZERO);
  ss->setAttributeAndModes(bf);

  SET_VIEWPORT_FROM_TEXTURE_SIZE( g_tmpTexture[0].get() );
  ss->setAttributeAndModes(new osg::Viewport(0, 0, vp_w, vp_h));

  ss->setTextureAttributeAndModes(0, textureA_.get() );
  ss->setTextureAttributeAndModes(1, textureB_.get() );

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_REPLACE);
  combiner->setCombine_Alpha(GL_REPLACE);

  combiner->setSource0_RGB(GL_TEXTURE);
  combiner->setSource0_Alpha(GL_TEXTURE);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(0, combiner);

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_ADD);
  combiner->setCombine_Alpha(GL_ADD);

  combiner->setSource0_RGB(GL_PREVIOUS_ARB);
  combiner->setSource0_Alpha(GL_PREVIOUS_ARB);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(1, combiner);

  grp->addChild(quad->getGeode());


  // PASS2, BLEND B
  geom = new MyGeometry( NULL, false, g_iRBin);
  quadParam.geomToUse = geom;
  quad = new Quad(quadParam);
  quads_.push_back(quad);
  ss = geom->getOrCreateStateSet();
  ss->setRenderBinDetails(g_iRBin++, "RenderBin");

  geom->setTexCoordArray(1, quad->getUVArray() );

  bf = new osg::BlendFunc();
  bf->setFunction(GL_ONE, GL_ZERO);
  ss->setAttributeAndModes(bf);

  SET_VIEWPORT_FROM_TEXTURE_SIZE( textureOutput_.get() );
  ss->setAttributeAndModes(new osg::Viewport(0, 0, vp_w, vp_h));

  ss->setTextureAttributeAndModes(0, textureB_.get() );
  ss->setTextureAttributeAndModes(1, g_tmpTexture[0].get() );

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_MODULATE);

  combiner->setSource0_RGB(GL_CONSTANT_ARB);
  combiner->setSource0_Alpha(GL_CONSTANT_ARB);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_ALPHA);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  combiner->setConstantColor(osg::Vec4f(1.0f, 1.0f, 1.0f, 1.0f));
  combiners_[0] = combiner;

  ss->setTextureAttributeAndModes(0, combiner);

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_REPLACE);

  combiner->setSource0_RGB(GL_PREVIOUS_ARB);
  combiner->setSource0_Alpha(GL_PREVIOUS_ARB);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(1, combiner);

  grp->addChild(quads_[1]->getGeode());


  // PASS3, BLEND A
  geom = new MyGeometry( textureOutput_.get(), true, g_iRBin);
  quadParam.geomToUse = geom;
  quad = new Quad(quadParam);
  quads_.push_back(quad);
  ss = geom->getOrCreateStateSet();
  ss->setRenderBinDetails(g_iRBin++, "RenderBin");

  geom->setTexCoordArray(1, quad->getUVArray() );

  bf = new osg::BlendFunc();
  bf->setFunction(GL_ONE, GL_ONE);
  ss->setAttributeAndModes(bf);

  ss->setAttributeAndModes(new osg::Viewport(0, 0, vp_w, vp_h));

  ss->setTextureAttributeAndModes(0, textureB_.get() );
  ss->setTextureAttributeAndModes(1, textureA_.get() );

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_MODULATE);

  combiner->setSource0_RGB(GL_CONSTANT_ARB);
  combiner->setSource0_Alpha(GL_CONSTANT_ARB);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  combiner->setConstantColor(osg::Vec4f(1.0f, 1.0f, 1.0f, 1.0f));
  combiners_[1] = combiner;

  ss->setTextureAttributeAndModes(0, combiner);

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_REPLACE);

  combiner->setSource0_RGB(GL_PREVIOUS_ARB);
  combiner->setSource0_Alpha(GL_PREVIOUS_ARB);
  combiner->setOperand0_RGB(GL_ONE_MINUS_SRC_ALPHA);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(1, combiner);

  grp->addChild(quad->getGeode());
}



// MULTIPLY RENDER TECHNIC - 2 PASS
// Description: Both parameters are simply multiplied by each other.
//              This returns a darker result than both input parameters in most cases (except one of them equals 1).
//              Completely white layers do not change the background at all (and vice versa) - completely black layers give a black result.
// A * B

void TextureLayersFlatten::MultiplyRenderTechnic::updateCombiners()
{
  Parent::updateCombiners();
  combiners_[0]->setConstantColor(osg::Vec4f(colorFactor_._v[0]*opacity_, colorFactor_._v[1]*opacity_, colorFactor_._v[2]*opacity_, opacity_));
  combiners_[1]->setConstantColor(osg::Vec4f(1.0f, 1.0f, 1.0f, opacity_));
}

void TextureLayersFlatten::MultiplyRenderTechnic::flushTextureCache()
{
  Parent::flushTextureCache();
}

void TextureLayersFlatten::MultiplyRenderTechnic::setup(osg::Group *_on)
{
  osg::TexEnvCombine *combiner;
  osg::BlendFunc *bf;
  osg::StateSet *ss;
  MyGeometry *geom;
  QuadParams quadParam;
  Quad *quad;
  int vp_w, vp_h;

  quadParam.minPt = osg::Vec2f(-1, -1);
  quadParam.maxPt = osg::Vec2f(1, 1);
  quadParam.bInvertUV = false;

  osg::Group *grp = new osg::Group();
  grp->setName("TextureLayersFlatten_MultiplyGroup");
  _on->addChild(grp);
  parentGroup_ = grp;

  // PASS1, A * B
  geom = new MyGeometry( g_tmpTexture[0].get(), false, g_iRBin);
  quadParam.geomToUse = geom;
  quad = new Quad(quadParam);
  quads_.push_back(quad);
  ss = geom->getOrCreateStateSet();
  ss->setRenderBinDetails(g_iRBin++, "RenderBin");

  geom->setTexCoordArray(1, quad->getUVArray());

  bf = new osg::BlendFunc();
  bf->setFunction(GL_ONE, GL_ZERO);
  ss->setAttributeAndModes(bf);

  SET_VIEWPORT_FROM_TEXTURE_SIZE( g_tmpTexture[0].get() );
  ss->setAttributeAndModes(new osg::Viewport(0, 0, vp_w, vp_h));

  ss->setTextureAttributeAndModes(0, textureA_.get() );
  ss->setTextureAttributeAndModes(1, textureB_.get() );

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_REPLACE);
  combiner->setCombine_Alpha(GL_REPLACE);

  combiner->setSource0_RGB(GL_TEXTURE);
  combiner->setSource0_Alpha(GL_TEXTURE);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(0, combiner);

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_MODULATE);

  combiner->setSource0_RGB(GL_PREVIOUS_ARB);
  combiner->setSource0_Alpha(GL_PREVIOUS_ARB);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(1, combiner);

  grp->addChild(quad->getGeode());


  // PASS2, BLEND B
  geom = new MyGeometry( NULL, false, g_iRBin);
  quadParam.geomToUse = geom;
  quad = new Quad(quadParam);
  quads_.push_back(quad);
  ss = geom->getOrCreateStateSet();
  ss->setRenderBinDetails(g_iRBin++, "RenderBin");

  geom->setTexCoordArray(1, quad->getUVArray() );

  bf = new osg::BlendFunc();
  bf->setFunction(GL_ONE, GL_ZERO);
  ss->setAttributeAndModes(bf);

  SET_VIEWPORT_FROM_TEXTURE_SIZE( textureOutput_.get() );
  ss->setAttributeAndModes(new osg::Viewport(0, 0, vp_w, vp_h));

  ss->setTextureAttributeAndModes(0, textureB_.get() );
  ss->setTextureAttributeAndModes(1, g_tmpTexture[0].get() );

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_MODULATE);

  combiner->setSource0_RGB(GL_CONSTANT_ARB);
  combiner->setSource0_Alpha(GL_CONSTANT_ARB);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_ALPHA);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  combiner->setConstantColor(osg::Vec4f(1.0f, 1.0f, 1.0f, 1.0f));
  combiners_[0] = combiner;

  ss->setTextureAttributeAndModes(0, combiner);

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_REPLACE);

  combiner->setSource0_RGB(GL_PREVIOUS_ARB);
  combiner->setSource0_Alpha(GL_PREVIOUS_ARB);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(1, combiner);

  grp->addChild(quad->getGeode());

  // PASS3, BLEND A
  geom = new MyGeometry( textureOutput_.get(), true, g_iRBin);
  quadParam.geomToUse = geom;
  quad = new Quad(quadParam);
  quads_.push_back(quad);
  ss = geom->getOrCreateStateSet();
  ss->setRenderBinDetails(g_iRBin++, "RenderBin");

  geom->setTexCoordArray(1, quad->getUVArray() );

  bf = new osg::BlendFunc();
  bf->setFunction(GL_ONE, GL_ONE);
  ss->setAttributeAndModes(bf);

  ss->setAttributeAndModes(new osg::Viewport(0, 0, vp_w, vp_h));

  ss->setTextureAttributeAndModes(0, textureB_.get() );
  ss->setTextureAttributeAndModes(1, textureA_.get() );

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_MODULATE);

  combiner->setSource0_RGB(GL_CONSTANT_ARB);
  combiner->setSource0_Alpha(GL_CONSTANT_ARB);
  combiner->setOperand0_RGB(GL_SRC_COLOR);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  combiner->setConstantColor(osg::Vec4f(1.0f, 1.0f, 1.0f, 1.0f));
  combiners_[1] = combiner;

  ss->setTextureAttributeAndModes(0, combiner);

  combiner = new osg::TexEnvCombine();
  combiner->setCombine_RGB(GL_MODULATE);
  combiner->setCombine_Alpha(GL_REPLACE);

  combiner->setSource0_RGB(GL_PREVIOUS_ARB);
  combiner->setSource0_Alpha(GL_PREVIOUS_ARB);
  combiner->setOperand0_RGB(GL_ONE_MINUS_SRC_ALPHA);
  combiner->setOperand0_Alpha(GL_SRC_ALPHA);

  combiner->setSource1_RGB(GL_TEXTURE);
  combiner->setSource1_Alpha(GL_TEXTURE);
  combiner->setOperand1_RGB(GL_SRC_COLOR);
  combiner->setOperand1_Alpha(GL_SRC_ALPHA);

  ss->setTextureAttributeAndModes(1, combiner);

  grp->addChild(quad->getGeode());
}


// TextureLayersFlatten::MyGeometry

void TextureLayersFlatten::MyGeometry::drawImplementation(osg::State &_state) const
{
  int i;

  if (bRendered_)
    return;

  osg::notify(osg::INFO) << "TextureLayersFlatten: calculate layered texture" << std::endl;

  int nbTexs = g_filterTexture.size();
  for (i = 0; i < nbTexs; i++) {
    osg::Texture2D *tex = g_filterTexture[i].get();
    if (!tex)
      continue;
    tex->setFilter(osg::Texture::MIN_FILTER, osg::Texture::NEAREST);
    tex->setFilter(osg::Texture::MAG_FILTER, osg::Texture::NEAREST);
  }

  Geometry::drawImplementation(_state);

  if (target_.valid()) {
    osg::Texture2D *tgt = (osg::Texture2D*) target_.get();

    _state.setActiveTextureUnit(0);
    _state.haveAppliedTextureAttribute(0, tgt);

    int w = tgt->getTextureWidth();
    int h = tgt->getTextureHeight();
    if (!w) {
      w = tgt->getImage()->s();
      h = tgt->getImage()->t();
    }

    tgt->copyTexImage2D(_state, 0, 0, w, h);
//    _state.haveAppliedTextureAttribute(0, target_);
  }

  for (i = 0; i < nbTexs; i++) {
    osg::Texture2D *tex = g_filterTexture[i].get();
    if (!tex)
      continue;
    tex->setFilter(osg::Texture::MIN_FILTER, osg::Texture::LINEAR);
    tex->setFilter(osg::Texture::MAG_FILTER, osg::Texture::LINEAR);
  }

  if (bClearFrameBufferAfterRecopy_) {
//    if (!target_.valid())
  //    OutputDebugString("clear\n");
    glClear(GL_COLOR_BUFFER_BIT);
  }

  bRendered_ = true;
}

// TextureLayersFlatten::TextureFilterGeometry

/*
void TextureLayersFlatten::TextureFilterGeometry::drawImplementation(osg::State &_state) const
{
  int i;
  int nbTexs = g_filterTexture.size();

//  OutputDebugString("toto\n");

  if (bSetFilter_) {
    for (i = 0; i < nbTexs; i++) {
      osg::Texture2D *tex = g_filterTexture[i].get();
      if (!tex)
        continue;
      tex->setFilter(osg::Texture::MIN_FILTER, osg::Texture::LINEAR);
      tex->setFilter(osg::Texture::MAG_FILTER, osg::Texture::LINEAR);
    }
  }
  else {
    for (i = 0; i < nbTexs; i++) {
      osg::Texture2D *tex = g_filterTexture[i].get();
      if (!tex)
        continue;
      tex->setFilter(osg::Texture::MIN_FILTER, osg::Texture::NEAREST);
      tex->setFilter(osg::Texture::MAG_FILTER, osg::Texture::NEAREST);
    }
  }
}

bool TextureLayersFlatten::TextureFilterGeometry::computeBound() const
{
  return true;
}
*/

//
TextureLayersFlatten::Quad::Quad(QuadParams &_params)
{
  geode_ = new osg::Geode();
  if (!_params.geomToUse)
    geom_ = new osg::Geometry();
  else
    geom_ = _params.geomToUse;

  geode_->addDrawable(geom_.get());
  geode_->setCullingActive(false);

  vertices_ = new osg::Vec3Array();
  osg::Vec3Array *vert = vertices_.get();
  vert->resize(4);
  geom_->setVertexArray(vert);

  vert[0][0] = osg::Vec3f(_params.minPt._v[0], _params.minPt._v[1], 0.1f);
  vert[0][1] = osg::Vec3f(_params.maxPt._v[0], _params.minPt._v[1], 0.1f);
  vert[0][2] = osg::Vec3f(_params.maxPt._v[0], _params.maxPt._v[1], 0.1f);
  vert[0][3] = osg::Vec3f(_params.minPt._v[0], _params.maxPt._v[1], 0.1f);

  uv_ = new osg::Vec2Array();
  if (!_params.bInvertUV) {
    uv_->push_back( osg::Vec2f(_params.minUV._v[0], _params.minUV._v[1]) );
    uv_->push_back( osg::Vec2f(_params.maxUV._v[0], _params.minUV._v[1]) );
    uv_->push_back( osg::Vec2f(_params.maxUV._v[0], _params.maxUV._v[1]) );
    uv_->push_back( osg::Vec2f(_params.minUV._v[0], _params.maxUV._v[1]) );
  }
  else {
    uv_->push_back( osg::Vec2f(_params.minUV._v[0], _params.maxUV._v[1]) );
    uv_->push_back( osg::Vec2f(_params.maxUV._v[0], _params.maxUV._v[1]) );
    uv_->push_back( osg::Vec2f(_params.maxUV._v[0], _params.minUV._v[1]) );
    uv_->push_back( osg::Vec2f(_params.minUV._v[0], _params.minUV._v[1]) );
  }
  geom_->setTexCoordArray(0, uv_.get());

  GLushort index[6];
  index[0] = 0;
  index[1] = 1;
  index[2] = 2;
  index[3] = 0;
  index[4] = 2;
  index[5] = 3;

  geom_->addPrimitiveSet(new osg::DrawElementsUShort(osg::PrimitiveSet::TRIANGLES, 6, index));
  geom_->setUseDisplayList(false);
  geom_->setUseVertexBufferObjects(false);

  osg::StateSet *state = geom_->getOrCreateStateSet();
  material_ = new osg::Material;
  state->setAttributeAndModes(material_.get(), osg::StateAttribute::ON);
  state->setMode(GL_LIGHTING, osg::StateAttribute::OFF);
  state->setMode(GL_CULL_FACE, osg::StateAttribute::OFF);
  material_->setDiffuse(osg::Material::FRONT_AND_BACK, osg::Vec4f(1, 1, 1, 1) );

  texture_ = NULL;

  osg::BlendFunc *bf = new osg::BlendFunc;
  bf->setFunction(_params.srcBlendFactor, _params.dstBlendFactor);
  state->setAttributeAndModes(bf, osg::StateAttribute::ON);

  if (_params.bDisableWriteMask) {
    state->setMode(GL_DEPTH_TEST, false);
  }
}


TextureLayersFlatten::Quad::~Quad()
{
}

////////////////////////////////////////////////////////////////////////


float rgb_max(float _r, float _g, float _b)
{
  if (_r > _g)
    return (_r > _b) ? _r : _b;
  return (_g > _b) ? _g : _b;
}

float rgb_min(float _r, float _g, float _b)
{
  if (_r < _g)
    return (_r < _b) ? _r : _b;
  return (_g < _b) ? _g : _b;
}

void RGBtoHSL(float _r, float _g, float _b, float &_h, float &_s, float &_l)
{
  float max, min, delta;

  max = rgb_max(_r, _g, _b);
  min = rgb_min(_r, _g, _b);

  _l = (max + min) / 2.0f;

  if (max == min) {
    _s = 0.0f;
    _h = -1.0f;
  }
  else {
    if (_l <= 0.5f)
      _s = (max - min) / (max + min);
    else
      _s = (max - min) / (2.0f - max - min);

    delta = max - min;

    if (delta == 0.0f)
      delta = 1.0f;

    if (_r == max) {
      _h = (_g - _b) / delta;
    }
    else if (_g == max) {
      _h = 2.0f + (_b - _r) / delta;
    }
    else if (_b == max) {
      _h = 4.0f + (_r - _g) / delta;
    }

    _h /= 6.0f;

    if (_h < 0.0f)
      _h += 1.0f;
  }
}

static float hsl_value(float _n1, float _n2, float _hue)
{
  float val;

  if (_hue > 6.0f)
    _hue -= 6.0f;
  else if (_hue < 0.0f)
    _hue += 6.0f;

  if (_hue < 1.0f)
    val = _n1 + (_n2 - _n1) * _hue;
  else if (_hue < 3.0f)
    val = _n2;
  else if (_hue < 4.0f)
    val = _n1 + (_n2 - _n1) * (4.0f - _hue);
  else
    val = _n1;

  return val;
}

void HSLtoRGB(float _h, float _s, float _l, float &_r, float &_g, float &_b)
{
  if (_s == 0) {
    // achromatic case
    _r = _l;
    _g = _l;
    _b = _l;
  }
  else {
    float m1, m2;

    if (_l <= 0.5f)
      m2 = _l * (1.0f + _s);
    else
      m2 = _l + _s - _l * _s;

    m1 = 2.0f * _l - m2;

    _r = hsl_value(m1, m2, _h * 6.0f + 2.0f);
    _g = hsl_value(m1, m2, _h * 6.0f);
    _b = hsl_value(m1, m2, _h * 6.0f - 2.0f);
  }
}

void InvertPremultipliedAlpha(osg::Image &_img)
{
  if (_img.getPixelFormat() != GL_RGBA)
    return;

  int w = _img.s();
  int h = _img.t();
  int size = w * h;

  unsigned char *pixs = _img.data();
  for (int i = 0; i < size; i++, pixs += 4) {
    int a = pixs[3];
    if (!a)
      continue;
    float fa = 255.0f / a;
    pixs[0] = int(pixs[0] * fa);
    pixs[1] = int(pixs[1] * fa);
    pixs[2] = int(pixs[2] * fa);
  }
}

} // namespace osgCal
