/* -*- c++ -*-
Copyright (C) 2003 <ryu@gpul.org>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
/*
*
* Copyright (C) 2004 Mekensleep
*
*	Mekensleep
*	24 rue vieille du temple
*	75004 Paris
*       licensing@mekensleep.com
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*
* Authors:
*  Cedric PINSON <cpinson@freesheep.org>
*  Loic Dachary <loic@gnu.org>
*
*/

/**
 * This demo offers two ways to load a core model from file:
 * - Use a function loadCoreModelFromCFG()
 * - Use a plugin which reads .cfg
 *
 * Press key 'm' in order to switch between loaded animations of model.
 *
 * When a coreModel is created a model is created from this templated model.
 * This demo works with osg 0.9.8 release. Please update your ogs version.
 * 
 * Author: Jerome Schmid <jerome.schmid@free.fr>
 */
#include <osgProducer/Viewer>
#include <osgProducer/OsgCameraGroup>
#include <osgGA/TrackballManipulator>
#include <osgGA/NodeTrackerManipulator>
#include <osg/PositionAttitudeTransform>
#include <osg/Drawable>
#include <osg/Timer>
#include <osg/Group>
#include <osg/Geode>
#include <osg/Geometry>
#include <osg/StateSet>
#include <osg/StateAttribute>
#include <osg/PolygonMode>
#include <osg/CullFace>

#include <osgCal/CoreModel>
#include <osgCal/Model>
#include <osgDB/WriteFile>
#include <osgDB/ReaderWriter>
#include <osgDB/FileNameUtils>
#include <osgDB/FileUtils>
#include <osgDB/ReadFile>

#include <osg/Notify>
#include <osg/LineWidth>

#include <iostream>
#include <string>


class MyManipulator: public osgGA::NodeTrackerManipulator {
public:
	MyManipulator(osgCal::Model *model) {
		this->model=model;
		nMov=0;
	}
	/** handle events, return true if handled, false otherwise.*/
	virtual bool handle(const osgGA::GUIEventAdapter& ea,osgGA::GUIActionAdapter& us) {
		if (ea.getEventType() == osgGA::GUIEventAdapter::KEYDOWN) {
			
			// Key to switch between animations
			if (ea.getKey() == 'm') {
				model->getCalModel()->getMixer()->clearCycle(nMov, 3);
				nMov = ++nMov%6;
				printf("Movement No.%d (Blend time 3sec)\n", nMov);
				model->getCalModel()->getMixer()->blendCycle(nMov, 1, 3);
				return true;
			}
			
		}
		return NodeTrackerManipulator::handle(ea,us);
	}

private:
	osgCal::Model* model;
	int nMov;
};

#define LINE_BUFFER_SIZE 4096

bool loadCoreModelFromCFG(	const std::string& dir, 
													const std::string& path,
													osgCal::CoreModel *coreModel)
{
	// Read file and update model
	CalCoreModel* calCoreModel	= coreModel->getCalCoreModel();
	std::cout << "Loading cal3d character from " << path.c_str() << std::endl;

	float scale = 1.0f;
	bool bScale = false;

	FILE* f = fopen(path.c_str(), "r");
	if(!f)
	{
		std::cerr << "opening failed!" << std::endl;
		return false;
	}

	char buffer[LINE_BUFFER_SIZE];

	while (fgets(buffer, LINE_BUFFER_SIZE,f))
	{
		// Ignore comments or empty lines
		if (*buffer == '#' || *buffer == 0)
			continue;

		char* equal = strchr(buffer, '=');
		if (equal)
		{
			// Terminates first token
			*equal++ = 0;
			// Removes ending newline (CR & LF)
			{
				int last = strlen(equal) - 1;
				if (equal[last] == '\n') equal[last] = 0;
				if (last > 0 && equal[last-1] == '\r') equal[last-1] = 0;
			}
			std::string fullpath = dir + "/" + std::string(equal);

			if (!strcmp(buffer, "scale")) 
			{ 
				bScale	= true;
				scale	= atof( equal );
				continue;
			}

			std::cout << "    subloading " << buffer << ": " << fullpath.c_str() << std::endl;

			if (!strcmp(buffer, "skeleton")) 
			{
				if(!calCoreModel->loadCoreSkeleton(fullpath))
				{
					std::cout << "LoadCFG error: " << CalError::getLastErrorDescription().c_str() << std::endl;
				}					
			} 
			else if (!strcmp(buffer, "animation"))
			{
				if(calCoreModel->loadCoreAnimation(fullpath) < 0)
				{
					std::cout << "LoadCFG error: " << CalError::getLastErrorDescription().c_str() << std::endl;
				}
			}
			else if (!strcmp(buffer, "mesh"))
			{
				if(calCoreModel->loadCoreMesh(fullpath) < 0)
				{
					std::cout << "LoadCFG error: " << CalError::getLastErrorDescription().c_str() << std::endl;
				}
			}
			else if (!strcmp(buffer, "material"))  
			{
				int materialId = calCoreModel->loadCoreMaterial(fullpath);

				if(materialId < 0) 
				{
					std::cout << "LoadCFG error: " << CalError::getLastErrorDescription().c_str() << std::endl;
				} 
				else 
				{
					calCoreModel->createCoreMaterialThread(materialId); 
					calCoreModel->setCoreMaterialId(materialId, 0, materialId);


					CalCoreMaterial* material = calCoreModel->getCoreMaterial(materialId);
					// the texture file path is relative to the CRF data directory
					for(std::vector<CalCoreMaterial::Map>::iterator i = material->getVectorMap().begin();
						i != material->getVectorMap().end();
						i++) 
					{
						i->strFilename = dir + "/" + i->strFilename;
					}

				}
			}
		}
	}

	// scaling must be done after everything has been created
	if( bScale )
	{
		std::cout << "scaling model by " << scale << " factor" << std::endl;
		calCoreModel->scale( scale );
	}
	fclose(f);

	return true;
}

#define USE_OSG_PLUGIN


int main() {

	//////// osgCal stuff //////////
	// create a new model
	osg::ref_ptr<osgCal::Model> modelReferenced = new osgCal::Model();
	osgCal::Model *model = modelReferenced.get();

#ifndef USE_OSG_PLUGIN
	
	// We will load here the coreModel 
	// from .cfg file using global function
	// ****************************************
	// Creating a core model, the template
	osg::ref_ptr<osgCal::CoreModel> coreModelReferenced	= new osgCal::CoreModel();
	osgCal::CoreModel *coreModel = coreModelReferenced.get();

	// Load model attributes from file
	bool bSuccess	= loadCoreModelFromCFG(	"data", 
		"data/paladin.cfg",
		coreModel);

	if( !bSuccess )
	{
		std::cout << "Cannot load correctly model config file." << std::endl;
		return -1;
	}

#else

	// Load model using plugin, osgPlugin cfg
	// **************************************
	osg::Object* object	= osgDB::readObjectFile("data/paladin.cfg");

	osg::ref_ptr<osgCal::CoreModel> coreModelRef	= dynamic_cast<osgCal::CoreModel*>( object );
	osgCal::CoreModel* coreModel	= coreModelRef.get();
  
#endif

	if(!model->setCoreModel(coreModel)) {
		std::cerr << "setCoreModel: " << CalError::getLastErrorDescription().c_str() << std::endl;
		return -1;
	}

	for(int coreMeshId = 0; coreMeshId < model->getCalCoreModel()->getCoreMeshCount(); coreMeshId++) {
		model->getCalModel()->attachMesh(coreMeshId);
	}

	// set the material set of the whole model
	model->getCalModel()->setMaterialSet(0);

	// Creating a concrete model using the core template
	if(!model->create()) {
		std::cerr << "create: " << CalError::getLastErrorDescription().c_str() << std::endl;
		return -1;
	}

	// OpenSceneGraph Stuff
	// create a viewer
	osgProducer::Viewer viewer;
	// set up the value with sensible default event handlers.
	viewer.setUpViewer( osgProducer::Viewer::STANDARD_SETTINGS );

	// Setting the first animation in loop mode, weight 1, and starting just now
	model->getCalModel()->getMixer()->blendCycle(0, 1.0f, 0.0f);

	// Create new manipulator. 
	MyManipulator* tm	= new MyManipulator(model);

	// attach manipulator to model node. 
	//tm->setNode( model );
	tm->setTrackNode( model );

	osg::Group* root	= new osg::Group();

/*
	// START_FRAME
	// You can decomment the lines from START_FRAME to END_FRAME in order to 
	// add a xyz axis frame.
	osg::Geode* geode = new osg::Geode();



	// create a xyz frame

	// create Geometry object to store all the vetices and lines primtive.
	osg::Geometry* linesGeom = new osg::Geometry();

	// this time we'll prealloacte the vertex array to the size we
	// need and then simple set them as array elements, 4 points
	// makes 3 line segments.
	osg::Vec3Array* vertices = new osg::Vec3Array(4);
	double lh= 100.0;
	(*vertices)[0].set(0.0, 0.0, 0.0);
	(*vertices)[1].set(lh, 0.0, 0.0);
	(*vertices)[2].set(0.0, lh, 0.0);
	(*vertices)[3].set(0.0, 0.0, lh);



	// pass the created vertex array to the points geometry object.
	linesGeom->setVertexArray(vertices);

	// set the colors as before, plus using the aobve
	osg::Vec4Array* colors = new osg::Vec4Array;
	colors->push_back(osg::Vec4(1.0f,0.0f,0.0f,1.0f));
	colors->push_back(osg::Vec4(0.0f,1.0f,0.0f,1.0f));
	colors->push_back(osg::Vec4(0.0f,0.0f,1.0f,1.0f));
	linesGeom->setColorArray(colors);

	osg::TemplateIndexArray
		<unsigned int, osg::Array::UIntArrayType,4,4> *colorIndexArray;
	colorIndexArray =
		new osg::TemplateIndexArray<unsigned int, osg::Array::UIntArrayType,4,4>;
	colorIndexArray->push_back(0); // vertex 0 assigned color array element 0
	colorIndexArray->push_back(0); // vertex 1 assigned color array element 0
	colorIndexArray->push_back(1); // vertex 2 assigned color array element 1
	colorIndexArray->push_back(2); // vertex 3 assigned color array element 2
	linesGeom->setColorIndices(colorIndexArray);		
	linesGeom->setColorBinding(osg::Geometry::BIND_PER_VERTEX);


	// set the normal in the same way color.
	osg::Vec3Array* normals = new osg::Vec3Array;
	normals->push_back(osg::Vec3(0.0f,-1.0f,0.0f));
	linesGeom->setNormalArray(normals);
	linesGeom->setNormalBinding(osg::Geometry::BIND_OVERALL);


	// This time we simply use primitive, and hardwire the number of coords to use 
	// since we know up front,
	osg::DrawElementsUInt* line1 =
		new osg::DrawElementsUInt(osg::PrimitiveSet::LINES, 0);
	line1->push_back(0);
	line1->push_back(1);

	osg::DrawElementsUInt* line2 =
		new osg::DrawElementsUInt(osg::PrimitiveSet::LINES, 0);
	line2->push_back(0);
	line2->push_back(2);

	osg::DrawElementsUInt* line3 =
		new osg::DrawElementsUInt(osg::PrimitiveSet::LINES, 0);
	line3->push_back(0);
	line3->push_back(3);

	linesGeom->addPrimitiveSet(line1);
	linesGeom->addPrimitiveSet(line2);
	linesGeom->addPrimitiveSet(line3);

	// add the points geometry to the geode.
	geode->addDrawable(linesGeom);
	osg::StateSet* stateSet	= new osg::StateSet();

	stateSet->setMode( GL_SMOOTH, osg::StateAttribute::OFF );
	stateSet->setMode( GL_FLAT, osg::StateAttribute::ON   | osg::StateAttribute::OVERRIDE );
	stateSet->setMode( GL_LIGHTING, osg::StateAttribute::OFF );
	osg::LineWidth* lineWidth	= new osg::LineWidth( 5.0 );
	
	stateSet->setAttribute( lineWidth );
	geode->setStateSet( stateSet );

	// Add to root node
	root->addChild( geode );
	
	// END_FRAME
	*/

	root->addChild( model );

	// Define scene data node. 
	viewer.setSceneData(root);

	// Add new manipulator to viewer
	unsigned int id = viewer.addCameraManipulator(tm);

	// Select this manipulator
	viewer.selectCameraManipulator(id);

	viewer.realize();

	// Retrieve sceneHandlerList. For each scene handler a camera
	// is associated. So here we have Theoretically one camera, so one
	// scene handler. But we use a classic list traversal, just for
	// general usage.
	osgProducer::OsgCameraGroup::SceneHandlerList &shlist = viewer.getSceneHandlerList();
	osgProducer::OsgCameraGroup::SceneHandlerList::iterator it;

	// Set some culling settings : Disable near far automatic computing to be sure
	// Set default culling
	viewer.getCullSettings().setDefaults();
	viewer.getCullSettings().setComputeNearFarMode( osg::CullSettings::DO_NOT_COMPUTE_NEAR_FAR );
	viewer.getCullSettings().setCullingMode( osg::CullSettings::DEFAULT_CULLING );
	
	for (it = shlist.begin(); it != shlist.end(); it++)
	{
		
		// Retrieve scene view from scene handler
		osgUtil::SceneView* pView	= (*it)->getSceneView();

		// Retrieve global state set
		osg::StateSet* globalStateSet = pView->getGlobalStateSet();
		if (!globalStateSet)
		{
			globalStateSet = new osg::StateSet;
			pView->setGlobalStateSet(globalStateSet);
		}

		// disable back face culling else the both cape sides are not rendered 
		globalStateSet->setMode(GL_CULL_FACE, osg::StateAttribute::OFF  | osg::StateAttribute::OVERRIDE );

	}

	while ( !viewer.done() ) 
	{
		viewer.sync();
		viewer.update();
		viewer.frame();
	}
	viewer.sync();

	return 0;
}

