/* -*-c++-*- */
/**
 * OsgAL - OpenSceneGraph Audio Library
 * Copyright (C) 2004 VRlab, Ume University
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.
 */

#include <osgAL/OccludeCallback>
#include <osgUtil/IntersectVisitor>
#include <iostream>
#include <osgAL/SoundNode>
#include <osgAL/Math>
using namespace osgAL;


OccludeCallback::OccludeCallback(osg::Node *root) : m_root(root), m_sound_node(0L), m_ear_distance(0.2), m_near_threshold(0.1f),
  m_was_occluded(false), m_delay(10)
{
}

/// we set here a empty node. This constructor must be called by osg when reading a file,
/// and later will be set the real node.
OccludeCallback::OccludeCallback() : m_root(new osg::Node()), m_sound_node(0L), m_ear_distance(0.2), m_near_threshold(0.1f),
m_was_occluded(false), m_delay(10)
{
}

void OccludeCallback::operator()(double /*distance*/, osg::Node * /*occluder*/, bool left_occluded, bool /*right_occluded*/)
{
  osgAL::SoundState *sound_state = m_sound_node->getSoundState();

  // Sound node is occluded by something
  if (left_occluded) {
    
    // Was it not occluded last frame?
    if (!m_was_occluded)
      m_start_tick = osg::Timer::instance()->tick(); // Then start timer

    // Linearly interpolate occlusion from 0 to max
    double dt = m_delay*osg::Timer::instance()->delta_s(m_start_tick, osg::Timer::instance()->tick());
    float scale = osgAL::mix(1.0f, 0.0f, dt);
    sound_state->setOccludeScale(scale);
    m_sound_node->getSoundState()->setOccluded(true);
  }
  else { // Is not occluded anymore

    // If occlusion is already shut of, do no more
    if (!sound_state->getOccluded())
      return;

    if (m_was_occluded) // Was it occluded last frame, then start timer
      m_start_tick = osg::Timer::instance()->tick();
    
    // Interpolate from max to 0 damping
    double dt = m_delay*osg::Timer::instance()->delta_s(m_start_tick, osg::Timer::instance()->tick());
    float scale = osgAL::mix(0.0f, 0.99f, dt);
    sound_state->setOccludeScale(scale);
    
    // When enough time have passed, disable occlusion
    if (dt > 1/m_delay)
      m_sound_node->getSoundState()->setOccluded(false);
  }
}

void OccludeCallback::apply(const osg::Matrix& listener_matrix, const osg::Vec3& sound_pos, osgAL::SoundNode *sound_node)
{
 m_sound_node = sound_node;

  osg::Node *occluder=0L;
  double distance = 0;

  osg::Matrix m = listener_matrix.inverse(listener_matrix);

  // Now shoot a ray from the ear to the source, see if it hits anything.
  osgUtil::IntersectVisitor iv;
  osg::ref_ptr<osg::LineSegment> left = new osg::LineSegment;

  osg::Vec3 start(m.getTrans()), end(sound_pos);
  osg::Vec3 dir = end-start;
  distance = dir.length();
  dir /= distance;

  left->set(start,start+dir*distance);
  
  if(left->valid())
	iv.addLineSegment(left.get());

  m_root->accept(iv);
  bool occluded = false;
  if (iv.hits()) {
    osgUtil::IntersectVisitor::HitList::iterator hit = iv.getHitList( left.get() ).begin();

    osg::Vec3 hit_point = hit->getWorldIntersectPoint();
    double d = (hit_point - start).length();
    double diff = fabs(d - distance);

    if ( diff > m_near_threshold) {
      occluded = true;

        osg::NodePath &np = hit->_nodePath;
      if (np.size())
        occluder = *(np.begin());

      this->operator ()(d, occluder, true, true);
    }
  }
 
  // If its not occluded this frame but it was the previous, restore the state
  if (!occluded)
    this->operator ()(0, NULL, false, false);

  // save the state of the occlusion to next frame
  m_was_occluded = occluded;    
}

