/* OpenVAS-Client
 * $Id$
 * Description: LSC Credentials management - module to create and unlink key
 *              files.
 *
 * Authors:
 * Felix Wolfsteller <felix.wolfsteller@intevation.de>
 *
 * Copyright:
 * Copyright (C) 2008 Intevation GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2,
 * or, at your option, any later version as published by the Free
 * Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * In addition, as a special exception, you have
 * permission to link the code of this program with the OpenSSL
 * library (or with modified versions of OpenSSL that use the same
 * license as OpenSSL), and distribute linked combinations including
 * the two. You must obey the GNU General Public License in all
 * respects for all of the code used other than OpenSSL. If you
 * modify this file, you may extend this exception to your version
 * of the file, but you are not obligated to do so. If you do not
 * wish to do so, delete this exception statement from your version.
 */

#include "openvas_ssh_login.h"
#include "openvas_ssh_key_create.h"
#include "nessus_i18n.h"
#include "error_dlg.h"
#include <glib/gstdio.h>
#include "context.h"

/**
 * Ensures the existance of a directory by creating it if it does not exist.
 * ATTENTION: can only create "one more sublevel"!
 * 
 * @param directory Path of directory to check or create.
 * 
 * @return TRUE if directory exists or was successfully created, FALSE otherwise.
 * 
 * @see ensure_dir
 */
static gboolean
ensure_single_dir (char* directory)
{
  //Should be moved together with file functions defined in context.h/c
  // to an own module e.g. file_check.c
  if (check_is_dir (directory) == 1)
    return TRUE;
  // Glib >= 2.8 defines g_mkdir_with_parents
  if (g_mkdir (directory, 0700) == 0)
    return TRUE;
  else
    return FALSE;
}

/**
 * @brief Ensures the existance of a directory, including parents.
 * 
 * This somewhat lazy method checks if a directory and its parent directories
 * exists. If not, it tries to make them. This method will be obselete once
 * depending on GLib >= 2.8 (g_mkdir_with_parents).
 * 
 * @return TRUE if the director(y|ies) exists or was sucessfully created, FALSE
 *         otherwise.
 */
gboolean
ensure_dir (char* dir)
{
  char* parentdir = estrdup (dir);
  GQueue* queue = g_queue_new ();
  gboolean failed = FALSE;

  // Push all non-existing parent directories.
  while (parentdir && check_is_dir (parentdir) == 0)
    {
      g_queue_push_head (queue, parentdir);
      parentdir = g_path_get_dirname (parentdir) ;
    }
  
  // Pop all non-existing directories, create them, release string.
  parentdir = g_queue_pop_head (queue);
  while ( parentdir )
    {
      if (failed == FALSE )
        failed = ! ensure_single_dir (parentdir);
      g_free (parentdir);
      parentdir = g_queue_pop_head (queue);
    }
  
  // Clean up and return.
  g_queue_free(queue);
  return !failed;
}

/**
 * @brief Creates a private key for local checks.
 * 
 * Forks and creates a key for local checks by calling
 * "openssl pkcs8 -topk8 -v2 des3 -in filepath -passin pass:passphrase -out
 *          filepath.p8 -passout pass:passphrase"
 * Directories within privkey_file will be created if they do not exist.
 * 
 * @param pubkey_file Path to file of public key (a trailing .pub will be stripped).
 * @param privkey_file Name of private key file to be created.
 * 
 * @param passphrase_pub The passphrase for the public key.
 * @param passphrase_priv Passhprase for the private key.
 * 
 * @return TRUE if successfull, FALSE otherwise.
 */
static
gboolean openvas_ssh_privkey_create(char* pubkey_file, char* privkey_file,
                                    char* passphrase_pub, char* passphrase_priv)
{
  gchar* astdout = NULL;
  gchar* astderr = NULL;
  GError* err    = NULL;
  gint exit_status;
  char* dir = NULL;
  gchar* pubkey_stripped = NULL;
  
  /* Sanity-check essential parameters */
  if(!passphrase_pub || !passphrase_priv)
  {
    show_error(_("Error creating private key file:\nPlease provide all information."));
    return FALSE;
  }
  
  /* Sanity check files */
  if(check_exists(pubkey_file) != 1)
  {
    show_error(_("Error creating private key file:\nPublic key %s not found."), pubkey_file);
    return FALSE;
  }
  if(check_exists(privkey_file) != 0 )
  {
    show_error(_("Error creating private key file:\nFile already exists."));
    return FALSE;
  }
  dir = g_path_get_dirname(privkey_file);
  if(ensure_dir(dir) != TRUE)
  {
    show_error(_("Error creating private key file:\nfolder %s not accessible."), dir);
    efree(&dir);
    return FALSE;
  }
  efree(&dir);
  
  // Strip ".pub" of public key filename, if any.
  if(g_str_has_suffix(pubkey_file, ".pub") == TRUE)
  {
    pubkey_stripped = emalloc(strlen(pubkey_file) - strlen(".pub") +1);
    g_strlcpy (pubkey_stripped, pubkey_file, strlen(pubkey_file) - strlen(".pub") + 1);
  }
  else
    pubkey_stripped = g_strdup(pubkey_file);

  /* Fire openssl */
  const char* command = g_strconcat("openssl pkcs8 -topk8 -v2 des3 -in ", pubkey_stripped,
                                    " -passin pass:", passphrase_pub, " -out ",
                                    privkey_file, " -passout pass:",
                                    passphrase_priv, NULL);
  efree (&pubkey_stripped);

  if(g_spawn_command_line_sync(command, &astdout, &astderr, &exit_status, &err) == FALSE
          || exit_status != 0 )
  {
    show_error(_("Error creating private key file.\nFor further information consult your shell."));
    printf("Error creating private key file.");
    printf("\tSpawned openssl process returned with %d.\n", exit_status);
    printf("\t\t stdout: %s\n", astdout);
    printf("\t\t stderr: %s\n", astderr);
    return FALSE;
  }

  return TRUE;
}

/**
 * Forks and creates a key for local checks by calling
 * "ssh-keygen -t rsa -f filepath -C comment -P passhprase -q"
 * A directory will be created if it does not exist.
 * 
 * @param comment Comment to use (will be freed).
 * @param passphrase The passphrase for the key (will be freed), must be longer
 *                   than 4 characters (+nul).
 * @param filepath Path to file of public key (a trailing .pub will be stripped).
 * 
 * @return TRUE if successfull, FALSE otherwise.
 */
static
gboolean openvas_ssh_pubkey_create(char* comment, char* passphrase,
                                   char* filepath)
{
  gchar* astdout = NULL;
  gchar* astderr = NULL;
  GError* err = NULL;
  gint exit_status;
  char* dir;
  char* file_pubstripped;
  
  /* Sanity-check essential parameters */
  if( !comment || comment[0] == '\0' )
  {
    show_error(_("Error creating public key file:\ncomment has to be set."));
    return FALSE;
  }
  if(!passphrase || strlen(passphrase) < 5)
  {
    show_error(_("Error creating public key file:\npassword must be longer than 4 characters."));
    return FALSE;
  }
  /* Sanity check files */
  dir = g_path_get_dirname(filepath);
  if(ensure_dir(dir) != TRUE)
  {
    show_error(_("Error creating public key file:\n%s is not accessable."), filepath);
    efree(&dir);
    return FALSE;
  }
  efree(&dir);
  if(check_exists(filepath) == 1)
  {
    show_error(_("Error creating public key file:\n%s already exists."), filepath);
    return FALSE;
  }
  
  // Strip ".pub" of filename, if any.
  if(g_str_has_suffix(filepath, ".pub") == TRUE)
  {
    file_pubstripped = emalloc(strlen(filepath) - strlen(".pub") +1);
    g_strlcpy(file_pubstripped, filepath, strlen(filepath) - strlen(".pub") + 1);
  }
  else
    file_pubstripped = g_strdup(filepath);
  
  /* Fire ssh-keygen */
  const char* command = g_strconcat("ssh-keygen -t rsa -f ", file_pubstripped, " -C ",
                                    comment, " -P ", passphrase, NULL);
  efree(&file_pubstripped);

  if(g_spawn_command_line_sync(command, &astdout, &astderr, &exit_status, &err) == FALSE
     || exit_status != 0 )
  {
    show_error(_("Error creating public key file.\nFor further information consult your shell."));
    printf("Error creating public key file.\n");
    printf("\tSpawned key-gen process returned with %d.\n", exit_status);
    printf("\t\t stdout: %s", astdout);
    printf("\t\t stderr: %s", astderr);
    return FALSE;
  }
  return TRUE;
}


/**
 * @brief Creates the public and private key files.
 * 
 * @param loginfo.
 * @return TRUE if things went good, FALSE if things went bad.
 */
gboolean openvas_ssh_key_create(openvas_ssh_login* loginfo)
{
  /* Create pubkey */
  gboolean success = openvas_ssh_pubkey_create(loginfo->comment, 
                                               loginfo->ssh_key_passphrase,
                                               loginfo->public_key_path);
  
  /* Eventually report failure */
  if(success == FALSE)
    return FALSE;
  
  /* Create private key */
  success = openvas_ssh_privkey_create(loginfo->public_key_path,
                                       loginfo->private_key_path,
                                       loginfo->ssh_key_passphrase,
                                       loginfo->ssh_key_passphrase);
  return success;
}

/**
 * @brief Unlinks pub. and private key files + identity file.
 * 
 * @param loginfo Login of which to unlink files.
 */
void
openvas_ssh_key_create_unlink_files (openvas_ssh_login* loginfo)
{
  char* identity_file = NULL;
  
  if (loginfo == NULL)
    return;

  // Create identity file path
  if (loginfo->public_key_path != NULL)
    {
      int len = strlen(loginfo->public_key_path) - strlen (".pub") + 1;
      if (len > 0)
        {
          identity_file = emalloc (len);
          g_strlcpy (identity_file, loginfo->public_key_path, len);
        }

      // Delete all the files
      unlink (identity_file);
      unlink (loginfo->private_key_path);
      unlink (loginfo->public_key_path);
    }
  
  efree (&identity_file);
}

/**
 * @brief Unlinks the pub and private key + identity files, removes
 *        Global->sshkeys entry for a single login and frees the memory.
 * 
 * If you want to 'harden' this delete, make sure that you rewrite the .login
 * file.
 * 
 * @param loginfo Login of which to unlink files and free memory.
 */
void
openvas_ssh_key_create_undo (openvas_ssh_login* loginfo)
{
  if (loginfo == NULL)
    return;
  
  openvas_ssh_key_create_unlink_files (loginfo);
  // Free associated memory.
  g_hash_table_remove (Global->sshkeys, loginfo->name);
}

/**
 * @brief Attempts creation of RPM packages to install a users public key file.
 * 
 * @param loginfo openvas_ssh_login struct to create rpm for.
 * 
 * @return Path to rpm file if successfull, NULL otherwise.
 */
gchar*
openvas_ssh_key_create_rpm (openvas_ssh_login* loginfo)
{
  // The scripts to create rpms are currently in trunk/tools/openvas-lsc-target-preparation.
  // Move to trunk/openvas-client/tools will be done when function is stable.
  gchar* oltap_path = g_strdup("../tools/openvas-lsc-target-preparation");
  gchar* rpm_path = NULL;

#ifdef DEV_SSH_RPM_EXPORT
  gint exit_status;
  gchar* pubkeyfile_contents = NULL;
  gsize  pubkeyfile_length;
  gchar* new_pubkey_filename;
  gchar* pubkey_basename;
  gchar** cmd;

  if (check_is_dir(oltap_path) == 0)
    {
      show_warning ("This feature is under developement.\n"
                    "Did you checkout the whole trunk and are running the client"
                    "from the /trunk/openvas-client - directory and configured with --enable-debug?");
      return NULL;
    }

  printf ("Attempting rpm build for %s with pubkey %s.\n", loginfo->name, loginfo->public_key_path);
  // First, copy public key into openvas lsc target prep tool directory
  if (g_file_get_contents (loginfo->public_key_path, &pubkeyfile_contents,
                           &pubkeyfile_length, NULL)
       == FALSE)
    {
      // Look whats in err
      printf ("file content getting failed\n");
      return NULL;
    }

  pubkey_basename = g_path_get_basename (loginfo->public_key_path);
  new_pubkey_filename = g_build_filename (oltap_path, pubkey_basename, NULL);
  FILE* fd = fopen (new_pubkey_filename, "w");
  if (fd == NULL)
    {
      show_error ("Could not copy key file %s to %s.",
                  loginfo->public_key_path, new_pubkey_filename);
      return NULL;
    }
  fprintf (fd, "%s", pubkeyfile_contents);
  fclose (fd);
  g_free (pubkeyfile_contents);

  // create-rpm
  printf ("--- RPM-builder: create-rpm\n");

  cmd = (gchar **) g_malloc (3 * sizeof (gchar *));
  cmd[0] = g_strdup ("./create-rpm.sh");
  cmd[1] = g_strdup (g_path_get_basename (loginfo->public_key_path) );
  cmd[2] = NULL;
  if (g_spawn_sync (oltap_path,
                    cmd,
                    NULL, // env
                    G_SPAWN_SEARCH_PATH,
                    NULL, // setup func
                    NULL,
                    NULL,
                    NULL,
                    &exit_status,
                    NULL                 ) == FALSE
      || exit_status != 0)
    {
      show_error(_("Error (%d) creating the rpm.\n"
                   "For further information consult your shell."), exit_status);
      return NULL;
    }

  g_free (cmd[0]);
  g_free (cmd[1]);
  g_free (cmd[2]);
  g_free (cmd);
  g_free (pubkey_basename);
  g_free (new_pubkey_filename);

  // e.g. RPMS/noarch/openvas-lsc-target-example_user-0.5-1.noarch.rpm
  //                   RPMBASENAME PUBKEYNAME VERSION -1.noarch.rpm
  gchar* oltap_rpmbase = g_build_filename (oltap_path, "RPMBASENAME", NULL);
  g_file_get_contents (oltap_rpmbase, &pubkeyfile_contents, &pubkeyfile_length, NULL);

  gchar* oltap_pubkey  = g_build_filename (oltap_path, "PUBKEYNAME", NULL);
  gchar** pkncontents;
  g_file_get_contents (oltap_pubkey, &pkncontents, &pubkeyfile_length, NULL);
  /*
  gchar* oltap_version = g_build_filename (oltap_path, "VERSION", NULL);
  g_file_get_contents (oltap_version, &pubkeyfile_contents, &pubkeyfile_length, NULL);
  */

  gchar* rpmfile = g_strconcat (g_strstrip(pubkeyfile_contents), "-",
                                g_strstrip(pkncontents), "-0.5-1.noarch.rpm", NULL);

  rpm_path = g_build_filename (oltap_path, "RPMS", "noarch", rpmfile, NULL);
#endif /* DEV_SSH_RPM_EXPORT */

  g_free (oltap_path);

  return rpm_path;
}
