//                                               -*- C++ -*-
/**
 *  @file  ConstantRandomVector.cxx
 *  @brief An implementation class for constant random vectors
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: ConstantRandomVector.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include "PersistentObjectFactory.hxx"
#include "ConstantRandomVector.hxx"
#include "UserDefined.hxx"
#include "SquareMatrix.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Model
    {

      typedef OT::Uncertainty::Distribution::UserDefined UserDefined;
      typedef UserDefined::PairCollection                PairCollection;
      typedef Base::Type::SquareMatrix                   SquareMatrix;

      CLASSNAMEINIT(ConstantRandomVector);

      static Base::Common::Factory<ConstantRandomVector> RegisteredFactory("ConstantRandomVector");

      /* Default constructor */
      ConstantRandomVector::ConstantRandomVector(const NumericalPoint & point, const String & name)
        : RandomVectorImplementation(name),
          point_(point)
      {
        setDescription( point.getDimension() );
      }


      /* Virtual constructor */
      ConstantRandomVector * ConstantRandomVector::clone() const
      {
        return new ConstantRandomVector(*this);
      }

      /* String converter */
      String ConstantRandomVector::__repr__() const {
        OSS oss;
        oss << "class=" << ConstantRandomVector::GetClassName();
        return oss;
      }



      /* Here is the interface that all derived class must implement */


      /* Dimension accessor */
      UnsignedLong ConstantRandomVector::getDimension() const
      {
        return point_.getDimension();
      }

      /* Realization accessor */
      ConstantRandomVector::NumericalPoint ConstantRandomVector::getRealization() const
      {
        return point_;
      }

      /* Numerical sample accessor */
      ConstantRandomVector::NumericalSample ConstantRandomVector::getNumericalSample(UnsignedLong size) const
      {
        return NumericalSample(size, point_);
      }

      /* Mean accessor */
      ConstantRandomVector::NumericalPoint ConstantRandomVector::getMean() const
      {
        return point_;
      }

      /* Covariance accessor */
      ConstantRandomVector::CovarianceMatrix ConstantRandomVector::getCovariance() const
      {
        return CovarianceMatrix(SquareMatrix(getDimension()).getImplementation());
      }

      /* Get the random vector corresponding to the i-th marginal component */
      RandomVectorImplementation::Implementation ConstantRandomVector::getMarginal(const UnsignedLong i) const /* throw(InvalidArgumentException) */
      {
        if (i >= getDimension()) throw InvalidArgumentException(HERE) << "The index of a marginal random vector must be in the range [0, dim-1]";
        return new ConstantRandomVector(NumericalPoint(1, point_[i]));
      }

      /* Get the marginal random vector corresponding to indices components */
      RandomVectorImplementation::Implementation ConstantRandomVector::getMarginal(const Indices & indices) const /* throw(InvalidArgumentException) */
      {
        if (!indices.check(getDimension() - 1)) throw InvalidArgumentException(HERE) << "The indices of a marginal random vector must be in the range [0, dim-1] and  must be different";
        UnsignedLong marginalDimension(indices.getSize());
        NumericalPoint marginalPoint(marginalDimension);
        for (UnsignedLong i = 0; i < marginalDimension; ++i)
          {
            marginalPoint[i] = point_[indices[i]];
          }
        return new ConstantRandomVector(marginalPoint);
      }

      /* Distribution accessor */
      Distribution ConstantRandomVector::getDistribution() const
      {
        PairCollection collection(1);
        collection[0].setX( point_ );
        collection[0].setP( 1.0 );
        return UserDefined(collection);
      }

      /* Method save() stores the object through the StorageManager */
      void ConstantRandomVector::save(StorageManager::Advocate & adv) const
      {
        RandomVectorImplementation::save(adv);
        adv.saveAttribute( "point_", point_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void ConstantRandomVector::load(StorageManager::Advocate & adv)
      {
        RandomVectorImplementation::load(adv);
        adv.loadAttribute( "point_", point_ );
      }

    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
