//                                               -*- C++ -*-
/**
 *  @file  ExponentialFactory.cxx
 *  @brief Factory for Exponential distribution
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: ExponentialFactory.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include "ExponentialFactory.hxx"
#include "Exponential.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      /* Default constructor */
      ExponentialFactory::ExponentialFactory():
	DistributionImplementationFactory()
      {
	// Nothing to do
      }

      /* Virtual constructor */
      ExponentialFactory * ExponentialFactory::clone() const
      {
	return new ExponentialFactory(*this);
      }

      /* Here is the interface that all derived class must implement */

      ExponentialFactory::DistributionImplementation * ExponentialFactory::buildImplementation(const NumericalSample & sample) const /* throw(InvalidArgumentException, InternalException) */
      {
	if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a Exponential distribution from an empty sample";
	if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build an Exponential distribution only from a sample of dimension 1, here dimension=" << sample.getDimension();
	NumericalScalar gamma(sample.getMin()[0]);
	// Correction for bias control
	NumericalScalar size(sample.getSize());
	if (gamma > 0.0)
	  {
	    gamma *= 1.0 - 1.0 / (2.0 + size);
	  }
	else
	  {
	    gamma *= 1.0 + 1.0 / (2.0 + size);
	  }
	NumericalScalar mean(sample.computeMean()[0]);
        if (mean == gamma) throw InternalException(HERE) << "Error: can build an Exponential distribution only from a sample with mean > min, here mean=" << mean << " and min=" << gamma;
	NumericalScalar lambda(1.0 / (mean - gamma));
	return Exponential(lambda, gamma).clone();
      }

      ExponentialFactory::DistributionImplementation * ExponentialFactory::buildImplementation(const NumericalPointCollection & parameters) const /* throw(InvalidArgumentException, InternalException) */
      {
	try {
	  Exponential distribution;
	  distribution.setParametersCollection(parameters);
	  return distribution.clone();
	}
	catch (InvalidArgumentException & ex)
	  {
	    throw InternalException(HERE) << "Error: cannot build a Exponential distribution from the given parameters";
	  }
      }

      ExponentialFactory::DistributionImplementation * ExponentialFactory::buildImplementation() const /* throw(InvalidArgumentException, InternalException) */
      {
	return Exponential().clone();
      }


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
