//                                               -*- C++ -*-
/**
 *  @file  Simulation.hxx
 *  @brief Simulation is a generic view of simulation methods for computing
 * probabilities and related quantities by sampling and estimation
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-03 16:04:26 +0100 (mer. 03 févr. 2010) $
 *  Id:      $Id: Simulation.hxx 1466 2010-02-03 15:04:26Z dutka $
 */
#ifndef OPENTURNS_SIMULATION_HXX
#define OPENTURNS_SIMULATION_HXX

#include "PersistentObject.hxx"
#include "Exception.hxx"
#include "HistoryStrategy.hxx"
#include "Compact.hxx"
#include "Last.hxx"
#include "Graph.hxx"
#include "SimulationResult.hxx"
#include "Event.hxx"


namespace OpenTURNS
{

  namespace Uncertainty 
  {

    namespace Algorithm
    {


      /**
       * @class Simulation
       */

      class Simulation
	: public Base::Common::PersistentObject
      {


	CLASSNAME;
      public:
	static const UnsignedLong    DefaultMaximumOuterSampling;
	static const NumericalScalar DefaultMaximumCoefficientOfVariation;
	static const UnsignedLong    DefaultBlockSize;

	typedef Base::Common::InvalidArgumentException InvalidArgumentException;
	typedef Model::Event                           Event;
	typedef Base::Stat::NumericalSample            NumericalSample;
	typedef Base::Stat::HistoryStrategy            HistoryStrategy;
	typedef Base::Stat::Compact                    Compact;
	typedef Base::Stat::Last                       Last;
	typedef Base::Graph::Graph                     Graph;
	typedef Base::Common::StorageManager           StorageManager;
 
	/** Constructor with parameters */
	explicit Simulation(const Event & event,
			    const Bool verbose = true,
			    const HistoryStrategy & convergenceStrategy = Compact(),
			    const HistoryStrategy & inputOutputStrategy = Last());

	/** Event accessor */
	Event getEvent() const;

	/** Result accessor */
	SimulationResult getResult() const;

	/** Maximum sample size accessor */
	void setMaximumOuterSampling(const UnsignedLong maximumOuterSampling);
	UnsignedLong getMaximumOuterSampling() const;

	/** Maximum coefficient of variation accessor */
	void setMaximumCoefficientOfVariation(const NumericalScalar maximumCoefficientOfVariation)
	  /* throw(InvalidArgumentException) */;
	NumericalScalar getMaximumCoefficientOfVariation() const;

	/** Block size accessor */
	void setBlockSize(const UnsignedLong blockSize) /* throw(InvalidArgumentException) */;
	UnsignedLong getBlockSize() const;

	/** Verbosity accessor */
	void setVerbose(const Bool verbose);
	Bool getVerbose() const;

	/** Convergence strategy accessor */
	void setConvergenceStrategy(const HistoryStrategy & convergenceStrategy);
	HistoryStrategy getConvergenceStrategy() const;

	/** Input output strategy accessor */
	void setInputOutputStrategy(const HistoryStrategy & inputOutputStrategy);
	HistoryStrategy getInputStrategy() const;
	HistoryStrategy getOutputStrategy() const;

	/** String converter */
	String __repr__() const;

	/** Performs the actual computation. */
	virtual void run();

	/** Draw the probability convergence at the given level */
	Graph drawProbabilityConvergence(const NumericalScalar level = SimulationResultImplementation::DefaultConfidenceLevel) const;

	/** Method save() stores the object through the StorageManager */
	void save(StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(StorageManager::Advocate & adv);

      protected:

	/** Result accessor */
	void setResult(const SimulationResult & result);

	/** Compute the block sample */
	virtual NumericalSample computeBlockSample() = 0;

      protected:

        /** For save/load mechanism*/
        Simulation(const Bool verbose = true,
          const HistoryStrategy & convergenceStrategy = Compact(),
          const HistoryStrategy & inputOutputStrategy = Last()) {};
  
        friend class Base::Common::Factory<Simulation>;

	/** History strategy for the probability and variance estimate */
	HistoryStrategy convergenceStrategy_;

	/** History strategy for the input and the output of the model */
	HistoryStrategy inputStrategy_;
	HistoryStrategy outputStrategy_;

	// Size of the atomic blocks of computation
	UnsignedLong blockSize_;

	// The event we are computing the probability of
	Event event_;

	// Result of the simulation
	SimulationResult result_;

      private:

	// Maximum number of outer iteration allowed
	UnsignedLong maximumOuterSampling_;

	// Maximum coefficient of variation allowed for convergence
	NumericalScalar maximumCoefficientOfVariation_;

	// Do we have to echo all the intermediate results?
	Bool verbose_;

      } ; /* class Simulation */


    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_SIMULATION_HXX */
