//                                               -*- C++ -*-
/**
 *  @file  QuadraticTaylor.cxx
 *  @brief Second order polynomial response surface by Taylor expansion
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: QuadraticTaylor.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include "QuadraticTaylor.hxx"
#include "QuadraticNumericalMathEvaluationImplementation.hxx"
#include "LinearNumericalMathGradientImplementation.hxx"
#include "ConstantNumericalMathHessianImplementation.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      typedef Base::Func::QuadraticNumericalMathEvaluationImplementation QuadraticNumericalMathEvaluationImplementation;
      typedef Base::Func::LinearNumericalMathGradientImplementation    LinearNumericalMathGradientImplementation;
      typedef Base::Func::ConstantNumericalMathHessianImplementation   ConstantNumericalMathHessianImplementation;

      CLASSNAMEINIT(QuadraticTaylor);

      QuadraticTaylor::QuadraticTaylor(const String & name):
	PersistentObject(name)
      {
	// Nothing to do
      }

      /* Constructor with parameters */
      QuadraticTaylor::QuadraticTaylor(const NumericalPoint & center,
				       const NumericalMathFunction & inputFunction,
				       const String & name):
	PersistentObject(name),
	center_(center),
	inputFunction_(inputFunction)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      QuadraticTaylor * QuadraticTaylor::clone() const
      {
	return new QuadraticTaylor(*this);
      }

      /* String converter */
      String QuadraticTaylor::__repr__() const
      {
	OSS oss;
	oss << "class=" << GetClassName()
	    << " name=" << getName()
	    << " center=" << center_
	    << " function=" << inputFunction_
	    << " responseSurface=" << responseSurface_
	    << " constant=" << constant_
	    << " linear=" << linear_
	    << " quadratic=" << quadratic_;
	return oss;
      }

      /* Response surface computation */
      void QuadraticTaylor::run()
      {
	/* Compute the three first terms of the Taylor expansion */
	constant_ = inputFunction_(center_);
	linear_ = inputFunction_.gradient(center_);
	quadratic_ = inputFunction_.hessian(center_);
	/* Build the several implementations and set it into the response surface */
	responseSurface_.setEvaluationImplementation(new QuadraticNumericalMathEvaluationImplementation(center_, constant_, linear_, quadratic_));
	responseSurface_.setGradientImplementation(new LinearNumericalMathGradientImplementation(center_, linear_, quadratic_));
	responseSurface_.setHessianImplementation(new ConstantNumericalMathHessianImplementation(quadratic_));
	responseSurface_.setDescription(inputFunction_.getDescription());
      }

      /* Center accessor */
      QuadraticTaylor::NumericalPoint QuadraticTaylor::getCenter() const
      {
	return center_;
      }

      /* Constant accessor */
      QuadraticTaylor::NumericalPoint QuadraticTaylor::getConstant() const
      {
	return constant_;
      }

      /* Linear accessor */
      QuadraticTaylor::Matrix QuadraticTaylor::getLinear() const
      {
	return linear_;
      }

      /* Quadratic accessor */
      QuadraticTaylor::SymmetricTensor QuadraticTaylor::getQuadratic() const
      {
	return quadratic_;
      }

      /* Function accessor */
      QuadraticTaylor::NumericalMathFunction QuadraticTaylor::getInputFunction() const
      {
	return inputFunction_;
      }

      /* Response surface accessor */
      QuadraticTaylor::NumericalMathFunction QuadraticTaylor::getResponseSurface() const
      {
	return responseSurface_;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
