//                                               -*- C++ -*-
/**
 *  @file  ExponentialRBF.cxx
 *  @brief Implementation of the Exponential RBF kernel
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: ExponentialRBF.hxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "ExponentialRBF.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      CLASSNAMEINIT(ExponentialRBF);


      /* Constructor with parameters */
      ExponentialRBF::ExponentialRBF(const NumericalScalar sigma)
	: SVMKernelImplementation(),
	  sigma_(sigma)
      {
	// Nothing to do
      }


      /* Virtual constructor */
      ExponentialRBF * ExponentialRBF::clone() const
      {
	return new ExponentialRBF(*this);
      }


      /* String converter */
      String ExponentialRBF::__repr__() const
      {
	return OSS() << "class=" << getClassName()
		     << " sigma=" << getSigma();
      }


      /* Sigma parameter accessor */
      NumericalScalar ExponentialRBF::getSigma() const
      {
	return sigma_;
      }

      void ExponentialRBF::setSigma(NumericalScalar sigma)
      {
	sigma_ = sigma;
      }


      /* Accessor to the parameter used for cross-validation */
      NumericalScalar ExponentialRBF::getParameter() const
      {
	return sigma_;
      }

      void ExponentialRBF::setParameter(NumericalScalar value)
      {
	sigma_ = value;
      }


      /* Parameters value and description accessor */
      ExponentialRBF::NumericalPointWithDescription ExponentialRBF::getParameters() const
      {
	NumericalPointWithDescription parameters(0);
	Description description(0);
	parameters.add(sigma_);
	description.add("sigma");
	parameters.setDescription(description);
	return parameters;
      }

      void ExponentialRBF::setParameters(const NumericalPointWithDescription & parameters)
      {
	if(parameters.getDimension() > 0)
	  sigma_ = parameters[0];
      }


      /* Operator () */
      NumericalScalar ExponentialRBF::operator() (const NumericalPoint & x1, const NumericalPoint & x2) const
      {
	NumericalPoint difference(x1 - x2);
	NumericalScalar value = exp(- difference.norm() / (2.0 * sigma_ * sigma_));
	return value;
      }


      /* Partial gradient */
      ExponentialRBF::NumericalPoint ExponentialRBF::partialGradient(const NumericalPoint & x1, const NumericalPoint & x2) const
      {
	UnsignedLong dimension = x1.getDimension();
	NumericalPoint difference(x1 - x2);
	NumericalScalar norm = difference.norm();
	NumericalPoint partialGradient(dimension, 0.0);
	if(norm > 0.0)
	  {
	    for(UnsignedLong i = 0; i < dimension; ++ i)
	      {
		partialGradient[i] = exp(- norm / (2.0 * sigma_ * sigma_)) * (- 1.0 / (2.0 * sigma_ * sigma_)) * (x1[i] - x2[i]) / norm;
	      }
	  }
	return partialGradient;
      }


      /* Partial hessian */
      ExponentialRBF::SymmetricMatrix ExponentialRBF::partialHessian(const NumericalPoint & x1, const NumericalPoint & x2) const
      {
	UnsignedLong dimension = x1.getDimension();
	NumericalPoint difference(x1 - x2);
	NumericalScalar norm = difference.norm();
	SymmetricMatrix partialHessian(dimension);
	if(norm > 0.0)
	  {
	    for(UnsignedLong i = 0; i < dimension; ++ i)
	      {
		for(UnsignedLong j = 0; j <= i; ++ j)
		  {
		    if(i == j)
		      {
			partialHessian(i, i) = exp(- norm / (2.0 * sigma_ * sigma_)) * (- 1.0 / (2.0 * sigma_ * sigma_)) / norm * ( (- 1.0 / (2.0 * sigma_ * sigma_)) * (x1[i] - x2[i]) * (x1[i] - x2[i]) / norm + 1.0 - 0.5 * (x1[i] - x2[i]) * (x1[i] - x2[i]) / (norm * norm) );
		      }
		    else
		      {
			partialHessian(i, j) = exp(- norm / (2.0 * sigma_ * sigma_)) * (- 1.0 / (2.0 * sigma_ * sigma_)) * (x1[i] - x2[i]) * (x1[j] - x2[j]) / (norm * norm) * ( (- 1.0 / (2.0 * sigma_ * sigma_)) - 1.0 / norm );
		      }
		  }
	      }
	  }
	return partialHessian;
      }


    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
