//                                               -*- C++ -*-
/**
 *  @file  FunctionalChaosAlgorithm.cxx
 *  @brief The class building chaos expansions
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include <cstdlib>

#include "FunctionalChaosAlgorithm.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"
#include "Collection.hxx"
#include "NumericalMathFunctionImplementation.hxx"
#include "OrthogonalProductPolynomialFactory.hxx"
#include "RosenblattEvaluation.hxx"
#include "InverseRosenblattEvaluation.hxx"
#include "MarginalTransformationEvaluation.hxx"
#include "MarginalTransformationGradient.hxx"
#include "MarginalTransformationHessian.hxx"
#include "FixedStrategy.hxx"
#include "LeastSquaresStrategy.hxx"
#include "Exception.hxx"
#include "ResourceMap.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      CLASSNAMEINIT(FunctionalChaosAlgorithm);

      static Base::Common::Factory<FunctionalChaosAlgorithm> RegisteredFactory("FunctionalChaosAlgorithm");

      typedef Base::Common::NotYetImplementedException                 NotYetImplementedException;
      typedef Base::Common::InvalidArgumentException                   InvalidArgumentException;
      typedef Base::Common::ResourceMap                                ResourceMap;
      typedef Base::Func::NumericalMathFunctionImplementation          NumericalMathFunctionImplementation;
      typedef MarginalTransformationEvaluation::DistributionCollection DistributionCollection;
      const NumericalScalar FunctionalChaosAlgorithm::DefaultMaximumResidual = 1.0e-6;//strtod(ResourceMap::GetInstance().get("FunctionalChaosAlgorithm-DefaultMaximumResidual").c_str(), NULL);

      /* Default constructor */
      FunctionalChaosAlgorithm::FunctionalChaosAlgorithm()
	: Base::Common::PersistentObject(),
	  model_(),
	  distribution_(),
	  adaptiveStrategy_(FixedStrategy(OrthogonalProductPolynomialFactory(), 0)),
	  projectionStrategy_(LeastSquaresStrategy()),
	  maximumResidual_(DefaultMaximumResidual)
      {
	// Nothing to do
      }

      
      /* Constructor */
      FunctionalChaosAlgorithm::FunctionalChaosAlgorithm(const NumericalMathFunction & model,
							 const Distribution & distribution,
							 const AdaptiveStrategy & adaptiveStrategy,
							 const ProjectionStrategy & projectionStrategy)
	: Base::Common::PersistentObject(),
	  model_(model),
	  distribution_(distribution),
	  adaptiveStrategy_(adaptiveStrategy),
	  projectionStrategy_(projectionStrategy),
	  maximumResidual_(DefaultMaximumResidual)
      {
	// Nothing to do
      }


      /* Constructor */
      FunctionalChaosAlgorithm::FunctionalChaosAlgorithm(const NumericalMathFunction & model,
							 const Distribution & distribution,
							 const AdaptiveStrategy & adaptiveStrategy)
	: Base::Common::PersistentObject(),
	  model_(model),
	  distribution_(distribution),
	  adaptiveStrategy_(adaptiveStrategy),
	  projectionStrategy_(LeastSquaresStrategy()),
	  maximumResidual_(DefaultMaximumResidual)
      {
	// Nothing to do
      }


      /* Virtual constructor */
      FunctionalChaosAlgorithm * FunctionalChaosAlgorithm::clone() const
      {
	return new FunctionalChaosAlgorithm(*this);
      }


      /* String converter */
      String FunctionalChaosAlgorithm::__repr__() const
      {
	return OSS() << "class=" << getClassName();
      }


      /* Maximum residual accessors */
      void FunctionalChaosAlgorithm::setMaximumResidual(NumericalScalar residual)
      {
	maximumResidual_ = residual;
      }

      NumericalScalar FunctionalChaosAlgorithm::getMaximumResidual() const
      {
	return maximumResidual_;
      }
      
      /* Computes the functional chaos */
      void FunctionalChaosAlgorithm::run()
      {
	// Create the isoprobabilistic transformation
	// If we call Z a random vector which probability distribution is measure and X a random vector which probability distribution is distribution_, and T the isoprobabilistic transformation, we have:
	// Z = T(X)
	// X = T^{-1}(Z)
	// The transformation is built using the isoprobabilistic transformations of both X and Z
	// First, get the measure upon which the orthogonal basis is built
	const Distribution measure(adaptiveStrategy_.getImplementation()->basis_.getMeasure());
	// Check that it is a product measure
	if (!measure.hasIndependentCopula()) throw InvalidArgumentException(HERE) << "Error: cannot use FunctionalChaosAlgorithm with an orthogonal basis not based on a product measure";
	// Has the distribution an independent copula? Simply use marginal transformations
	if (distribution_.hasIndependentCopula())
	  {
	    const UnsignedLong dimension(distribution_.getDimension());
	    // We use empty collections to avoid the construction of default distributions
	    DistributionCollection marginalX(0);
	    DistributionCollection marginalZ(0);
	    for (UnsignedLong i = 0; i < dimension; ++i)
	      {
		marginalX.add(distribution_.getMarginal(i));
		marginalZ.add(measure.getMarginal(i));
	      }
	    // The distributions have an independent copula, they can be converted one into another by marginal transformation. T is such that T(X) = Z
	    const MarginalTransformationEvaluation evaluationT(MarginalTransformationEvaluation(marginalX, marginalZ));
	    const MarginalTransformationGradient gradientT(evaluationT);
	    const MarginalTransformationHessian hessianT(evaluationT);
	    transformation_ = NumericalMathFunction(evaluationT.clone(), gradientT.clone(), hessianT.clone());
	    const MarginalTransformationEvaluation evaluationTinv(MarginalTransformationEvaluation(marginalZ, marginalX));
	    const MarginalTransformationGradient gradientTinv(evaluationTinv);
	    const MarginalTransformationHessian hessianTinv(evaluationTinv);
	    inverseTransformation_ = NumericalMathFunction(evaluationTinv.clone(), gradientTinv.clone(), hessianTinv.clone());
	  }
	else
	  {
	    // Is the input distribution mapped into a normal standard space?
	    // We know that the standard space associated with Z is a normal space, as Z has an independent copula. We can check that X has the same standard space if its standard distribution has also an independent copula
	    if (distribution_.getStandardDistribution().hasIndependentCopula())
	      {
		// The distributions share the same standard space, it is thus possible to transform one into the other by composition between their isoprobabilistic transformations. T = T^{-1}_Z o T_X and T^{-1} = T^{-1}_X o T_Z
		const NumericalMathFunction TX(distribution_.getIsoProbabilisticTransformation());
		const NumericalMathFunction TinvX(distribution_.getInverseIsoProbabilisticTransformation());
		const NumericalMathFunction TZ(measure.getIsoProbabilisticTransformation());
		const NumericalMathFunction TinvZ(measure.getInverseIsoProbabilisticTransformation());
		transformation_ = NumericalMathFunction(TinvZ, TX);
		inverseTransformation_ = NumericalMathFunction(TinvX, TZ);
	      }
	    // The standard space is not normal, use Rosenblatt transformation instead of the native iso-probabilistic transformation.
	    else
	      {
		// The distributions share the same standard space if we use a Rosenblatt transforation, it is thus possible to transform one into the other by composition between their isoprobabilistic transformations. T = T^{-1}_Z o T_X and T^{-1} = T^{-1}_X o T_Z
		const NumericalMathFunction TX(NumericalMathFunctionImplementation(new RosenblattEvaluation(distribution_.getImplementation())));
		const NumericalMathFunction TinvX(NumericalMathFunctionImplementation(new InverseRosenblattEvaluation(distribution_.getImplementation())));
		const NumericalMathFunction TZ(measure.getIsoProbabilisticTransformation());
		const NumericalMathFunction TinvZ(measure.getInverseIsoProbabilisticTransformation());
		transformation_ = NumericalMathFunction(TinvZ, TX);
		inverseTransformation_ = NumericalMathFunction(TinvX, TZ);
	      }
	  } // Non-independent input copula
	// Build the composed model g = f o T^{-1}, which is a function of Z so it can be decomposed upon an orthonormal basis based on Z distribution
	composedModel_ = NumericalMathFunction(model_, inverseTransformation_);
	// Initialize the projection basis Phi_k_p_ and I_p_
	adaptiveStrategy_.computeInitialBasis();
	// Compute the associated coefficients alpha_k_p_ and the residual residual_p_
	// First, set the correct measure to the projection strategy
	projectionStrategy_.setMeasure(measure);
	projectionStrategy_.computeCoefficients(composedModel_, adaptiveStrategy_.getImplementation()->Psi_k_p_, adaptiveStrategy_.getImplementation()->I_p_, adaptiveStrategy_.getImplementation()->addedPsi_k_ranks_, adaptiveStrategy_.getImplementation()->conservedPsi_k_ranks_, adaptiveStrategy_.getImplementation()->removedPsi_k_ranks_);
	// The basis is adapted under the following conditions:
	// + the current residual is small enough
	// + the adaptive strategy has no more vector to propose
	if (projectionStrategy_.getImplementation()->residual_p_ < maximumResidual_)
	  {
	    result_ = FunctionalChaosResult(model_, distribution_, transformation_, inverseTransformation_, composedModel_, adaptiveStrategy_.getImplementation()->basis_, adaptiveStrategy_.getImplementation()->I_p_, projectionStrategy_.getImplementation()->alpha_k_p_, adaptiveStrategy_.getImplementation()->Psi_k_p_, projectionStrategy_.getImplementation()->residual_p_);
	    return;
	  }
	// Update the basis

	adaptiveStrategy_.updateBasis(projectionStrategy_.getImplementation()->alpha_k_p_, projectionStrategy_.getImplementation()->residual_p_);
	// Check if there is still something to do
	while ((adaptiveStrategy_.getImplementation()->addedPsi_k_ranks_.getSize() > 0) || (adaptiveStrategy_.getImplementation()->removedPsi_k_ranks_.getSize() > 0))
	  {
	    projectionStrategy_.computeCoefficients(composedModel_, adaptiveStrategy_.getImplementation()->Psi_k_p_, adaptiveStrategy_.getImplementation()->I_p_, adaptiveStrategy_.getImplementation()->addedPsi_k_ranks_, adaptiveStrategy_.getImplementation()->conservedPsi_k_ranks_, adaptiveStrategy_.getImplementation()->removedPsi_k_ranks_);
	    // The basis is adapted under the following conditions:
	    // + the current residual is small enough
	    // + the adaptive strategy has no more vector to propose
	    if (projectionStrategy_.getImplementation()->residual_p_ < maximumResidual_)
	      {
		result_ = FunctionalChaosResult(model_, distribution_, transformation_, inverseTransformation_, composedModel_, adaptiveStrategy_.getImplementation()->basis_, adaptiveStrategy_.getImplementation()->I_p_, projectionStrategy_.getImplementation()->alpha_k_p_, adaptiveStrategy_.getImplementation()->Psi_k_p_, projectionStrategy_.getImplementation()->residual_p_);
		return;
	      }
	    adaptiveStrategy_.updateBasis(projectionStrategy_.getImplementation()->alpha_k_p_, projectionStrategy_.getImplementation()->residual_p_);
	  }
	result_ = FunctionalChaosResult(model_, distribution_, transformation_, inverseTransformation_, composedModel_, adaptiveStrategy_.getImplementation()->basis_, adaptiveStrategy_.getImplementation()->I_p_, projectionStrategy_.getImplementation()->alpha_k_p_, adaptiveStrategy_.getImplementation()->Psi_k_p_, projectionStrategy_.getImplementation()->residual_p_);
      }


      /* Get the functional chaos result */
      FunctionalChaosResult FunctionalChaosAlgorithm::getResult() const
      {
	return result_;
      }


      /* Method save() stores the object through the StorageManager */
      void FunctionalChaosAlgorithm::save(StorageManager::Advocate & adv) const
      {
	PersistentObject::save(adv);
      }


      /* Method load() reloads the object from the StorageManager */
      void FunctionalChaosAlgorithm::load(StorageManager::Advocate & adv)
      {
	PersistentObject::load(adv);
      }



    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
