
//                                               -*- C++ -*-
/**
 *  @file  NatafEllipticalCopulaEvaluation.cxx
 *  @brief Class for the Nataf transformation evaluation for elliptical
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: NatafEllipticalCopulaEvaluation.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include "OTprivate.hxx"
#include "NatafEllipticalCopulaEvaluation.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      /*
       * @class NatafEllipticalCopulaEvaluation
       *
       * This class offers an interface for the Nataf function for elliptical copula
       */

      CLASSNAMEINIT(NatafEllipticalCopulaEvaluation);

      /* Parameter constructor */
      NatafEllipticalCopulaEvaluation::NatafEllipticalCopulaEvaluation(const Distribution & standardDistribution,
								       const SquareMatrix & inverseCholesky):
	NumericalMathEvaluationImplementation(),
	standardDistribution_(standardDistribution),
	inverseCholesky_(inverseCholesky)
      {
	Description description;
	for (UnsignedLong i = 0; i < inverseCholesky_.getNbColumns(); ++i) description.add(OSS() << "x" << i);
	for (UnsignedLong i = 0; i < inverseCholesky_.getNbRows(); ++i)	description.add(OSS() << "y" << i);
      }

      /* Virtual constructor */
      NatafEllipticalCopulaEvaluation * NatafEllipticalCopulaEvaluation::clone() const
      {
	return new NatafEllipticalCopulaEvaluation(*this);
      }

      /* String converter */
      String NatafEllipticalCopulaEvaluation::__repr__() const
      {
	OSS oss;
	oss << "class=" << NatafEllipticalCopulaEvaluation::GetClassName()
	    << " description=" << getDescription()
            << " standardDistribution=" << standardDistribution_
            << " inverseCholesky=" << inverseCholesky_;
                   
	return oss;
      }

      /*
       * Evaluation
       * This function transforms an elliptical copula with correlation matrix R into the associated standard elliptical distribution:
       * Yi(x) = Q(xi), where Q = F^{-1} and F is the CDF of the standard elliptical distribution
       * T(x) = G.Y(x), where G = L^{-1} and L is the Cholesky factor of R: L.L^t = R, L is lower triangular
       */
      NatafEllipticalCopulaEvaluation::NumericalPoint NatafEllipticalCopulaEvaluation::operator () (const NumericalPoint & in) const
	/* throw (InvalidArgumentException, InternalException) */
      {
	const UnsignedLong dimension(getOutputDimension());
	NumericalPoint result(dimension);
	const Distribution standardMarginal(standardDistribution_.getMarginal(0));
	// First, filter the commmon marginal distribution
	for (UnsignedLong i = 0; i < dimension; ++i) result[i] = standardMarginal.computeQuantile(in[i])[0];
	// Second, decorrelate the components
	return inverseCholesky_ * result;
      }

      /* Gradient according to the marginal parameters. Currently, the dependence parameters are not taken into account. */

      NatafEllipticalCopulaEvaluation::Matrix NatafEllipticalCopulaEvaluation::parametersGradient(const NumericalPoint & in) const
      {
	return Matrix(0, getInputDimension());
      }
	  
      /* Accessor for input point dimension */
      UnsignedLong NatafEllipticalCopulaEvaluation::getInputDimension() const
	/* throw(InternalException) */
      {
	return inverseCholesky_.getNbColumns();
      }

      /* Accessor for output point dimension */
      UnsignedLong NatafEllipticalCopulaEvaluation::getOutputDimension() const
	/* throw(InternalException) */
      {
	return inverseCholesky_.getNbRows();
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
