//                                               -*- C++ -*-
/**
 *  @file  NumericalSampleImplementation.hxx
 *  @brief The class NumericalSampleImplementation implements blank free samples
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: NumericalSampleImplementation.hxx 1473 2010-02-04 15:44:49Z dutka $
 */
#ifndef OPENTURNS_NUMERICALSAMPLEIMPLEMENTATION_HXX
#define OPENTURNS_NUMERICALSAMPLEIMPLEMENTATION_HXX

#include "NumericalPoint.hxx"
#include "Description.hxx"
#include "Indices.hxx"
#include "PersistentCollection.hxx"
#include "SquareMatrix.hxx"
#include "CovarianceMatrix.hxx"
#include "CorrelationMatrix.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Stat
    {

      /**
       * @class NumericalSampleImplementation
       */

      class NumericalSampleImplementation
	: public Type::PersistentCollection<Type::NumericalPoint>
      {
	CLASSNAME;
	friend class Common::Factory<NumericalSampleImplementation>;

	typedef Type::PersistentCollection<Type::NumericalPoint> InternalType;

      public:

	typedef InternalType::iterator               iterator;
	typedef InternalType::const_iterator         const_iterator;
	typedef InternalType::reverse_iterator       reverse_iterator;
	typedef InternalType::const_reverse_iterator const_reverse_iterator;

      public:

	/* Some typedefs for easy reading */
	typedef Type::NumericalPoint             NumericalPoint;
	typedef Type::Description                Description;
	typedef Type::Indices                    Indices;
	typedef Common::FileNotFoundException    FileNotFoundException;
	typedef Common::InvalidArgumentException InvalidArgumentException;
	typedef Common::InternalException        InternalException;
	typedef Stat::CovarianceMatrix           CovarianceMatrix;
	typedef Stat::CorrelationMatrix          CorrelationMatrix;
	typedef Type::SquareMatrix               SquareMatrix;
	typedef Common::StorageManager           StorageManager;
	typedef Common::PersistentObject         PersistentObject;

	enum ExternalFileFormat { CSV };

      public:

	/** Constructor from file */
	static NumericalSampleImplementation GetFromFile(const FileName & fileName, const ExternalFileFormat format = CSV);

	//private:
	friend class BuildMethodMap;

	//private:
	friend class ExportMethodMap;

	/** Factory of NumericalSampleImplementation from CSV file */
	static NumericalSampleImplementation BuildFromCSVFile(const FileName & fileName) /* throw(FileNotFoundException, InternalException) */;

	/** Store a sample in a temporary text file, one realization by line. Returns the file name. */
	String storeToTemporaryFile() const;

	/** Export a sample as a matrix, one row by realization, in a format suitable to exchange with R */
	String streamToRFormat() const;

      protected:

	/** 
	 * Default constructor is protected
	 */
	NumericalSampleImplementation();

      public:

	/** Standard constructor */
	NumericalSampleImplementation(UnsignedLong size,
				      UnsignedLong dim);

	/** Constructor from a NumericalPoint */
	NumericalSampleImplementation(UnsignedLong size,
				      const NumericalPoint & point);

#ifndef SWIG
	/** Partial copy constructor */
	NumericalSampleImplementation(const NumericalSampleImplementation & other,
                                      iterator first,
                                      iterator last);
#endif

	/** Virtual constructor */
	NumericalSampleImplementation * clone() const;

	/** Description Accessor */
	void setDescription(const Description & description);
	Description getDescription() const;

	/** Comparison operator */
	Bool operator ==(const NumericalSampleImplementation & other) const;

	/**
	 * String converter
	 * This method shows human readable information on the
	 * internal state of an NumericalSampleImplementation. It is used when streaming
	 * the NumericalSampleImplementation or for user information.
	 */
	virtual String __repr__() const;
	virtual String __str__() const;

	/** Dimension accessor */
	UnsignedLong getDimension() const;

	/** Appends an element to the collection */
	void add(const NumericalPoint & point) /* throw(InvalidArgumentException) */;

	/** 
	 * Gives the mean of the sample, based on the formula
         * mean = sum of the elements in the sample / size of the sample
         */
	virtual NumericalPoint computeMean() const;

	/** 
	 * Gives the covariance matrix of the sample, normalization by 1 / (size - 1) if size > 1
         */
	virtual CovarianceMatrix computeCovariance() const;

	/**
	 * Gives the standard deviation of the sample, i.e. the square-root of the covariance matrix.
	 */
	SquareMatrix computeStandardDeviation() const;

	/**
	 * Gives the standard deviation of each component of the sample
	 */
	virtual NumericalPoint computeStandardDeviationPerComponent() const;

	/**
	 * Gives the Pearson correlation matrix of the sample
	 */
	virtual CorrelationMatrix computePearsonCorrelation() const;

	/**
	 * Gives the Spearman correlation matrix of the sample
	 */
	virtual CorrelationMatrix computeSpearmanCorrelation() const;

	/**
	 * Gives the Kendall correlation matrix of the sample
	 */
	virtual CorrelationMatrix computeKendallTau() const;

	/**
	 * Gives the range of the sample (by component)
	 */
	NumericalPoint computeRangePerComponent() const;
	
	/**
	 * Gives the median of the sample (by component)
	 */
	NumericalPoint computeMedianPerComponent() const;
	
	/**
	 * Gives the variance of the sample (by component)
	 */
	virtual NumericalPoint computeVariancePerComponent() const;
	
	/**
	 * Gives the skewness of the sample (by component)
	 */
	virtual NumericalPoint computeSkewnessPerComponent() const;
	
	/**
	 * Gives the kurtosis of the sample (by component)
	 */
	virtual NumericalPoint computeKurtosisPerComponent() const;
	
	/**
	 * Gives the centered moment of order k of the sample (by component)
	 */
	NumericalPoint computeCenteredMomentPerComponent(const UnsignedLong k) const;
	
	/**
	 * Gives the quantile per component of the sample
	 */
	virtual NumericalPoint computeQuantilePerComponent(const NumericalScalar prob) const;

	/**
	 * Gives the N-dimension quantile of the sample
	 */
	NumericalPoint computeQuantile(const NumericalScalar prob) const;

	/**
	 * Get the empirical CDF of the sample
	 */
	NumericalScalar computeEmpiricalCDF(const NumericalPoint & point) const;

	/** Maximum accessor */
	virtual NumericalPoint getMax() const;

	/** Minimum accessor */
	virtual NumericalPoint getMin() const;

	/** Ranked sample */
	NumericalSampleImplementation rank() const;

	/** Ranked component */
	NumericalSampleImplementation rank(const UnsignedLong index) const;

	/** Sorted sample */
	NumericalSampleImplementation sort() const;

	/** Sorted component */
	NumericalSampleImplementation sort(const UnsignedLong index) const;

	/** Sorted component */
	NumericalSampleImplementation sortAccordingToAComponent(const UnsignedLong index) const;

	/** Get the i-th marginal sample */
	NumericalSampleImplementation getMarginal(const UnsignedLong index) const /* throw(InvalidArgumentException) */;

	/** Get the marginal sample corresponding to indices dimensions */
	NumericalSampleImplementation getMarginal(const Indices & indices) const /* throw(InvalidArgumentException) */;

	/**
	 * Translate realizations in-place
	 */
	void translate(const NumericalPoint & translation);

	/**
	 * Scale realizations componentwise in-place
	 */
	void scale(const NumericalPoint & scaling);

	/** Save to CSV file */
	void exportToCSVFile(const FileName & filename,
			     const Bool withDescription = false) const;

 	/** Method save() stores the object through the StorageManager */
	virtual void save(StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	virtual void load(StorageManager::Advocate & adv);

      protected:


      private:

	/** The dimension of the sample */
	UnsignedLong dimension_;

 	/** The description of all components */
	Description::Implementation p_description_;

      }; /* class NumericalSampleImplementation */


    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_NUMERICALSAMPLEIMPLEMENTATION_HXX */
