//                                               -*- C++ -*-
/**
 * @file  AnalyticalNumericalMathEvaluationImplementation.cxx
 * @brief The class that implements the composition between numerical
 *        math functions implementations
 *
 * (C) Copyright 2005-2010 EADS
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * \author $LastChangedBy: dutka $
 * \date   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 */

#include "AnalyticalNumericalMathEvaluationImplementation.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      CLASSNAMEINIT(AnalyticalNumericalMathEvaluationImplementation);

      static Common::Factory<AnalyticalNumericalMathEvaluationImplementation> RegisteredFactory("AnalyticalNumericalMathEvaluationImplementation");

      /* Default constructor */
      AnalyticalNumericalMathEvaluationImplementation::AnalyticalNumericalMathEvaluationImplementation()
	: NumericalMathEvaluationImplementation(),
	  inputVariables_(0),
	  isInitialized_(false),
	  inputVariablesNames_(),
	  outputVariablesNames_(),
	  formulas_(),
	  parsers_(0)
      {
	// Nothing to do
      } // AnalyticalNumericalMathEvaluationImplementation

      /* Default constructor */
      AnalyticalNumericalMathEvaluationImplementation::AnalyticalNumericalMathEvaluationImplementation(const Description & inputVariablesNames,
												       const Description & outputVariablesNames,
												       const Description & formulas)
	/* throw(InvalidArgumentException) */
	: NumericalMathEvaluationImplementation(),
	  inputVariables_(NumericalPoint(inputVariablesNames.getSize()).getCollection()),
	  isInitialized_(false),
	  inputVariablesNames_(inputVariablesNames),
	  outputVariablesNames_(outputVariablesNames),
	  formulas_(formulas),
	  parsers_(ParserCollection(outputVariablesNames_.getSize()))
      {
        Description description(inputVariablesNames_);
	try
	  {
	    const UnsignedLong outputSize(outputVariablesNames_.getSize());
	    const UnsignedLong inputSize(inputVariablesNames_.getSize());
	    // For each parser of a formula, do
	    for (UnsignedLong outputVariableIndex = 0; outputVariableIndex < outputSize; ++outputVariableIndex)
	      {
		// First, define all the variable names and associate them
		// to the corresponding component of the input vector
		for(UnsignedLong inputVariableIndex = 0; inputVariableIndex < inputSize; ++inputVariableIndex)
		  {
		    parsers_[outputVariableIndex].DefineVar(inputVariablesNames_[inputVariableIndex].c_str(), &inputVariables_[inputVariableIndex]);
		  }
		// Second, define the several formulas
		parsers_[outputVariableIndex].SetExpr(formulas_[outputVariableIndex].c_str());
		// Third, add the current output variable name to the description
		description.add(outputVariablesNames_[outputVariableIndex]);
	      }
	    // Set the description
	    setDescription(description);
	  }
	catch(mu::Parser::exception_type & ex)
	  {
	    throw InvalidArgumentException(HERE) << "Error constructing an analytical function, message=" << ex.GetMsg() << " formula=" << ex.GetExpr() << " token=" << ex.GetToken() << " position=" << ex.GetPos();
	  }
      } // AnalyticalNumericalMathEvaluationImplementation


      /* Virtual constructor */
      AnalyticalNumericalMathEvaluationImplementation * AnalyticalNumericalMathEvaluationImplementation::clone() const
      {
	return new AnalyticalNumericalMathEvaluationImplementation(*this);
      }


      /* Comparison operator */
      Bool AnalyticalNumericalMathEvaluationImplementation::operator ==(const AnalyticalNumericalMathEvaluationImplementation & other) const
      {
	return true;
      }
  
      /* String converter */
      String AnalyticalNumericalMathEvaluationImplementation::__repr__() const {
	OSS oss;
	oss << "class=" << AnalyticalNumericalMathEvaluationImplementation::GetClassName()
	    << " name=" << getName()
	    << " inputVariablesNames=" << inputVariablesNames_
	    << " outputVariablesNames=" << outputVariablesNames_
	    << " formulas=" << formulas_;
	return oss;
      }
  

      /* Check if the muParser wrappers have already been initialized */
      void AnalyticalNumericalMathEvaluationImplementation::checkInitialization() const
      {
	if (isInitialized_) return;
	const UnsignedLong outputSize(outputVariablesNames_.getSize());
	const UnsignedLong inputSize(inputVariablesNames_.getSize());
	// For each parser of a formula, do
	for (UnsignedLong outputVariableIndex = 0; outputVariableIndex < outputSize; ++outputVariableIndex)
	  {
	    // First, define all the variable names and associate them
	    // to the corresponding component of the input vector
	    for(UnsignedLong inputVariableIndex = 0; inputVariableIndex < inputSize; ++inputVariableIndex)
	      {
		parsers_[outputVariableIndex].DefineVar(inputVariablesNames_[inputVariableIndex].c_str(), &inputVariables_[inputVariableIndex]);
	      }
	    // Second, define the several formulas
	    parsers_[outputVariableIndex].SetExpr(formulas_[outputVariableIndex].c_str());
	  }
	isInitialized_ = true;
      }

      /* Operator () */
      AnalyticalNumericalMathEvaluationImplementation::NumericalPoint AnalyticalNumericalMathEvaluationImplementation::operator() (const NumericalPoint & in) const
	/* throw(InvalidArgumentException, InternalException) */
      {
	if (in.getDimension() != getInputDimension()) throw InvalidArgumentException(HERE) << "Error: trying to evaluate a NumericalMathFunction with an argument of invalid dimension";
	NumericalPoint out(getOutputDimension());
	checkInitialization();
	++callsNumber_;
	for (UnsignedLong i = 0; i < in.getDimension(); ++i) inputVariables_[i] = in[i];
	try
	  {
	    for (UnsignedLong index = 0; index < out.getDimension(); ++index) out[index] = parsers_[index].Eval();
	  }
	catch(mu::Parser::exception_type & ex)
	  {
	    throw InternalException(HERE) << ex.GetMsg();
	  }
	return out;
      }

      /* Accessor for input point dimension */
      UnsignedLong AnalyticalNumericalMathEvaluationImplementation::getInputDimension() const
	/* throw(InternalException) */
      {
	return inputVariablesNames_.getSize();
      }
      
      /* Accessor for output point dimension */
      UnsignedLong AnalyticalNumericalMathEvaluationImplementation::getOutputDimension() const
	/* throw(InternalException) */
      {
	return outputVariablesNames_.getSize();
      }
     
      /* Method save() stores the object through the StorageManager */
      void AnalyticalNumericalMathEvaluationImplementation::save(StorageManager::Advocate & adv) const
      {
	NumericalMathEvaluationImplementation::save(adv);
	adv.saveAttribute( "inputVariablesNames_", inputVariablesNames_ );
	adv.saveAttribute( "outputVariablesNames_", outputVariablesNames_ );
	adv.saveAttribute( "formulas_", formulas_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void AnalyticalNumericalMathEvaluationImplementation::load(StorageManager::Advocate & adv)
      {
	NumericalMathEvaluationImplementation::load(adv);
	adv.loadAttribute( "inputVariablesNames_", inputVariablesNames_ );
	adv.loadAttribute( "outputVariablesNames_", outputVariablesNames_ );
	adv.loadAttribute( "formulas_", formulas_ );
      }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
