//                                               -*- C++ -*-
/**
 *  @file  PersistentObjectFactory.hxx
 *  @brief Class PersistentObjectFactory reloads a PersistentObject from a storage manager
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: PersistentObjectFactory.hxx 1473 2010-02-04 15:44:49Z dutka $
 */
#ifndef OPENTURNS_PERSISTENTOBJECTFACTORY_HXX
#define OPENTURNS_PERSISTENTOBJECTFACTORY_HXX

#include "PersistentObject.hxx"
#include "StorageManager.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {

      /**
       * @class PersistentObjectFactory
       *
       * @brief Reloads a PersistentObject from a storage manager
       * @see PersistentObject
       * @see StorageManager
       */

      class PersistentObjectFactory
      {
      public:

	typedef Common::StorageManager StorageManager;

	/** @copydoc PersistentObject::clone() const */
	virtual PersistentObjectFactory * clone() const = 0;

	/** Method build() creates a new PersistentObject from the storage manager
	 * @param mgr The storage manager that points to the object to be reloaded
	 * @return A pointer to a newly allocated object
	 * @internal
	 */
	virtual PersistentObject * build(StorageManager & mgr) const = 0;

	/** Method assign() fills a PersistentObject with another one (must have same type)
	 *
	 * This method performs an assignment between the two objects after having casted them to the same type
	 * @param po The persistent object to be refilled (may be empty, ie default constructed)
	 * @param other The persistent object of the same type as \em po that should be copied to \em po
	 * @internal
	 */
	virtual void assign(PersistentObject & po, const PersistentObject & other) const = 0;

	/** This method register the factory into the Catalog
	 * @see Catalog
	 * @internal
	 */
	void registerMe(const String & className) const;

	/** Accessor to PersistentObject's shadowed id */
	void setShadowedId(PersistentObject & obj, Id id) const;
	Id getShadowedId(PersistentObject & obj) const;

      protected:

      private:

      }; /* class PersistentObjectFactory */


      /**
       * @class Factory
       * @brief Eease the creation of factories for any PersistentObject based class
       * @tparam PERSISTENT The class which the %Factory is bounded to
       */
      template <class PERSISTENT>
      class Factory
	: PersistentObjectFactory
      {
      public:
	/** Constructor
	 *
	 * @param className The name of the class the %Factory is bounded to
	 * @internal See the CLASSNAME_INIT macro.
	 */
	Factory(const String & className)
	{
	  registerMe(className);
	}

	/** Virtual constructor */
	virtual Factory * clone() const
	{
	  return new Factory(*this);
	}

	/** Method build() creates a new PersistentObject from the storage manager */
	virtual PersistentObject * build(StorageManager & mgr) const
	{
	  StorageManager::Advocate adv ( mgr.readObject() );
	  
	  PERSISTENT * p_rebuildObject = new PERSISTENT();
	  assert(p_rebuildObject && "PersistentObject not allocated");
	  //try {

	  p_rebuildObject->load(adv);

	  //} catch (Exception & ex) {
	  // Nothing to do
	  // The rebuild object remains the default one
	  //}
	  return p_rebuildObject;
	}
	
	/** Method assign() fills a PersistentObject with another one (must have same type) */
	virtual void assign(PersistentObject & po, const PersistentObject & other) const
	{
	  PERSISTENT & ref_po          = static_cast<PERSISTENT &>(po);
	  const PERSISTENT & ref_other = static_cast<const PERSISTENT &>(other);
	  ref_po = ref_other;
	}

      }; /* end class Factory */


    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_PERSISTENTOBJECTFACTORY_HXX */
