//                                               -*- C++ -*-
/**
 *  @file  UsualRandomVector.cxx
 *  @brief An implementation class for distribution-based random vectors
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: UsualRandomVector.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include "PersistentObjectFactory.hxx"
#include "UsualRandomVector.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Model {

      typedef Base::Common::NotYetImplementedException NotYetImplementedException;

      CLASSNAMEINIT(UsualRandomVector);

      static Base::Common::Factory<UsualRandomVector> RegisteredFactory("UsualRandomVector");

      /* Default constructor */
      UsualRandomVector::UsualRandomVector(const Distribution & distribution, const String & name)
	: RandomVectorImplementation(name),
	  distribution_(distribution)
      {
	// Initialize any other class members here
	// At last, allocate memory space if needed, but go to destructor to free it
	setDescription(distribution.getDescription());
      }

      /* Virtual constructor */
      UsualRandomVector * UsualRandomVector::clone() const
      {
	return new UsualRandomVector(*this);
      }

      /* String converter */
      String UsualRandomVector::__repr__() const {
	OSS oss;
	oss << "class=" << UsualRandomVector::GetClassName()
	    << " distribution=" << distribution_;
	return oss;
      }
  


      /* Here is the interface that all derived class must implement */


      /* Dimension accessor */
      UnsignedLong UsualRandomVector::getDimension() const
      {
	return distribution_.getDimension();
      }

      /* Realization accessor */
      UsualRandomVector::NumericalPoint UsualRandomVector::getRealization() const
      {
	return distribution_.getRealization();
      }

      /* Numerical sample accessor */
      UsualRandomVector::NumericalSample UsualRandomVector::getNumericalSample(UnsignedLong size) const
      {
	return distribution_.getNumericalSample(size);
      }

      /* Mean accessor */
      UsualRandomVector::NumericalPoint UsualRandomVector::getMean() const
      {
	return distribution_.getMean();
      }

      /* Covariance accessor */
      UsualRandomVector::CovarianceMatrix UsualRandomVector::getCovariance() const
      {
	return distribution_.getCovariance();
      }

      /* Get the random vector corresponding to the i-th marginal component */
      RandomVectorImplementation::Implementation UsualRandomVector::getMarginal(const UnsignedLong i) const throw(InvalidArgumentException)
      {
	if (i >= getDimension()) throw InvalidArgumentException(HERE) << "The index of a marginal random vector must be in the range [0, dim-1]";
	return new UsualRandomVector(distribution_.getMarginal(i));
      }

      /* Get the marginal random vector corresponding to indices components */
      RandomVectorImplementation::Implementation UsualRandomVector::getMarginal(const Indices & indices) const throw(InvalidArgumentException)
      {
	if (!indices.check(getDimension() - 1)) throw InvalidArgumentException(HERE) << "The indices of a marginal random vector must be in the range [0, dim-1] and  must be different";
	return new UsualRandomVector(distribution_.getMarginal(indices));
      }

      /* Distribution accessor */
      Distribution UsualRandomVector::getDistribution() const
      {
	return distribution_;
      }

      /* Method save() stores the object through the StorageManager */
      void UsualRandomVector::save(const StorageManager::Advocate & adv) const
      {
	RandomVectorImplementation::save(adv);
	adv.writeValue(distribution_, StorageManager::MemberNameAttribute, "distribution_");
      }

      /* Method load() reloads the object from the StorageManager */
      void UsualRandomVector::load(const StorageManager::Advocate & adv)
      {
	RandomVectorImplementation::load(adv);
	adv.readValue(distribution_, StorageManager::MemberNameAttribute, "distribution_");
      }

    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
