//                                               -*- C++ -*-
/**
 *  @file  SklarCopula.cxx
 *  @brief The SklarCopula distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2008-09-09 23:26:29 +0200 (mar, 09 sep 2008) $
 *  Id:      $Id: SklarCopula.cxx 924 2008-09-09 21:26:29Z lebrun $
 */
#include "SklarCopula.hxx"
#include "PersistentObjectFactory.hxx"
#include "Exception.hxx"
#include "MarginalTransformationEvaluation.hxx"
#include "MarginalTransformationGradient.hxx"
#include "MarginalTransformationHessian.hxx"
#include "InverseMarginalTransformationEvaluation.hxx"
#include "InverseMarginalTransformationGradient.hxx"
#include "InverseMarginalTransformationHessian.hxx"
#include "RosenblattEvaluation.hxx"
#include "InverseRosenblattEvaluation.hxx"
#include "NumericalMathFunctionImplementation.hxx"
#include "RandomGenerator.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Model {

      typedef Base::Common::NotYetImplementedException                      NotYetImplementedException;
      typedef Base::Stat::RandomGenerator                                   RandomGenerator;
      typedef Algorithm::MarginalTransformationEvaluation                   MarginalTransformationEvaluation;
      typedef Algorithm::MarginalTransformationGradient                     MarginalTransformationGradient;
      typedef Algorithm::MarginalTransformationHessian                      MarginalTransformationHessian;
      typedef Algorithm::InverseMarginalTransformationEvaluation            InverseMarginalTransformationEvaluation;
      typedef Algorithm::InverseMarginalTransformationGradient              InverseMarginalTransformationGradient;
      typedef Algorithm::InverseMarginalTransformationHessian               InverseMarginalTransformationHessian;
      typedef Algorithm::RosenblattEvaluation                               RosenblattEvaluation;
      typedef Algorithm::InverseRosenblattEvaluation                        InverseRosenblattEvaluation;
      typedef Base::Func::NumericalMathFunctionImplementation               NumericalMathFunctionImplementation;
      typedef NumericalMathFunctionImplementation::EvaluationImplementation EvaluationImplementation;
      typedef NumericalMathFunctionImplementation::GradientImplementation   GradientImplementation;
      typedef NumericalMathFunctionImplementation::HessianImplementation    HessianImplementation;

      CLASSNAMEINIT(SklarCopula);

      static Base::Common::Factory<SklarCopula> RegisteredFactory("SklarCopula");

      /* Default constructor */
      SklarCopula::SklarCopula()
	: CopulaImplementation("SklarCopula"),
	  distribution_(),
	  marginalCollection_()
      {
	setDimension( 0 );
      }

      /* Parameters constructor */
      SklarCopula::SklarCopula(const Distribution distribution)
	throw(InvalidArgumentException)
	: CopulaImplementation("SklarCopula"),
	  distribution_(distribution),
	  marginalCollection_(distribution.getDimension())
      {
	// We set the dimension of the SklarCopula distribution
	const UnsignedLong dimension(distribution.getDimension());
	setDimension( dimension );
	// Extract all the 1D marginal distributions
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    marginalCollection_[i] = distribution.getMarginal(i);
	  }
      }

      /* Comparison operator */
      Bool SklarCopula::operator ==(const SklarCopula & other) const
      {
	Bool sameObject = false;

	if (this != &other) { // Other is NOT me, so I have to realize the comparison
	  if ( (distribution_ == other.distribution_) )
	    sameObject = true;

	} else sameObject = true;

	return sameObject;
      }
  
      /* String converter */
      String SklarCopula::__repr__() const
      {
	OSS oss;
	oss << "class=" << SklarCopula::GetClassName()
	    << " name=" << getName()
	    << " dimension=" << getDimension()
	    << " distribution=" << distribution_;
	return oss;
      }
  
      /* Virtual constructor */
      SklarCopula * SklarCopula::clone() const
      {
	return new SklarCopula(*this);
      }

      /* Get one realization of the distribution
	 F(x_1,\dots,x_n) = C(F_1(x_1),\dots,F_n(x_n))
	 so a realization of C is a realization of F marginaly composed with F_i */
      SklarCopula::NumericalPoint SklarCopula::getRealization() const
      {
	const UnsignedLong dimension(getDimension());
	// Special case for independent copula, for improved performance
	if (hasIndependentCopula()) return RandomGenerator::Generate(dimension);
	NumericalPoint realization(distribution_.getRealization());
	for (UnsignedLong i = 0; i < dimension; ++i) realization[i] = marginalCollection_[i].computeCDF(realization[i]);
	return realization;
      }

      /* Get the DDF of the distribution
	 F(x_1,\dots,x_n) = C(u_1,\dots,u_n)
	 where u_i = F_i(x_i)
	 so p(x_1,\dots,x_n) = c(u_1,\dots,u_n)\prod_{i=1}^n p_i(x_i)
	 and dp/dx_k(x_1,\dots,x_n) = [dc/du_k(u_1,\dots,u_n)p_k(x_k) +
	 p'_k(x_k)/p_k(x_k)c(u_1,\dots,u_n)]\prod_{i=1}^n p_i(x_i)
	 but c(u_1,\dots,u_n) = p(x_1,\dots,u_n)/\prod_{i=1}^n p_i(x_i)
	 so dp/dx_k(x_1,\dots,x_n) = dc/du_k(u_1,\dots,u_n)p_k(x_k)\prod_{i=1}^n p_i(x_i) +
	 p'_k(x_k)/p_k(x_k)p(x_1,\dots,x_n)
      */
      SklarCopula::NumericalPoint SklarCopula::computeDDF(const NumericalPoint & point) const
      {
	const UnsignedLong dimension(getDimension());
	NumericalPoint x(point);
	NumericalPoint pdfX(dimension);
	NumericalPoint ddfX(dimension);
	NumericalScalar factor(1.0);
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    const NumericalScalar ui(point[i]);
	    if ((ui <= 0.0) || ui >= 1.0) return NumericalPoint(dimension, 0.0);
	    const NumericalPoint xi(marginalCollection_[i].computeQuantile(ui));
	    x[i] = xi[0];
	    pdfX[i] = marginalCollection_[i].computePDF(xi);
	    ddfX[i] = marginalCollection_[i].computeDDF(xi)[0];
	    factor *= pdfX[i];
	    if (factor == 0.0) return NumericalPoint(dimension, 0.0);
	  }
	const NumericalScalar pdfDistribution(distribution_.computePDF(x));
	NumericalPoint result(distribution_.computeDDF(x));
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    result[i] -= ddfX[i] * pdfDistribution / pdfX[i];
	  }
	return result * (1.0 / factor);
      }

      /* Get the PDF of the distribution
	 F(x_1,\dots,x_n) = C(F_1(x_1),\dots,F_n(x_n))
	 so p(x_1,\dots,x_n) = c(F_1(x_1),\dots,F_n(x_n))\prod_{i=1}^n p_i(x_i) */
      NumericalScalar SklarCopula::computePDF(const NumericalPoint & point) const
      {
	const UnsignedLong dimension(getDimension());
	NumericalPoint x(point);
	NumericalScalar factor(1.0);
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    const NumericalScalar ui(point[i]);
	    if ((ui <= 0.0) || ui >= 1.0) return 0.0;
	    const NumericalPoint xi(marginalCollection_[i].computeQuantile(ui));
	    x[i] = xi[0];
	    factor *= marginalCollection_[i].computePDF(xi);
	    if (factor == 0.0) return 0.0;
	  }
	return distribution_.computePDF(x) / factor;
      }

      /* Get the CDF of the distribution
	 F(x_1,\dots,x_n) = C(F_1(x_1),\dots,F_n(x_n)) */
      NumericalScalar SklarCopula::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
	const UnsignedLong dimension(getDimension());
	NumericalPoint x(point);
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    const NumericalScalar ui(std::min(point[i], 1.0));
	    if (ui <= 0.0) return 0.0;
	    x[i] = marginalCollection_[i].computeQuantile(point[i])[0];
	  }
	return distribution_.computeCDF(x);
      }

      /* Compute the probability content of an interval */
      NumericalScalar SklarCopula::computeProbability(const Interval & interval) const
      {
	const UnsignedLong dimension(getDimension());
	if (interval.getDimension() != dimension) throw InvalidArgumentException(HERE) << "Error: the given interval has a dimension not compatible with the distribution dimension";
	// Reduce the given interval to the support of the distribution, which is the nD unit cube
	const Interval intersect(interval.intersect(Interval(dimension)));
	// If the intersection is empty
	if (intersect.isNumericallyEmpty()) return 0.0;
	const NumericalPoint lowerBoundIntersect(intersect.getLowerBound());
	const NumericalPoint upperBoundIntersect(intersect.getUpperBound());
	NumericalPoint lowerBound(dimension);
	NumericalPoint upperBound(dimension);
	Interval::BoolCollection finiteLowerBound(dimension);
	Interval::BoolCollection finiteUpperBound(dimension);
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    if (lowerBoundIntersect[i] == 0.0)
	      {
		finiteLowerBound[i] = false;
		lowerBound[i] = -1.0;
	      }
	    else
	      {
		finiteLowerBound[i] = true;
		lowerBound[i] = marginalCollection_[i].computeQuantile(lowerBoundIntersect[i])[0];
	      }
	    if (upperBoundIntersect[i] == 1.0)
	      {
		finiteUpperBound[i] = false;
		upperBound[i] = 1.0;
	      }
	    else
	      {
		finiteUpperBound[i] = true;
		upperBound[i] = marginalCollection_[i].computeQuantile(upperBoundIntersect[i])[0];
	      }
	  }
	return distribution_.computeProbability(Interval(lowerBound, upperBound, finiteLowerBound, finiteUpperBound));
      }

      /* Get the PDFGradient of the distribution */
      SklarCopula::NumericalPoint SklarCopula::computePDFGradient(const NumericalPoint & point) const
      {
	throw Base::Common::NotYetImplementedException(HERE);
      }

      /* Get the CDFGradient of the distribution */
      SklarCopula::NumericalPoint SklarCopula::computeCDFGradient(const NumericalPoint & point) const
      {
	throw Base::Common::NotYetImplementedException(HERE);
      }

      /* Get the quantile of the distribution
	 F(x_1,\dots,x_n) = C(F_1(x_1),\dots,F_n(x_n)) */
      SklarCopula::NumericalPoint SklarCopula::computeQuantile(const NumericalScalar prob,
							       const Bool tail) const
      {
	const UnsignedLong dimension(getDimension());
	NumericalPoint uq(distribution_.computeQuantile(prob));
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    uq[i] = marginalCollection_[i].computeCDF(uq[i]);
	  }
	return uq;
      }

      /* Compute the PDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
      NumericalScalar SklarCopula::computeConditionalPDF(const NumericalScalar x, const NumericalPoint & y) const
      {
	const UnsignedLong conditioningDimension(y.getDimension());
	if (conditioningDimension >= getDimension()) throw InvalidArgumentException(HERE) << "Error: cannot compute a conditional PDF with a conditioning point of dimension greater or equal to the distribution dimension.";
	// Special case for no conditioning or independent copula
	if ((conditioningDimension == 0) || (hasIndependentCopula())) return getMarginal(conditioningDimension)->computePDF(x);
	// General case
	NumericalPoint u(conditioningDimension);
	for (UnsignedLong i = 0; i < conditioningDimension; ++i)
	  {
	    u[i] = marginalCollection_[i].computeQuantile(y[i])[0];
	  }
	const NumericalScalar ux(marginalCollection_[conditioningDimension].computeQuantile(x)[0]);
	const NumericalScalar pdf(marginalCollection_[conditioningDimension].computePDF(ux));
	if (pdf == 0.0) return 0.0;
	return distribution_.computeConditionalPDF(ux, u) / pdf;
      }

      /* Compute the CDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
      NumericalScalar SklarCopula::computeConditionalCDF(const NumericalScalar x, const NumericalPoint & y) const
      {
	const UnsignedLong conditioningDimension(y.getDimension());
	if (conditioningDimension >= getDimension()) throw InvalidArgumentException(HERE) << "Error: cannot compute a conditional CDF with a conditioning point of dimension greater or equal to the distribution dimension.";
	// Special case for no conditioning or independent copula
	if ((conditioningDimension == 0) || (hasIndependentCopula())) return marginalCollection_[conditioningDimension].computeQuantile(x)[0];
	// General case
	NumericalPoint u(conditioningDimension);
	for (UnsignedLong i = 0; i < conditioningDimension; ++i)
	  {
	    u[i] = marginalCollection_[i].computeQuantile(y[i])[0];
	  }
	return distribution_.computeConditionalCDF(marginalCollection_[conditioningDimension].computeQuantile(x)[0], u);
      }

      /* Get the i-th marginal distribution */
      SklarCopula::Implementation SklarCopula::getMarginal(const UnsignedLong i) const throw(InvalidArgumentException)
      {
	if (i >= getDimension()) throw InvalidArgumentException(HERE) << "The index of a marginal distribution must be in the range [0, dim-1]";
	return marginalCollection_[i].getImplementation();
      }

      /* Get the distribution of the marginal distribution corresponding to indices dimensions */
      SklarCopula::Implementation SklarCopula::getMarginal(const Indices & indices) const throw(InvalidArgumentException)
      {
	// This call will check that indices are correct
	return new SklarCopula(distribution_.getMarginal(indices));
      }

      /* Get the isoprobabilist transformation */
      SklarCopula::IsoProbabilisticTransformation SklarCopula::getIsoProbabilisticTransformation() const
      {
	// Special case for distributions with elliptical copula
	if (distribution_.getImplementation()->hasEllipticalCopula())
	  {
	    // Get the IsoProbabilisticTransformation from the copula
	    IsoProbabilisticTransformation isoprobabilistic(distribution_.getIsoProbabilisticTransformation());
	    // Get the right function implementations
	    EvaluationImplementation p_rightFunction = new InverseMarginalTransformationEvaluation(marginalCollection_);
	    // Get the right gradient implementations
	    GradientImplementation p_rightGradient = new InverseMarginalTransformationGradient(marginalCollection_);
	    // Get the right hessian implementations
	    HessianImplementation p_rightHessian = new InverseMarginalTransformationHessian(marginalCollection_);
	    IsoProbabilisticTransformation right(p_rightFunction, p_rightGradient, p_rightHessian);
	    IsoProbabilisticTransformation transformation(isoprobabilistic, right);
	    return transformation;
	  }
	// Else simply use the Rosenblatt transformation
	else return NumericalMathFunctionImplementation(new RosenblattEvaluation(clone()));
      }

      /* Get the inverse isoprobabilist transformation */
      SklarCopula::InverseIsoProbabilisticTransformation SklarCopula::getInverseIsoProbabilisticTransformation() const
      {
	// Special case for the elliptical copula
	if (distribution_.getImplementation()->hasEllipticalCopula())
	  {
	    // Get the inverse IsoProbabilisticTransformation from the distribution
	    InverseIsoProbabilisticTransformation inverseIsoprobabilistic(distribution_.getInverseIsoProbabilisticTransformation());
	    // Get the left and right function implementations
	    EvaluationImplementation p_leftFunction = new MarginalTransformationEvaluation(marginalCollection_);
	    // Get the left and right gradient implementations
	    GradientImplementation p_leftGradient = new MarginalTransformationGradient(marginalCollection_);
	    // Get the left and right hessian implementations
	    HessianImplementation p_leftHessian = new MarginalTransformationHessian(marginalCollection_);
	    InverseIsoProbabilisticTransformation left(p_leftFunction, p_leftGradient, p_leftHessian);
	    InverseIsoProbabilisticTransformation transformation(left, inverseIsoprobabilistic);
	    return transformation;
	  }
	// Else simply use the inverse  Rosenblatt transformation
	else return NumericalMathFunctionImplementation(new InverseRosenblattEvaluation(clone()));
      }

      /* Get the standard distribution */
      SklarCopula::Implementation SklarCopula::getStandardDistribution() const
      {
	return distribution_.getStandardDistribution().getImplementation();
      }

      /* Parameters value and description accessor */
      SklarCopula::NumericalPointWithDescriptionCollection SklarCopula::getParametersCollection() const
      {
	NumericalPointWithDescriptionCollection parameters(0);
	const NumericalPointWithDescriptionCollection distributionParameters(distribution_.getParametersCollection());
	const UnsignedLong dimension(distribution_.getDimension());
	// If the underlying distribution has dependence parameters
	if (distributionParameters.getSize() == dimension + 1)
	  {
	    parameters.add(distributionParameters[dimension]);
	  }
	return parameters;
      }

      /* Tell if the distribution has independent copula */
      Bool SklarCopula::hasIndependentCopula() const
      {
	return distribution_.getImplementation()->hasIndependentCopula();
      }

      /* Tell if the distribution has elliptical copula */
      Bool SklarCopula::hasEllipticalCopula() const
      {
	return distribution_.getImplementation()->hasEllipticalCopula();
      }

      /* Distribution accessor */
      void SklarCopula::setDistribution(const Distribution distribution)
	throw(InvalidArgumentException)
      {
	distribution_ = distribution;
      }

      /* Distribution accessor */
      Distribution SklarCopula::getDistribution() const
      {
	return distribution_;
      }

      /* Method save() stores the object through the StorageManager */
      void SklarCopula::save(const StorageManager::Advocate & adv) const
      {
	CopulaImplementation::save(adv);
	adv.writeValue(distribution_, StorageManager::MemberNameAttribute, "distribution_");
	adv.writeValue(marginalCollection_, StorageManager::MemberNameAttribute, "marginalCollection_");
      }

      /* Method load() reloads the object from the StorageManager */
      void SklarCopula::load(const StorageManager::Advocate & adv)
      {
	CopulaImplementation::load(adv);
	adv.readValue(distribution_, StorageManager::MemberNameAttribute, "distribution_");
	adv.readValue(marginalCollection_, StorageManager::MemberNameAttribute, "marginalCollection_");
      }
      



    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
