//                                               -*- C++ -*-
/**
 *  @file  RandomVector.hxx
 *  @brief The class that implements all random vectors
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: RandomVector.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_RANDOMVECTOR_HXX
#define OPENTURNS_RANDOMVECTOR_HXX

#include "TypedInterfaceObject.hxx"
#include "Pointer.hxx"
#include "RandomVectorImplementation.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Model {


      /**
       * @class RandomVector
       *
       * The class that implements all random vectors
       */
      class RandomVector 
	: public Base::Common::TypedInterfaceObject<RandomVectorImplementation>
      {
	CLASSNAME;

      public:

	/* Some typedefs for easy reading */
	typedef RandomVectorImplementation::Implementation           Implementation;
	typedef RandomVectorImplementation::Indices                  Indices;
	typedef RandomVectorImplementation::Antecedent               Antecedent;
	typedef RandomVectorImplementation::NumericalMathFunction    NumericalMathFunction;
	typedef RandomVectorImplementation::NumericalPoint           NumericalPoint;
	typedef RandomVectorImplementation::NumericalSample          NumericalSample;
	typedef RandomVectorImplementation::CovarianceMatrix         CovarianceMatrix;
	typedef RandomVectorImplementation::ComparisonOperator       ComparisonOperator;
	typedef RandomVectorImplementation::FunctionalChaosResult    FunctionalChaosResult;
	typedef RandomVectorImplementation::InvalidArgumentException InvalidArgumentException;
	typedef RandomVectorImplementation::Description              Description;

	/** Default constructor */
	explicit RandomVector() {} ;

	/** Default constructor */
	RandomVector(const RandomVectorImplementation & implementation,
		     const String & name = OT::DefaultName);


	/** Constructor from implementation */
	RandomVector(const Implementation & p_implementation,
		     const String & name = OT::DefaultName);

#ifndef SWIG
	/** Constructor from implementation pointer */
	RandomVector(RandomVectorImplementation * p_implementation,
		     const String & name = OT::DefaultName);
#endif

	/** Constructor for constant vector */
	explicit RandomVector(const NumericalPoint & point,
			      const String & name = OT::DefaultName);

	/** Constructor for distribution-based vector */
	explicit RandomVector(const Distribution & distribution,
			      const String & name = OT::DefaultName);

	/** Constructor for composite vector */
	RandomVector(const NumericalMathFunction & function, 
		     const RandomVector & antecedent,
		     const String & name = OT::DefaultName);
	
	/** Constructor for functional chaos vector */
	RandomVector(const FunctionalChaosResult & functionalChaosResult);
	

	/** String converter */
	String __repr__() const;

	/** Description Accessor */
	void setDescription(const Description & description);
	Description getDescription() const;

	/** Is the underlying random vector composite ? */
	Bool isComposite() const;

	/** Dimension accessor */
	UnsignedLong getDimension() const;

	/** Realization accessor */
	NumericalPoint getRealization() const;

	/** Numerical sample accessor */
	NumericalSample getNumericalSample(const UnsignedLong size) const;

	/** Get the random vector corresponding to the i-th marginal component */
	RandomVector getMarginal(const UnsignedLong i) const throw(InvalidArgumentException);

	/** Get the marginal random vector corresponding to indices components */
	RandomVector getMarginal(const Indices & indices) const throw(InvalidArgumentException);

	/** Mean accessor */
	NumericalPoint getMean() const;

	/** Covariance accessor */
	CovarianceMatrix getCovariance() const;

	/** This method allows to access the antecedent RandomVector in case of a composite RandomVector */
	Antecedent getAntecedent() const;

	/** This method allows to access the NumericalMathFunction in case of a composite RandomVector */
	NumericalMathFunction getFunction() const;

	/** This method allows to access the Distribution in case of a usual RandomVector */
	Distribution getDistribution() const;

	/** Operator accessor */
	ComparisonOperator getOperator() const;

	/** Threshold accessor */
	NumericalScalar getThreshold() const;

      protected:
	/** Constructor for event vector */
	explicit RandomVector(const RandomVector & antecedent,
			      const ComparisonOperator & op,
			      const NumericalScalar threshold,
			      const String & name = OT::DefaultName) throw(InvalidArgumentException);

      }; /* class RandomVector */


    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_RANDOMVECTOR_HXX */
