//                                               -*- C++ -*-
/**
 *  @file  Student.hxx
 *  @brief The Student distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: Student.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_STUDENT_HXX
#define OPENTURNS_STUDENT_HXX

#include "OTprivate.hxx"
#include "EllipticalDistribution.hxx"
#include "Exception.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class Student
       *
       * The Student distribution.
       */
      class Student
        : public Model::EllipticalDistribution
      {
        static const UnsignedLong MaximumNumberOfPoints;
        static const UnsignedLong MinimumNumberOfPoints;
        static const NumericalScalar MaximumCDFEpsilon;
        static const NumericalScalar MinimumCDFEpsilon;

        CLASSNAME;
      public:

        typedef Base::Common::InvalidArgumentException            InvalidArgumentException;
        typedef Model::EllipticalDistribution                     EllipticalDistribution;  // required by SWIG
        typedef EllipticalDistribution::NumericalPoint            NumericalPoint;
        typedef EllipticalDistribution::NumericalSample           NumericalSample;
        typedef EllipticalDistribution::CovarianceMatrix          CovarianceMatrix;
        typedef EllipticalDistribution::Indices                   Indices;
        typedef EllipticalDistribution::Interval                  Interval;
        typedef EllipticalDistribution::NumericalPointWithDescriptionCollection  NumericalPointWithDescriptionCollection;
        typedef EllipticalDistribution::NotDefinedException       NotDefinedException;
        typedef EllipticalDistribution::StorageManager            StorageManager;


        /** Default constructor */
        Student(const NumericalScalar nu = 3.0,
                const UnsignedLong dimension = 1);

        /** Parameters constructor */
        Student(const NumericalScalar nu,
                const NumericalScalar mu,
                const NumericalScalar sigma = 1.0)
          throw(InvalidArgumentException);

        /** Parameters constructor */
        Student(const NumericalScalar nu,
                const NumericalPoint & mu,
                const NumericalPoint & sigma,
                const CorrelationMatrix & R)
          throw(InvalidArgumentException);


        /** Comparison operator */
        Bool operator ==(const Student & other) const;

        /** String converter */
	String __repr__() const;



        /* Interface inherited from Distribution */

        /** Virtual constructor */
        virtual Student * clone() const;

        /** Get one realization of the distribution */
        virtual NumericalPoint getRealization() const;

        /** Get the CDF of the distribution */
        using EllipticalDistribution::computeCDF;
        virtual NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

        /** Get the probability content of an interval */
        NumericalScalar computeProbability(const Interval & interval) const;

        /** Get the PDFGradient of the distribution */
        virtual NumericalPoint computePDFGradient(const NumericalPoint & point) const;

        /** Get the CDFGradient of the distribution */
        virtual NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

        /** Compute the radial distribution CDF */
        NumericalScalar computeRadialDistributionCDF(const NumericalScalar radius) const;

        /** Compute the PDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
        NumericalScalar computeConditionalPDF(const NumericalScalar x, const NumericalPoint & y) const;

        /** Compute the CDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
        NumericalScalar computeConditionalCDF(const NumericalScalar x, const NumericalPoint & y) const;

        /** Compute the quantile of Xi | X1, ..., Xi-1, i.e. x such that CDF(x|y) = q with x = Xi, y = (X1,...,Xi-1) */
        NumericalScalar computeConditionalQuantile(const NumericalScalar q, const NumericalPoint & y) const;

        /** Get the i-th marginal distribution */
        Implementation getMarginal(const UnsignedLong i) const throw(InvalidArgumentException);

        /** Get the distribution of the marginal distribution corresponding to indices dimensions */
        Implementation getMarginal(const Indices & indices) const throw(InvalidArgumentException);

        /** Get the mean of the distribution */
        virtual NumericalPoint getMean() const throw(NotDefinedException);

        /** Get the standard deviation of the distribution */
        virtual NumericalPoint getStandardDeviation() const throw(NotDefinedException);

        /** Get the skewness of the distribution */
        virtual NumericalPoint getSkewness() const throw(NotDefinedException);

        /** Get the kurtosis of the distribution */
        virtual NumericalPoint getKurtosis() const throw(NotDefinedException);

        /** Get the covariance of the distribution */
        virtual CovarianceMatrix getCovariance() const throw(NotDefinedException);

        /** Parameters value and description accessor */
        virtual NumericalPointWithDescriptionCollection getParametersCollection() const;

        /* Interface specific to Student */

        /** Nu accessor */
        void setNu(const NumericalScalar nu)
          throw(InvalidArgumentException);
        NumericalScalar getNu() const;

        /** Mu accessor */
        void setMu(const NumericalScalar mu);
        NumericalScalar getMu() const;

        /** Compute the density generator of the ellipticalal generator, i.e.
         *  the function phi such that the density of the distribution can
         *  be written as p(x) = phi(t(x-mu)R^(-1)(x-mu))                      */
        virtual NumericalScalar computeDensityGenerator(const NumericalScalar betaSquare) const;

        /** Compute the derivative of the density generator */
        virtual NumericalScalar computeDensityGeneratorDerivative(const NumericalScalar betaSquare) const;

        /** Compute the seconde derivative of the density generator */
        virtual NumericalScalar computeDensityGeneratorSecondDerivative(const NumericalScalar betaSquare) const;

        /** Method save() stores the object through the StorageManager */
        virtual void save(const StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        virtual void load(const StorageManager::Advocate & adv);

      protected:


      private:

        /** Compute the numerical range of the distribution given the parameters values */
        void computeRange();

        /** Quantile computation for dimension=1 */
        NumericalScalar computeScalarQuantile(const NumericalScalar prob,
                                              const NumericalScalar initialGuess = 0.0,
                                              const NumericalScalar initialStep = 1.0,
                                              const NumericalScalar precision = DefaultQuantileEpsilon) const;

        /** The nu of the Student distribution */
        NumericalScalar nu_;

        NumericalScalar normalizationFactor_;
      }; /* class Student */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_STUDENT_HXX */
