 //                                               -*- C++ -*-
/**
 *  @file  JacobiFactory.cxx
 *  @brief Jacobi polynomial factory
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "JacobiFactory.hxx"
#include "PersistentObjectFactory.hxx"
#include "Beta.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      CLASSNAMEINIT(JacobiFactory);

      static Base::Common::Factory<JacobiFactory> RegisteredFactory("JacobiFactory");

      typedef Distribution::Beta                     Beta;
      typedef Base::Common::InvalidArgumentException InvalidArgumentException;


      /* Default constructor: (1, 1) order Jacobi polynomial associated with the default Beta() = Beta(2, 4, -1, 1) distribution which is equal to the Epanechnikov distribution */
       JacobiFactory::JacobiFactory()
	: OrthogonalUniVariatePolynomialFactory(Beta()),
	  alpha_(1.0),
	  beta_(1.0)
      {
	initializeCache();
      }


      /* Parameter constructor: (alpha, beta) is the order of the Jacobi polynomial, associated with the Beta(beta + 1, alpha + beta + 2, -1, 1) distribution */
      JacobiFactory::JacobiFactory(const NumericalScalar alpha,
				   const NumericalScalar beta)
	: OrthogonalUniVariatePolynomialFactory(Beta(beta + 1.0, alpha + beta + 2.0, -1.0, 1.0)  ),
	  alpha_(alpha),
	  beta_(beta)
      {
	if ((alpha <= -1.0) || (beta <= -1.0)) throw InvalidArgumentException(HERE) << "Error: must have alpha>-1 and beta>-1 to build Jacobi polynomials.";
	initializeCache();
      }

      /* Virtual constructor */
      JacobiFactory * JacobiFactory::clone() const
      {
	return new JacobiFactory(*this);
      }


      /* Determine the coefficients of the first Orthonormal Jacobi polynomial*/
      JacobiFactory::Coefficients JacobiFactory::getP0Coefficients() const
      {
	Coefficients Jacobi0Coefficients(1); 
	Jacobi0Coefficients[0] = 1.0;
	return Jacobi0Coefficients;
      }


      /* Determine the coefficients of the second Orthonormal Jacobi polynomial*/
      JacobiFactory::Coefficients JacobiFactory::getP1Coefficients() const
      {
	Coefficients Jacobi1Coefficients(2); 
	const NumericalScalar factor(0.5 * sqrt((alpha_ + beta_ + 3.0) / ((alpha_ + 1.0) * (beta_ + 1.0))));
	Jacobi1Coefficients[0] = (alpha_ - beta_) * factor;
	Jacobi1Coefficients[1] = (alpha_ + beta_ + 2.0) * factor;
	return Jacobi1Coefficients;
      }

      /* Calculate the coefficients of recurrence a0, a1, a2 such that
	 Pn(x) = (a0 * x + a1) * Pn-1(x) + a2 * Pn-2(x) */
      JacobiFactory::Coefficients JacobiFactory::getRecurrenceCoefficients(const UnsignedLong n) const
      {
	if (n < 2) throw InvalidArgumentException(HERE) << "Error: cannot compute recurrence coefficients for n < 2.";
	Coefficients recurrenceCoefficients(3);
	const NumericalScalar nAlpha(n + alpha_);
	const NumericalScalar nBeta(n + beta_);
	const NumericalScalar nAlphaBeta(nAlpha + beta_);
	const NumericalScalar twoNAlphaBeta(nAlpha + nBeta);
	const NumericalScalar factor(sqrt((twoNAlphaBeta + 1.0) / (n * nAlpha * nBeta * nAlphaBeta)));
	const NumericalScalar sqrtFactor(0.5 * sqrt(twoNAlphaBeta - 1.0));
	recurrenceCoefficients[0] = sqrtFactor * factor * twoNAlphaBeta;
	recurrenceCoefficients[1] = sqrtFactor * factor * (alpha_ - beta_) * (alpha_ + beta_) / (twoNAlphaBeta - 2.0);
	recurrenceCoefficients[2] = -factor * twoNAlphaBeta * sqrt((nAlpha - 1.0) * (nBeta - 1.0) * (nAlphaBeta - 1.0) * (n - 1.0) / (twoNAlphaBeta - 3.0)) / (twoNAlphaBeta - 2.0);
	return recurrenceCoefficients;
      }

      /* Alpha accessor */
      NumericalScalar JacobiFactory::getAlpha() const
      {
	return alpha_;
      }

      /* Beta accessor */
      NumericalScalar JacobiFactory::getBeta() const
      {
	return beta_;
      }

      /* String converter */
      String JacobiFactory::__repr__() const
      {
	return OSS() << "class=" << getClassName()
		     << " alpha=" << alpha_
		     << " beta=" << beta_
		     << " measure=" << measure_;
      }


    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
