//                                               -*- C++ -*-
/**
 *  @file  LeastSquaresStrategy.cxx
 *  @brief An implementation of projection strategy as a leastSquares
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "LeastSquaresStrategy.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"
#include "MonteCarloExperiment.hxx"
#include "LeastSquaresAlgorithm.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      CLASSNAMEINIT(LeastSquaresStrategy);

      static Base::Common::Factory<LeastSquaresStrategy> RegisteredFactory("LeastSquaresStrategy");

      typedef Base::Optim::LeastSquaresAlgorithm LeastSquaresAlgorithm;

      /* Default constructor */
      LeastSquaresStrategy::LeastSquaresStrategy()
	: ProjectionStrategyImplementation(),
	  p_weightedExperiment_(MonteCarloExperiment().clone()),
	  inputSample_(0, 0),
	  outputSample_(0, 0)
      {
	// The leastSquares strategy impose its distribution to the weighted experiment
	p_weightedExperiment_->setDistribution(getMeasure());
      }

      /* Parameter constructor */
      LeastSquaresStrategy::LeastSquaresStrategy(const Distribution & measure)
	: ProjectionStrategyImplementation(measure),
	  p_weightedExperiment_(MonteCarloExperiment().clone()),
	  inputSample_(0, 0),
	  outputSample_(0, 0)
      {
	// The leastSquares strategy impose its distribution to the weighted experiment
	p_weightedExperiment_->setDistribution(getMeasure());
      }

      /* Parameter constructor */
      LeastSquaresStrategy::LeastSquaresStrategy(const WeightedExperiment & weightedExperiment)
	: ProjectionStrategyImplementation(weightedExperiment.getDistribution()),
	  p_weightedExperiment_(weightedExperiment.clone()),
	  inputSample_(0, 0),
	  outputSample_(0, 0)
      {
	// The leastSquares strategy get its measure from the weighted experiment, nothing special to do
      }

      /* Parameter constructor */
      LeastSquaresStrategy::LeastSquaresStrategy(const Distribution & measure,
						 const WeightedExperiment & weightedExperiment)
	: ProjectionStrategyImplementation(measure),
	  p_weightedExperiment_(weightedExperiment.clone()),
	  inputSample_(0, 0),
	  outputSample_(0, 0)
      {
	// The leastSquares strategy impose the distribution of the weighted experiment
	p_weightedExperiment_->setDistribution(getMeasure());
      }

      
      /* Virtual constructor */
      LeastSquaresStrategy * LeastSquaresStrategy::clone() const
      {
	return new LeastSquaresStrategy(*this);
      }

      /* Measure accessor */
      void LeastSquaresStrategy::setMeasure(const Distribution & measure)
      {
	ProjectionStrategyImplementation::setMeasure(measure);
	// Set the measure as the distribution of the weighted experiment
	p_weightedExperiment_->setDistribution(getMeasure());
      }

      /* Experiment accessors */
      void LeastSquaresStrategy::setExperiment(const WeightedExperiment & weightedExperiment)
      {
	p_weightedExperiment_ = weightedExperiment.clone();
	p_weightedExperiment_->setDistribution(getMeasure());
      }

      WeightedExperiment LeastSquaresStrategy::getExperiment() const
      {
	return *p_weightedExperiment_;
      }

     /* String converter */
      String LeastSquaresStrategy::__repr__() const
      {
	return OSS() << "class=" << getClassName()
		     << " experiment=" << p_weightedExperiment_->__repr__();
      }


      /* Compute the components alpha_k_p_ by projecting the model on the partial L2 basis
	 For the moment, there is no specific strategy for improving the approximation of
	 the L2 integral by a finite sum: the same input sample is used for all the calls
	 to this method */
      void LeastSquaresStrategy::computeCoefficients(const NumericalMathFunction & function,
						     const NumericalMathFunctionCollection & partialBasis,
						     const Indices & indices,
						     const Indices & addedIndices,
						     const Indices & removedIndices)
      {
	// If the input sample has not yet been generated
	// we have to generate it and to evaluate the model
	// upon this sample
	if (inputSample_.getSize() == 0)
	  {
	    inputSample_  = p_weightedExperiment_->generate();
	    outputSample_ = function(inputSample_);
	  }
	// Compute the coefficients and the residual using the least square algorithm
	LeastSquaresAlgorithm algo(inputSample_, outputSample_, p_weightedExperiment_->getWeight(), partialBasis);
	alpha_k_p_ = algo.getCoefficients();
	residual_p_ = algo.getResidual();
      }


      /* Method save() stores the object through the StorageManager */
      void LeastSquaresStrategy::save(const StorageManager::Advocate & adv) const
      {
	ProjectionStrategyImplementation::save(adv);
      }


      /* Method load() reloads the object from the StorageManager */
      void LeastSquaresStrategy::load(const StorageManager::Advocate & adv)
      {
	ProjectionStrategyImplementation::load(adv);
      }



    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
