//                                               -*- C++ -*-
/**
 *  @file  InverseMarginalTransformationGradient.cxx
 *  @brief Class for the Nataf transformationGradient evaluation for elliptical
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: InverseMarginalTransformationGradient.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include "InverseMarginalTransformationGradient.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {



      CLASSNAMEINIT(InverseMarginalTransformationGradient);

      /* Parameter constructor */
      InverseMarginalTransformationGradient::InverseMarginalTransformationGradient(const DistributionCollection & distributionCollection):
	NumericalMathGradientImplementation(),
	distributionCollection_(distributionCollection)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      InverseMarginalTransformationGradient * InverseMarginalTransformationGradient::clone() const
      {
	return new InverseMarginalTransformationGradient(*this);
      }

      /* Gradient */
      InverseMarginalTransformationGradient::Matrix InverseMarginalTransformationGradient::gradient(const NumericalPoint & in) const
	throw (InvalidArgumentException, InternalException)
      {
	// As the marginal transformation is a function from Rn to Rn, the gradient is a square matrix
	UnsignedLong dimension(getInputNumericalPointDimension());
	Matrix result(dimension, dimension);
	// Apply quantile derivative, i.e. 1 / PDF(Quantile) over the components
	// GoF=I -> F'.G'oF=1 -> G'=1/F'oG
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    result(i, i) = 1.0 / distributionCollection_[i].computePDF(distributionCollection_[i].computeQuantile(in[i]));
	  }
	return result;
      }
	  
      /* Accessor for input point dimension */
      UnsignedLong InverseMarginalTransformationGradient::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return distributionCollection_.getSize();
      }

      /* Accessor for output point dimension */
      UnsignedLong InverseMarginalTransformationGradient::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return distributionCollection_.getSize();
      }

      /* String converter */
      String InverseMarginalTransformationGradient::__repr__() const
      {
	OSS oss;
	oss << "class=" << InverseMarginalTransformationGradient::GetClassName();                   
	return oss;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

