//                                               -*- C++ -*-
/**
 *  @file  InverseMarginalTransformationEvaluation.cxx
 *  @brief Class for the Nataf transformation evaluation for elliptical
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: InverseMarginalTransformationEvaluation.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include "InverseMarginalTransformationEvaluation.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {



      CLASSNAMEINIT(InverseMarginalTransformationEvaluation);

      /* Parameter constructor */
      InverseMarginalTransformationEvaluation::InverseMarginalTransformationEvaluation(const DistributionCollection & distributionCollection):
	NumericalMathEvaluationImplementation(),
	distributionCollection_(distributionCollection)
      {
	Description description;
	UnsignedLong size(distributionCollection.getSize());
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    OSS oss;
	    oss << "x" << i;
	    description.add(oss);
	  }
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    OSS oss;
	    oss << "y" << i;
	    description.add(oss);
	  }
	setDescription(description);
      }

      /* Virtual constructor */
      InverseMarginalTransformationEvaluation * InverseMarginalTransformationEvaluation::clone() const
      {
	return new InverseMarginalTransformationEvaluation(*this);
      }

      /* Evaluation */
      InverseMarginalTransformationEvaluation::NumericalPoint InverseMarginalTransformationEvaluation::operator () (const NumericalPoint & in) const
	throw (InvalidArgumentException, InternalException)
      {
	UnsignedLong dimension(getOutputNumericalPointDimension());
	NumericalPoint result(dimension);
	// Apply quantile over the components
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    result[i] = distributionCollection_[i].computeQuantile(in[i])[0];
	  }
	return result;
      }

      /*
       * Gradient according to the marginal parameters.
       * F(Q(y, p), p) = Id
       * F:RxRp->R
       * Q:RxRp->R
       * dF/dp = dF/dx(Q(y, p), p) . dQ/dp(y, p) + dF/dp(Q(y, p), p)
       *       = f(Q(y, p), p) . dQ/dp(y, p) + dF/dp(Q(y, p), p)
       *       = 0
       * => dQ/dp(y, p) = - dF/dp(Q(y, p), p) / f(Q(y, p), p)
       */
      InverseMarginalTransformationEvaluation::Matrix InverseMarginalTransformationEvaluation::parametersGradient(const NumericalPoint & in) const
      {
	const NumericalPoint parameters(getParameters());
	const UnsignedLong parametersDimension(parameters.getDimension());
	const UnsignedLong inputDimension(getInputNumericalPointDimension());
	Matrix result(parametersDimension, inputDimension);
	UnsignedLong rowIndex(0);
	for (UnsignedLong j = 0; j < inputDimension; ++j)
	  {
	    const NumericalPoint x(distributionCollection_[j].computeQuantile(in[j]));
	    const NumericalPoint marginalCDFGradient(distributionCollection_[j].computeCDFGradient(x));
	    const NumericalScalar pdf(distributionCollection_[j].computePDF(x));
	    const NumericalPoint marginalQuantileGradient((-1.0 / pdf) * marginalCDFGradient);
	    const UnsignedLong marginalParametersDimension(marginalCDFGradient.getDimension());
	    for (UnsignedLong i = 0; i < marginalParametersDimension; ++i)
	      {
		result(rowIndex, j) = marginalQuantileGradient[i];
		rowIndex++;
	      }
	  }
	return result;
      }

      /* Accessor for input point dimension */
      UnsignedLong InverseMarginalTransformationEvaluation::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return distributionCollection_.getSize();
      }

      /* Accessor for output point dimension */
      UnsignedLong InverseMarginalTransformationEvaluation::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return distributionCollection_.getSize();
      }

      /* String converter */
      String InverseMarginalTransformationEvaluation::__repr__() const
      {
	OSS oss;
	oss << "class=" << InverseMarginalTransformationEvaluation::GetClassName();                   
	return oss;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

