//                                               -*- C++ -*-
/**
 *  @file  SORMResult.hxx
 *  @brief SORMResult implements the First Order Reliability Method
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2008-05-16 00:03:02 +0200 (ven, 16 mai 2008) $
 *  Id:      $Id: SORM.hxx 807 2008-05-15 22:03:02Z lebrun $
 */
#ifndef OPENTURNS_SORMResult_HXX
#define OPENTURNS_SORMResult_HXX

#include "AnalyticalResult.hxx"
#include "Event.hxx"
#include "SquareMatrix.hxx"
#include "Distribution.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {
      /**
       * @class SORMResult
       * SORMResult stores the SORM result 
       */
      class SORMResult:
	public AnalyticalResult
      {
	CLASSNAME;
      public:
	typedef AnalyticalResult::StorageManager  StorageManager;
	typedef AnalyticalResult::NumericalPoint  NumericalPoint;
	typedef AnalyticalResult::Event           Event;
	typedef Base::Type::SquareMatrix          SquareMatrix;
	typedef Base::Common::NotDefinedException NotDefinedException;
        typedef Model::Distribution               Distribution;

	/** Standard constructor */
	SORMResult(const NumericalPoint & standardSpaceDesignPoint,
		   const Event & limitStateVariable,
		   const Bool isStandardPointOriginInFailureSpace,
		   const String & name = OT::DefaultName);

	/* Default constructor (required by SWIG :-<) */
	explicit SORMResult();

	/** Virtual constructor */
	virtual SORMResult * clone() const;

	/** EventProbabilityBreitung accessor */
	NumericalScalar getEventProbabilityBreitung() const;

	/** EventProbabilityHohenBichler accessor */
	NumericalScalar getEventProbabilityHohenBichler() const;

	/** EventProbabilityTvedt accessor */
	NumericalScalar getEventProbabilityTvedt() const;

	/** GeneralisedReliabilityIndexBreitung accessor */
	NumericalScalar getGeneralisedReliabilityIndexBreitung() const;

	/** GeneralisedReliabilityIndexHohenBichler accessor */
	NumericalScalar getGeneralisedReliabilityIndexHohenBichler() const;

	/** GeneralisedReliabilityIndexTvedt accessor */
	NumericalScalar getGeneralisedReliabilityIndexTvedt() const;

	/** SortedCurvatures accessor */
	NumericalPoint getSortedCurvatures() const;

	/** String converter */
	String __repr__() const;

	/** Method save() stores the object through the StorageManager */
	void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(const StorageManager::Advocate & adv);

      private:

	/** the function that evaluates the curvatures of the standard limite state fucntion at the standard design point */
	void computeCurvatures();

	/** The function that actually evaluates the event probability with FORM approximation */
	void computeEventProbabilityBreitung() throw(NotDefinedException);

	/** The function that actually evaluates the event probability with FORM approximation */
	void computeEventProbabilityHohenBichler() throw(NotDefinedException);

	/** The function that actually evaluates the event probability with FORM approximation */
	void computeEventProbabilityTvedt() throw(NotDefinedException);

	/** The function that actually evaluates the generalised reliability index with FORM approximation */
	void computeGeneralisedReliabilityIndex();

	/** type of gradients and hessian attributes are due to the constraint : limitStateVariable is scalar only in Open Turns v std */
	/** It should be a SymmetricMatrix but there is a bug in linear algebra with these matrices */
	SquareMatrix hessianLimitStateFunction_; 
	NumericalPoint gradientLimitStateFunction_; 
	NumericalPoint sortedCurvatures_;  
	NumericalScalar eventProbabilityBreitung_;
	NumericalScalar eventProbabilityHohenBichler_;
	NumericalScalar eventProbabilityTvedt_;      
	NumericalScalar generalisedReliabilityIndexBreitung_;
	NumericalScalar generalisedReliabilityIndexHohenBichler_;
	NumericalScalar generalisedReliabilityIndexTvedt_;
	Distribution standardDistribution_;
	Distribution standardMarginal_;
      }; // class Result

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_SORMResult_HXX */
