//                                               -*- C++ -*-
/**
 *  @file  NumericalPoint.hxx
 *  @brief NumericalPoint implements the classical mathematical point
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: NumericalPoint.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_NUMERICALPOINT_HXX
#define OPENTURNS_NUMERICALPOINT_HXX

#include "TypedCollectionInterfaceObject.hxx"
#include "NumericalPointImplementation.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      /**
       * @class NumericalPoint
       *
       * NumericalPoint implements the classical mathematical point
       */

      class NumericalPoint :
	public Common::TypedCollectionInterfaceObject<NumericalPointImplementation>
      {
	CLASSNAME;

      public:

	typedef Common::InvalidArgumentException                        InvalidArgumentException;
	typedef NumericalPointImplementation::NumericalScalarCollection NumericalScalarCollection;

	/** Default constructor */
	NumericalPoint();

	/** Constructor with size */
	NumericalPoint(const UnsignedLong size,
		       const NumericalScalar value = 0.0);

	/** Constructor from implementation */
	NumericalPoint(const NumericalPointImplementation & implementation);

// 	/** Description Accessor */
// 	void setDescription(const Description & description);
// 	Description getDescription() const;

	/** Collection accessor */
	const NumericalScalarCollection & getCollection() const;

	/** String converter */
	virtual String __repr__() const;
	virtual String __str__() const;

	/* Method __eq__() is for Python */
	Bool __eq__(const NumericalPoint & rhs) const;

	/* Method __ne__() is for Python */
	Bool __ne__(const NumericalPoint & rhs) const;

	/** Addition operator */
	NumericalPoint operator +(const NumericalPoint & rhs) const
	  throw (InvalidArgumentException);

	/** Substraction operator */
	NumericalPoint operator -(const NumericalPoint & rhs) const
	  throw (InvalidArgumentException);

	/** In-place addition operator */
	NumericalPoint & operator +=(const NumericalPoint & other) throw (InvalidArgumentException);
	
	/** In-place substraction operator */
	NumericalPoint & operator -=(const NumericalPoint & other) throw (InvalidArgumentException);

	/** Product operator */
	NumericalPoint operator *(const NumericalScalar scalar) const;

	/**  In-place product operator */
	NumericalPoint & operator *=(const NumericalScalar scalar);
	
	/**  Norm */
	NumericalScalar norm() const;
	
	/**  Norm^2 */
	NumericalScalar norm2() const;
	
	/** Dot product operator */
	static NumericalScalar dot(const NumericalPoint & lhs,
				   const NumericalPoint & rhs)
	  throw (InvalidArgumentException);

      protected:

	// This class is intented to be used only by derived class that want to usethe constructor from implementation pointer
	class Derived {};

	/** Constructor from implementation */
	NumericalPoint(NumericalPointImplementation * p_implementation, Derived);


      }; /* class NumericalPoint */


#ifndef SWIG
      /** Comparison operator */
      Bool operator ==(const NumericalPoint & lhs,
		       const NumericalPoint & rhs);
      
      /** Ordering operator */
      Bool operator <(const NumericalPoint & lhs,
		      const NumericalPoint & rhs);
      
      /** Product operator */
      NumericalPoint operator *(const NumericalScalar scalar,
				const NumericalPoint & point);
#endif
      

    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_NUMERICALPOINT_HXX */
