//                                               -*- C++ -*-
/**
 *  @file  MatrixImplementation.hxx
 *  @brief MatrixImplementation implements the Matrix class
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: MatrixImplementation.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_MATRIXIMPLEMENTATION_HXX
#define OPENTURNS_MATRIXIMPLEMENTATION_HXX

#include "PersistentCollection.hxx"
#include "Description.hxx"
#include "NumericalPoint.hxx"
#include "Exception.hxx"
#include "Collection.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      /**
       * @class MatrixImplementation
       *
       * MatrixImplementation implements the classical mathematical MatrixImplementation
       */

      class MatrixImplementation
	: public PersistentCollection<NumericalScalar>
      
      {
	CLASSNAME;
	
#ifndef SWIG
	/** Declaration of friend operators */
	friend MatrixImplementation operator * (const NumericalScalar s,
						const MatrixImplementation & m)
	{
	  return m.operator*(s);
	}
#endif
	
	
      public:

	// All the pivots with a magnitude less than this threshold are considered as zero
	static const NumericalScalar DefaultSmallPivot;
	
	typedef Common::InvalidDimensionException InvalidDimensionException ;
	typedef Common::InvalidArgumentException  InvalidArgumentException ;
	typedef Collection<NumericalComplex>      NumericalComplexCollection;
	
	/** Default constructor */
	MatrixImplementation();
	
	/** Constructor with size (rowDim and colDim) */
	MatrixImplementation(const UnsignedLong rowDim,
			     const UnsignedLong colDim);
	
	/** Constructor from range of external collection */
	template <class InputIterator>
	MatrixImplementation(const UnsignedLong rowDim,
			     const UnsignedLong colDim,
			     const InputIterator first,
			     const InputIterator last);
	
	/** Constructor from external collection */
	/** If the dimensions of the matrix and of the collection */
	/** do not correspond, either the collection is truncated */
	/** or the rest of the matrix is filled with zeros */
	MatrixImplementation(const UnsignedLong rowDim,
			     const UnsignedLong colDim,
			     const Collection<NumericalScalar> & elementsValues);
	

	/** Virtual constructor */
	virtual MatrixImplementation * clone() const;

	/** String converter */
	virtual String __repr__() const;
	virtual String __str__() const;
	
	/** Operator () gives access to the elements of the MatrixImplementation (to modify these elements) */
	/** The element of the MatrixImplementation is designated by its row number i and its column number j */
	NumericalScalar & operator () (const UnsignedLong i,
				       const UnsignedLong j) throw(InvalidDimensionException);
	
	/** Operator () gives access to the elements of the MatrixImplementation (read only) */
	/** The element of the MatrixImplementation is designated by its row number i and its column number j */
	const NumericalScalar & operator () (const UnsignedLong i,
					     const UnsignedLong j) const throw(InvalidDimensionException);
	
	/** Get the dimensions of the MatrixImplementation */
	/** Number of rows */
	const UnsignedLong getNbRows() const ;
	/** Number of columns */
	const UnsignedLong getNbColumns() const ;
	/** Dimension (for square matrices only */
	const UnsignedLong getDimension() const ;
	
	/** MatrixImplementation transpose */
	MatrixImplementation transpose () const ;
	
	/** MatrixImplementation addition (must have the same dimensions) */
	MatrixImplementation operator + (const MatrixImplementation & m) const throw(InvalidDimensionException);
	
	/** MatrixImplementation substraction (must have the same dimensions) */
	MatrixImplementation operator - (const MatrixImplementation & m) const throw(InvalidDimensionException);
	
	/** MatrixImplementation multiplications (must have consistent dimensions) */
	MatrixImplementation genProd (const MatrixImplementation & m) const throw(InvalidDimensionException);
	MatrixImplementation symProd (const MatrixImplementation & m,
				      const char symSide) const throw(InvalidDimensionException);
	
	/** MatrixImplementation integer power */
	MatrixImplementation genPower(const UnsignedLong n) const;
	MatrixImplementation symPower(const UnsignedLong n) const;

	/** Multiplications with a NumericalPoint (must have consistent dimensions) */
	NumericalPoint genVectProd (const NumericalPoint & pt) const throw(InvalidDimensionException);
	NumericalPoint symVectProd (const NumericalPoint & pt) const throw(InvalidDimensionException);

	/** Multiplication with a NumericalScalar */
	MatrixImplementation operator * (const NumericalScalar s) const ;
	
	/** Division by a NumericalScalar*/
	MatrixImplementation operator / (const NumericalScalar s) const throw(InvalidArgumentException);
	
	/** Symmetrize MatrixImplementation in case it is a symmetric matrix (stored as a triangular matrix) */
	void symmetrize() const;
	
	/** Resolution of a linear system in case of a rectangular matrix */
	NumericalPoint solveLinearSystemRect(const NumericalPoint & b) const throw(InvalidDimensionException);
	MatrixImplementation solveLinearSystemRect(const MatrixImplementation & b) const throw(InvalidDimensionException);
	
	/** Resolution of a linear system in case of a square matrix */
	NumericalPoint solveLinearSystemSquare(const NumericalPoint & b) const throw(InvalidDimensionException);
	MatrixImplementation solveLinearSystemSquare(const MatrixImplementation & b) const throw(InvalidDimensionException);
	
	/** Resolution of a linear system in case of a symmetric matrix */
	NumericalPoint solveLinearSystemSym(const NumericalPoint & b) const throw(InvalidDimensionException);
	MatrixImplementation solveLinearSystemSym(const MatrixImplementation & b) const throw(InvalidDimensionException);
	
	/** Compute determinant */
	NumericalScalar computeDeterminant() const;
	NumericalScalar computeDeterminantSym() const;
	
	/** Compute eigenvalues */
	NumericalComplexCollection computeEigenValuesSquare() const;
	NumericalPoint computeEigenValuesSym() const;
	
	/** Check if the matrix is SPD */
	virtual Bool isPositiveDefinite() const;
	
	/** Build the Cholesky factorization of the matrix */
	virtual MatrixImplementation computeCholesky() const;
	
	/** Comparison operators */
	Bool operator == (const MatrixImplementation & rhs) const ;
	inline Bool operator != (const MatrixImplementation & rhs) const { return !((*this) == rhs); }
	
	/** Empty returns true if there is no element in the MatrixImplementation */
	const Bool isEmpty() const;

 	/** Method save() stores the object through the StorageManager */
	void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(const StorageManager::Advocate & adv);

      protected:
        
	/** MatrixImplementation Dimensions */
	UnsignedLong nbRows_;
	UnsignedLong nbColumns_;
	
        /** Position conversion function : the indices i & j are used to compute the actual position of the element in the collection */
	inline UnsignedLong convertPosition (const UnsignedLong i,
					     const UnsignedLong j) const ;

      }; /* class MatrixImplementation */
      
      inline UnsignedLong MatrixImplementation::convertPosition (const UnsignedLong i,
								 const UnsignedLong j) const
      {
        return i + nbRows_ * j ;
      }
                  
      /** Constructor from range of external collection */
      template <class InputIterator>
      MatrixImplementation::MatrixImplementation(const UnsignedLong rowDim,
                                                 const UnsignedLong colDim,
						 const InputIterator first,
						 const InputIterator last)
	: PersistentCollection<NumericalScalar>(rowDim * colDim, 0.0),
	  nbRows_(rowDim),
	  nbColumns_(colDim)
      {
        this->assign(first,last);
      }

    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_MATRIXIMPLEMENTATION_HXX */
