//                                               -*- C++ -*-
/**
 *  @file  ProductPolynomialEvaluationImplementation.cxx
 *  @brief This is a nD polynomial build as a product of n 1D polynomial
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "ProductPolynomialEvaluationImplementation.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      using Func::UniVariatePolynomial;
      TEMPLATE_CLASSNAMEINIT(PersistentCollection<UniVariatePolynomial>);

      static Common::Factory<PersistentCollection<UniVariatePolynomial> > RegisteredFactory("PersistentCollection<UniVariatePolynomial>");
    } /* namespace Type */

    namespace Func
    {

      CLASSNAMEINIT(ProductPolynomialEvaluationImplementation);

      static Common::Factory<ProductPolynomialEvaluationImplementation> RegisteredFactory("ProductPolynomialEvaluationImplementation");

      typedef Common::NotYetImplementedException NotYetImplementedException;

      /* Default constructor */
      ProductPolynomialEvaluationImplementation::ProductPolynomialEvaluationImplementation()
	: NumericalMathEvaluationImplementation(),
	  polynomials_()
      {
	// Nothing to do
      }


      /* Constructor */
      ProductPolynomialEvaluationImplementation::ProductPolynomialEvaluationImplementation(const PolynomialCollection & coll)
	: NumericalMathEvaluationImplementation(),
	  polynomials_(coll)
      {
	// Nothing to do
	Description description(0);
	for (UnsignedLong i = 0; i < getInputNumericalPointDimension(); ++i) description.add(OSS() << "x" << i);
	for (UnsignedLong i = 0; i < getOutputNumericalPointDimension(); ++i) description.add(OSS() << "y" << i);
	setDescription(description);
      }


      /* Virtual constructor */
      ProductPolynomialEvaluationImplementation * ProductPolynomialEvaluationImplementation::clone() const
      {
	return new ProductPolynomialEvaluationImplementation(*this);
      }


      /* String converter */
      String ProductPolynomialEvaluationImplementation::__repr__() const
      {
	return OSS() << "class=" << getClassName();
      }


	/* Operator (): Evaluate a product of 1D polynomials for one sample */
      ProductPolynomialEvaluationImplementation::NumericalPoint ProductPolynomialEvaluationImplementation::operator() (const NumericalPoint & in) const
	throw(InvalidArgumentException, InternalException)
      {
	const UnsignedLong inDimension(in.getDimension());
	if (inDimension != getInputNumericalPointDimension()) throw InvalidArgumentException(HERE) << "Error: trying to evaluate a ProductPolynomialFunction with an argument of invalid dimension";
        NumericalScalar productEvaluation(1.0) ;
        for (UnsignedLong i = 0; i < inDimension; ++i)
	      {
		productEvaluation *= polynomials_[i](in[i]);
	      }
	return NumericalPoint(1, productEvaluation);
      }	


      /* Accessor for input point dimension */
      UnsignedLong ProductPolynomialEvaluationImplementation::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return polynomials_.getSize();
      }

      /* Accessor for output point dimension */
      UnsignedLong ProductPolynomialEvaluationImplementation::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return 1;
      }


      /* Method save() stores the object through the StorageManager */
      void ProductPolynomialEvaluationImplementation::save(const StorageManager::Advocate & adv) const
      {
	NumericalMathEvaluationImplementation::save(adv);
	adv.writeValue(polynomials_, StorageManager::MemberNameAttribute, "polynomials_");
      }

      /* Method load() reloads the object from the StorageManager */
      void ProductPolynomialEvaluationImplementation::load(const StorageManager::Advocate & adv)
      {
	NumericalMathEvaluationImplementation::load(adv);
	adv.readValue(polynomials_, StorageManager::MemberNameAttribute, "polynomials_");
      }


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
