//                                               -*- C++ -*-
/**
 *  @file  NumericalMathFunction.cxx
 *  @brief Abstract top-level class for all distributions
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: NumericalMathFunction.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include "NumericalMathFunction.hxx"
#include "ComposedNumericalMathFunction.hxx"
#include "IndicatorNumericalMathEvaluationImplementation.hxx"
#include "LinearCombinationEvaluationImplementation.hxx"
#include "LinearCombinationGradientImplementation.hxx"
#include "LinearCombinationHessianImplementation.hxx"
#include "NoNumericalMathGradientImplementation.hxx"
#include "NoNumericalMathHessianImplementation.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      CLASSNAMEINIT(NumericalMathFunction);

      /* Default constructor */
      NumericalMathFunction::NumericalMathFunction()
	: Common::TypedInterfaceObject<NumericalMathFunctionImplementation>(new NumericalMathFunctionImplementation())
      {
	// Nothing to do
      }

      /* Default constructor */
      NumericalMathFunction::NumericalMathFunction(const String & name)
	: Common::TypedInterfaceObject<NumericalMathFunctionImplementation>(new NumericalMathFunctionImplementation(name))
      {
	// Nothing to do
      }

      /* Constructor from NumericalMathFunctionImplementation */
      NumericalMathFunction::NumericalMathFunction(const NumericalMathFunctionImplementation & implementation)
	: Common::TypedInterfaceObject<NumericalMathFunctionImplementation>(implementation.clone())
      {
	// Nothing to do
      }

      /* Constructor from implementation */
      NumericalMathFunction::NumericalMathFunction(const Implementation & p_implementation)
	: Common::TypedInterfaceObject<NumericalMathFunctionImplementation>(p_implementation)
      {
	// Nothing to do
      }

      /* Constructor from implementation pointer */
      NumericalMathFunction::NumericalMathFunction(NumericalMathFunctionImplementation * p_implementation)
	: Common::TypedInterfaceObject<NumericalMathFunctionImplementation>(p_implementation)
      {
	// Nothing to do
      }


      /* Composition constructor */
      NumericalMathFunction::NumericalMathFunction(const NumericalMathFunction & left,
						   const NumericalMathFunction & right)
	: Common::TypedInterfaceObject<NumericalMathFunctionImplementation>(new ComposedNumericalMathFunction(left.getImplementation(), right.getImplementation()))
      {
	// Nothing to do
      }

      /* Analytical formula constructor */
      NumericalMathFunction::NumericalMathFunction(const Description & inputVariablesNames,
						   const Description & outputVariablesNames,
						   const Description & formulas)
	: Common::TypedInterfaceObject<NumericalMathFunctionImplementation>(new NumericalMathFunctionImplementation(inputVariablesNames, outputVariablesNames, formulas))
      {
	// Nothing to do
      }

      /* Indicator function constructor */
      NumericalMathFunction::NumericalMathFunction(const NumericalMathFunction & function,
						   const ComparisonOperator & comparisonOperator,
						   const NumericalScalar threshold)
	: Common::TypedInterfaceObject<NumericalMathFunctionImplementation>(new NumericalMathFunctionImplementation(new IndicatorNumericalMathEvaluationImplementation(function.getEvaluationImplementation(), comparisonOperator, threshold), new NoNumericalMathGradientImplementation(), new NoNumericalMathHessianImplementation()))
      {
	// Nothing to do
      }

      /* Linear combination function constructor */
      NumericalMathFunction::NumericalMathFunction(const NumericalMathFunctionCollection & functionCollection,
						   const NumericalPoint & coefficients)
	: Common::TypedInterfaceObject<NumericalMathFunctionImplementation>(new NumericalMathFunctionImplementation())
      {
	LinearCombinationEvaluationImplementation evaluation(functionCollection, coefficients);
	setEvaluationImplementation(evaluation.clone());
	setGradientImplementation(new LinearCombinationGradientImplementation(evaluation));
	setHessianImplementation(new LinearCombinationHessianImplementation(evaluation));
      }

      /* Simplified analytical formula constructor */
      NumericalMathFunction::NumericalMathFunction(const String & inputVariableName,
						   const String & formula,
						   const String & outputVariableName)
	: Common::TypedInterfaceObject<NumericalMathFunctionImplementation>(new NumericalMathFunctionImplementation(Description(1, inputVariableName), Description(1, outputVariableName), Description(1, formula)))
      {
	// Nothing to do
      }

      /* Constructor from implementations */
      NumericalMathFunction::NumericalMathFunction(const EvaluationImplementation & evaluationImplementation,
						   const GradientImplementation & gradientImplementation,
						   const HessianImplementation  & hessianImplementation)
	: Common::TypedInterfaceObject<NumericalMathFunctionImplementation>(new NumericalMathFunctionImplementation(evaluationImplementation, gradientImplementation, hessianImplementation))
      {
	// Nothing to do
      }

      /* Constructor from a wrapper file */
      NumericalMathFunction::NumericalMathFunction(const WrapperFile & wrapperFile)
	: Common::TypedInterfaceObject<NumericalMathFunctionImplementation>(new NumericalMathFunctionImplementation(wrapperFile))
      {
	// Nothing to do
      }

      /* Comparison operator */
      Bool NumericalMathFunction::operator ==(const NumericalMathFunction & other) const
      {
	return true;
      }
  
      /* String converter */
      String NumericalMathFunction::__repr__() const {
	return OSS() << "class=" << NumericalMathFunction::GetClassName()
		     << " name=" << getName()
		     << " implementation=" << getImplementation()->__repr__();
      }
  
      /* Description Accessor */
      void NumericalMathFunction::setDescription(const Description & description)
      {
	getImplementation()->setDescription(description);
      }

      /* Description Accessor */
      NumericalMathFunction::Description NumericalMathFunction::getDescription() const
      {
	return getImplementation()->getDescription();
      }

      /* Input description Accessor */
      NumericalMathFunction::Description NumericalMathFunction::getInputDescription() const
      {
	return getImplementation()->getInputDescription();
      }

      /* Output description Accessor */
      NumericalMathFunction::Description NumericalMathFunction::getOutputDescription() const
      {
	return getImplementation()->getOutputDescription();
      }

      /* Enable or disable the internal cache */
      void NumericalMathFunction::enableCache() const
      {
	getImplementation()->enableCache();
      }

      void NumericalMathFunction::disableCache() const
      {
	getImplementation()->disableCache();
      }

      Bool NumericalMathFunction::isCacheEnabled() const
      {
	return getImplementation()->isCacheEnabled();
      }

      /* Multiplication operator between two functions with the same input dimension and 1D output dimension */
      NumericalMathFunction NumericalMathFunction::operator * (const NumericalMathFunction & right) const
      {
	return getImplementation()->operator * (*(right.getImplementation()));
      }

      /* Function implementation accessors */
      void NumericalMathFunction::setEvaluationImplementation(const EvaluationImplementation & functionImplementation)
      {
	getImplementation()->setEvaluationImplementation(functionImplementation);
      }

      /* Function implementation accessors */
      const NumericalMathFunction::EvaluationImplementation & NumericalMathFunction::getEvaluationImplementation() const
      {
	return getImplementation()->getEvaluationImplementation();
      }

      
      /* Gradient implementation accessors */
      void NumericalMathFunction::setGradientImplementation(const NumericalMathGradientImplementation & gradientImplementation)
      {
	getImplementation()->setGradientImplementation(GradientImplementation(gradientImplementation.clone()));
      }

      /* Gradient implementation accessors */
      void NumericalMathFunction::setGradientImplementation(const GradientImplementation & gradientImplementation)
      {
	getImplementation()->setGradientImplementation(gradientImplementation);
      }

      /* Gradient implementation accessors */
      const NumericalMathFunction::GradientImplementation & NumericalMathFunction::getGradientImplementation() const
      {
	return getImplementation()->getGradientImplementation();
      }

      
      /* Hessian implementation accessors */
      void NumericalMathFunction::setHessianImplementation(const NumericalMathHessianImplementation & hessianImplementation)
      {
	getImplementation()->setHessianImplementation(HessianImplementation(hessianImplementation.clone()));
      }

      void NumericalMathFunction::setHessianImplementation(const HessianImplementation & hessianImplementation)
      {
	getImplementation()->setHessianImplementation(hessianImplementation);
      }

      /* Hessian implementation accessors */
      const NumericalMathFunction::HessianImplementation & NumericalMathFunction::getHessianImplementation() const
      {
	return getImplementation()->getHessianImplementation();
      }

      /* Initial Function implementation accessors */
      const NumericalMathFunction::EvaluationImplementation & NumericalMathFunction::getInitialEvaluationImplementation() const
      {
	return getImplementation()->getInitialEvaluationImplementation();
      }

      /* Initial gradient implementation accessors */
      const NumericalMathFunction::GradientImplementation & NumericalMathFunction::getInitialGradientImplementation() const
      {
	return getImplementation()->getInitialGradientImplementation();
      }

      /* Initial hessian implementation accessors */
      const NumericalMathFunction::HessianImplementation & NumericalMathFunction::getInitialHessianImplementation() const
      {
	return getImplementation()->getInitialHessianImplementation();
      }


      /** Gradient according to the marginal parameters */
      NumericalMathFunction::Matrix NumericalMathFunction::parametersGradient(const NumericalPoint & in) const
      {
	return getImplementation()->parametersGradient(in);
      }

      /** Parameters value and description accessor */
      NumericalMathFunction::NumericalPointWithDescription NumericalMathFunction::getParameters() const
      {
	return getImplementation()->getParameters();
      }

      void NumericalMathFunction::setParameters(const NumericalPointWithDescription & parameters)
      {
	getImplementation()->setParameters(parameters);
      }

	
      /* Operator () */
      NumericalMathFunction::NumericalPoint NumericalMathFunction::operator() (const NumericalPoint & in) const
	throw(InvalidArgumentException,InternalException)
      {
	return getImplementation()->operator()(in);
      }

      /* Operator () */
      NumericalMathFunction::NumericalSample NumericalMathFunction::operator() (const NumericalSample & inSample) const
	throw(InvalidArgumentException,InternalException)
      {
	return getImplementation()->operator()(inSample);
      }

      /* Method gradient() returns the Jacobian transposed matrix of the function at point */
      NumericalMathFunction::Matrix NumericalMathFunction::gradient(const NumericalPoint & in) const
	throw(InvalidArgumentException,InternalException)
      {
	return getImplementation()->gradient(in);
      }

      /* Method hessian() returns the symmetric tensor of the function at point */
      NumericalMathFunction::SymmetricTensor NumericalMathFunction::hessian(const NumericalPoint & in) const
	throw(InvalidArgumentException,InternalException)
      {
	return getImplementation()->hessian(in);
      }




      /* Accessor for input point dimension */
      UnsignedLong NumericalMathFunction::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return getImplementation()->getInputNumericalPointDimension();
      }
      
      /* Accessor for output point dimension */
      UnsignedLong NumericalMathFunction::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return getImplementation()->getOutputNumericalPointDimension();

      }
     
      /* Accessor for input point dimension */
      UnsignedLong NumericalMathFunction::getInputDimension() const
	throw(InternalException)
      {
	return getImplementation()->getInputDimension();
      }
      
      /* Accessor for output point dimension */
      UnsignedLong NumericalMathFunction::getOutputDimension() const
	throw(InternalException)
      {
	return getImplementation()->getOutputDimension();

      }
     
      /* Get the i-th marginal function */
      NumericalMathFunction NumericalMathFunction::getMarginal(const UnsignedLong i) const throw(InvalidArgumentException)
      {
	return *(getImplementation()->getMarginal(i));
      }

      /* Get the function corresponding to indices components */
      NumericalMathFunction NumericalMathFunction::getMarginal(const Indices & indices) const throw(InvalidArgumentException)
      {
	return *(getImplementation()->getMarginal(indices));
      }

      /* Number of calls to the evaluation */
      UnsignedLong NumericalMathFunction::getEvaluationCallsNumber() const
      {
	return getImplementation()->getEvaluationCallsNumber();
      }

      /* Number of calls to the gradient */
      UnsignedLong NumericalMathFunction::getGradientCallsNumber() const
      {
	return getImplementation()->getGradientCallsNumber();
      }

      /* Number of calls to the hessian */
      UnsignedLong NumericalMathFunction::getHessianCallsNumber() const
      {
	return getImplementation()->getHessianCallsNumber();
      }

      /* Static methods for documentation of analytical fnctions */
      NumericalMathFunction::Description  NumericalMathFunction::GetValidConstants()
      {
	return NumericalMathFunctionImplementation::GetValidConstants();
      }

      NumericalMathFunction::Description  NumericalMathFunction::GetValidFunctions()
      {
	return NumericalMathFunctionImplementation::GetValidFunctions();
      }

      NumericalMathFunction::Description  NumericalMathFunction::GetValidOperators()
      {
	return NumericalMathFunctionImplementation::GetValidOperators();
      }



    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
