//                                               -*- C++ -*-
/**
 * @file  IndicatorNumericalMathEvaluationImplementation.cxx
 * @brief The class that implements the composition between numerical
 *        math functions implementations
 *
 * (C) Copyright 2005-2006 EADS
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * \author $LastChangedBy: lebrun $
 * \date   $LastChangedDate: 2008-07-03 08:31:34 +0200 (jeu, 03 jui 2008) $
 */

#include "IndicatorNumericalMathEvaluationImplementation.hxx"
#include "PersistentObjectFactory.hxx"
#include "Less.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      CLASSNAMEINIT(IndicatorNumericalMathEvaluationImplementation);

      static Common::Factory<IndicatorNumericalMathEvaluationImplementation> RegisteredFactory("IndicatorNumericalMathEvaluationImplementation");

      /* Default constructor */
      IndicatorNumericalMathEvaluationImplementation::IndicatorNumericalMathEvaluationImplementation()
	: NumericalMathEvaluationImplementation(),
	  p_evaluation_(),
	  comparisonOperator_(),
	  threshold_(0.0)
      {
	// Nothing to do
      } // IndicatorNumericalMathEvaluationImplementation

      /* Default constructor */
      IndicatorNumericalMathEvaluationImplementation::IndicatorNumericalMathEvaluationImplementation(const EvaluationImplementation & p_evaluation,
												     const ComparisonOperator & comparisonOperator,
												     const NumericalScalar threshold)
	throw(InvalidArgumentException)
	: NumericalMathEvaluationImplementation(),
	  p_evaluation_(),
	  comparisonOperator_(comparisonOperator),
	  threshold_(threshold)
      {
	setEvaluationImplementation(p_evaluation);
	setDescription(p_evaluation->getDescription());
      } // IndicatorNumericalMathEvaluationImplementation


      /* Virtual constructor */
      IndicatorNumericalMathEvaluationImplementation * IndicatorNumericalMathEvaluationImplementation::clone() const
      {
	return new IndicatorNumericalMathEvaluationImplementation(*this);
      }


      /* Comparison operator */
      Bool IndicatorNumericalMathEvaluationImplementation::operator ==(const IndicatorNumericalMathEvaluationImplementation & other) const
      {
	return true;
      }
  
      /* String converter */
      String IndicatorNumericalMathEvaluationImplementation::__repr__() const {
	OSS oss;
	oss << "class=" << IndicatorNumericalMathEvaluationImplementation::GetClassName()
	    << " name=" << getName()
	    << " evaluation=" << p_evaluation_->__repr__()
	    << " comparisonOperator=" << comparisonOperator_
	    << " threshold=" << threshold_;
	return oss;
      }
  
      /* Operator () */
      IndicatorNumericalMathEvaluationImplementation::NumericalPoint IndicatorNumericalMathEvaluationImplementation::operator() (const NumericalPoint & in) const
	throw(InvalidArgumentException, InternalException)
      {
	return NumericalPoint(1, (comparisonOperator_.compare(p_evaluation_->operator()(in)[0], threshold_) ? 1.0 : 0.0));
      }

      /* Accessor for input point dimension */
      UnsignedLong IndicatorNumericalMathEvaluationImplementation::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return p_evaluation_->getInputNumericalPointDimension();
      }
      
      /* Accessor for output point dimension */
      UnsignedLong IndicatorNumericalMathEvaluationImplementation::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return 1;
      }
     
      /* Accessor for the underlying evaluation */
      IndicatorNumericalMathEvaluationImplementation::EvaluationImplementation IndicatorNumericalMathEvaluationImplementation::getEvaluationImplementation() const
      {
	return p_evaluation_;
      }

      void IndicatorNumericalMathEvaluationImplementation::setEvaluationImplementation(const EvaluationImplementation & p_evaluation)
      {
	if (p_evaluation->getOutputNumericalPointDimension() != 1) throw InvalidArgumentException(HERE) << "Error: cannot use an evaluation implementation with output dimension not equal to 1";
	p_evaluation_ = p_evaluation;
      }

      /* Accessor for the comparison operator */
      IndicatorNumericalMathEvaluationImplementation::ComparisonOperator IndicatorNumericalMathEvaluationImplementation::getComparisonOperator() const
      {
	return comparisonOperator_;
      }

      void IndicatorNumericalMathEvaluationImplementation::setComparisonOperator(const ComparisonOperator & comparisonOperator)
      {
	comparisonOperator_ = comparisonOperator;
      }

      /* Accessor for the threshold */
      NumericalScalar IndicatorNumericalMathEvaluationImplementation::getThreshold() const
      {
	return threshold_;
      }

      void IndicatorNumericalMathEvaluationImplementation::setThreshold(const NumericalScalar threshold)
      {
	threshold_ = threshold;
      }

      /* Method save() stores the object through the StorageManager */
      void IndicatorNumericalMathEvaluationImplementation::save(const StorageManager::Advocate & adv) const
      {
	NumericalMathEvaluationImplementation::save(adv);
	adv.writeValue(*p_evaluation_, StorageManager::MemberNameAttribute, "p_evaluation_");
	adv.writeValue(comparisonOperator_, StorageManager::MemberNameAttribute, "comparisonOperator_");
	adv.writeValue("threshold_", threshold_);
      }

//       /* Method load() reloads the object from the StorageManager */
//       void IndicatorNumericalMathEvaluationImplementation::load(const StorageManager::Advocate & adv)
//       {
// 	NumericalMathEvaluationImplementation::load(adv);
// 	StorageManager::List objList = adv.getList(StorageManager::ObjectEntity);
// 	NumericalMathEvaluationImplementation evaluation;
// 	objList.readValue(evaluation,           StorageManager::MemberNameAttribute, "p_evaluation_");
// 	objList.readValue(comparisonOperator_, StorageManager::MemberNameAttribute, "comparisonOperator_");

// 	String name;
// 	NumericalScalar value;
// 	objList = adv.getList(StorageManager::NumericalScalarEntity);
// 	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
// 	  if (objList.readValue(name, value)) {
// 	    if (name == "threshold_") threshold_ = value;
// 	  }
// 	}
// 	*this = IndicatorNumericalMathEvaluationImplementation(new NumericalMathEvaluationImplementation(evaluation), comparisonOperator_, threshold_);
//       }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
