//                                               -*- C++ -*-
/**
 *  @file  CenteredFiniteDifferenceGradient.cxx
 *  @brief Class for the creation of a numerical math gradient implementation
 *         form a numerical math evaluation implementation by using centered
 *         finite difference formula.
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: CenteredFiniteDifferenceGradient.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include <cstdlib>

#include "CenteredFiniteDifferenceGradient.hxx"
#include "PersistentObjectFactory.hxx"
#include "ResourceMap.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Diff {

      CLASSNAMEINIT(CenteredFiniteDifferenceGradient);

      typedef Func::NumericalMathEvaluationImplementation NumericalMathEvaluationImplementation;
      typedef Common::ResourceMap                         ResourceMap;

      static Common::Factory<CenteredFiniteDifferenceGradient> RegisteredFactory("CenteredFiniteDifferenceGradient");

      const NumericalScalar CenteredFiniteDifferenceGradient::DefaultEpsilon = 1.0e-5;//strtod(ResourceMap::GetInstance().get("CenteredFiniteDifferenceGradient-DefaultEpsilon").c_str(), NULL);

      /* Default constructor */
      CenteredFiniteDifferenceGradient::CenteredFiniteDifferenceGradient() :
        NumericalMathGradientImplementation()
      {
	// Nothing to do
      }

      /* Parameter constructor */
      CenteredFiniteDifferenceGradient::CenteredFiniteDifferenceGradient(const NumericalPoint & epsilon,
									 const EvaluationImplementation & p_evaluation)
        throw(InvalidDimensionException) :
        NumericalMathGradientImplementation(),
        epsilon_(epsilon),
        p_evaluation_(p_evaluation)
      {
	/* Check if the dimension of the constant term is compatible with the linear and quadratic terms */
        if (epsilon.getDimension() != p_evaluation->getInputNumericalPointDimension()) throw InvalidDimensionException(HERE) << "Epsilon dimension is incompatible with the given evaluation";
	/* Check if any epsilon component is exactly zero */
	for(UnsignedLong i = 0; i < epsilon.getDimension(); i++)
	  {
	    if (epsilon[i] == 0.0) throw InvalidArgumentException(HERE) << "At least one of the components of epsilon is equal to 0.0, namely component " << i;
	  }
      }

      /* Parameter constructor */
      CenteredFiniteDifferenceGradient::CenteredFiniteDifferenceGradient(const NumericalScalar epsilon,
									 const EvaluationImplementation & p_evaluation)
        throw(InvalidDimensionException) :
        NumericalMathGradientImplementation(),
        epsilon_(NumericalPoint(p_evaluation->getInputNumericalPointDimension(), epsilon)),
        p_evaluation_(p_evaluation)
      {
	/* Check if epsilon is exactly zero */
	if (epsilon == 0.0) throw InvalidArgumentException(HERE) << "The given scalar epsilon is equal to 0.0";
      }

      /* Virtual constructor */
      CenteredFiniteDifferenceGradient * CenteredFiniteDifferenceGradient::clone() const
      {
	return new CenteredFiniteDifferenceGradient(*this);
      }

      /* Comparison operator */
      Bool CenteredFiniteDifferenceGradient::operator ==(const CenteredFiniteDifferenceGradient & other) const
      {
	return (epsilon_ == other.getEpsilon());
      }
  
      /* String converter */
      String CenteredFiniteDifferenceGradient::__repr__() const {
	OSS oss;
	oss << "class=" << CenteredFiniteDifferenceGradient::GetClassName()
	    << " name=" << getName()
	    << " epsilon=" << epsilon_.__repr__()
            << " evaluation=" << p_evaluation_->__repr__();
	return oss;
      }
  
      /* Accessor for epsilon */
      CenteredFiniteDifferenceGradient::NumericalPoint CenteredFiniteDifferenceGradient::getEpsilon() const
      {
	return epsilon_;
      }

      /* Accessor for the evaluation */
      CenteredFiniteDifferenceGradient::EvaluationImplementation CenteredFiniteDifferenceGradient::getEvaluation() const
      {
	return p_evaluation_;
      }

      /* Here is the interface that all derived class must implement */
	
      /* Gradient () */
      CenteredFiniteDifferenceGradient::Matrix CenteredFiniteDifferenceGradient::gradient(const NumericalPoint & in) const
	throw(InvalidArgumentException,InternalException)
      {
	UnsignedLong inputDimension(in.getDimension());
	if (inputDimension != epsilon_.getDimension()) throw InvalidArgumentException(HERE) << "Invalid input dimension";
	/* At which points do we have to compute the evaluation for the centered finite difference. We need 2*dim points. */
	Stat::NumericalSample gridPoints(2 * inputDimension, in);
	for(UnsignedLong i = 0; i < inputDimension; i++)
	  {
	    gridPoints[2 * i][i] += epsilon_[i];
	    gridPoints[2 * i + 1][i] -= epsilon_[i];
	  }
	/* Evaluate the evaluation */
	Stat::NumericalSample gridValues(p_evaluation_->operator()(gridPoints));
	/* Compute the gradient */
	Matrix result(p_evaluation_->getInputNumericalPointDimension(), p_evaluation_->getOutputNumericalPointDimension());
	for (UnsignedLong i = 0; i < result.getNbRows(); i++)
	  {
	    for (UnsignedLong j = 0; j < result.getNbColumns(); j++)
	      {
		/* result(i, j) = (f_j(x + e_i) - f_j(x - e_i)) / (2 * e_i) ~ df_j / dx_i */
		result(i, j) = (gridValues[2 * i][j] - gridValues[2 * i + 1][j]) / (2.0 * epsilon_[i]);
	      }
	  }
	return result;
      }

      /* Accessor for input point dimension */
      UnsignedLong CenteredFiniteDifferenceGradient::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return p_evaluation_->getInputNumericalPointDimension();
      }
      
      /* Accessor for output point dimension */
      UnsignedLong CenteredFiniteDifferenceGradient::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return p_evaluation_->getOutputNumericalPointDimension();
      }
      
      /* Method save() stores the object through the StorageManager */
      void CenteredFiniteDifferenceGradient::save(const StorageManager::Advocate & adv) const
      {
	NumericalMathGradientImplementation::save(adv);
	adv.writeValue(epsilon_, StorageManager::MemberNameAttribute, "epsilon_");
	adv.writeValue(*p_evaluation_, StorageManager::MemberNameAttribute, "evaluation_");
      }

      /* Method load() reloads the object from the StorageManager */
      void CenteredFiniteDifferenceGradient::load(const StorageManager::Advocate & adv)
      {
	NumericalMathGradientImplementation::load(adv);
	adv.readValue(epsilon_, StorageManager::MemberNameAttribute, "epsilon_");
	Common::TypedInterfaceObject<NumericalMathEvaluationImplementation> evaluation;
	adv.readValue(evaluation, StorageManager::MemberNameAttribute, "evaluation_");
	p_evaluation_ = evaluation.getImplementation();
      }
      
    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
