//                                               -*- C++ -*-
/**
 *  @file  StorageManager.hxx
 *  @brief StorageManager provides an interface for different storage classes
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: StorageManager.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_STORAGEMANAGER_HXX
#define OPENTURNS_STORAGEMANAGER_HXX

#include <iostream>              // for std::ostream
#include <vector>                // for std::vector
#include <map>                   // for std::map
#include "Object.hxx"
#include "Pointer.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {

      /* Forward declaration of PersistentObject */
      class PersistentObject;

      /* Forward declaration of InterfaceObject */
      class InterfaceObject;

      /* Forward declaration of Study */
      class Study;

      /**
       * @class StorageManager
       *
       * StorageManager defines a common interface implemented by
       * different classes that must store and reload object onto
       * a persistent medium
       */

      class StorageManager
	: public Object
      {
	CLASSNAME;
      public:

	/** The way we use the storage manager */
	enum MethodCall { SAVE = 0, LOAD };

	/** The names of entities are listed in this enum */
	enum EntityName { ObjectEntity = 0, NumericalScalarEntity, NumericalComplexEntity,
			  StringEntity, UnsignedLongEntity,
			  BoolEntity, 
			  EntityNameEnumSize };

	/** The names of attributes are listed in this enum */
	enum AttributeName { IdAttribute = 0, ClassAttribute, NameAttribute,
			     SizeAttribute, DimensionAttribute, IndexAttribute,
			     StudyVisibleAttribute, MemberNameAttribute,
			     VersionAttribute, KeyAttribute, ValueAttribute,
			     MaxSizeAttribute, 
			     AttributeNameEnumSize };


#ifndef SWIG
	class ListImplementation
	{
	public:
	  virtual ~ListImplementation() {};
	  virtual ListImplementation * clone() const = 0;

	  virtual UnsignedLong getSize() const = 0;
	  virtual void firstValueToRead() = 0;
	  virtual Bool moreValuesToRead() = 0;
	  virtual void nextValueToRead() = 0;
	  virtual Bool readValue(UnsignedLong & index, NumericalScalar & value) = 0;
	  virtual Bool readValue(UnsignedLong & index, NumericalComplex & value) = 0;
	  virtual Bool readValue(UnsignedLong & index, InterfaceObject & obj) = 0;
	  virtual Bool readValue(UnsignedLong & index, PersistentObject & obj) = 0;
	  virtual Bool readValue(UnsignedLong & index, String & st) = 0;
	  virtual Bool readValue(UnsignedLong & index, UnsignedLong & value) = 0;
	  virtual Bool readValue(UnsignedLong & index, Bool & value) = 0;
	  virtual Bool readValue(String & name, NumericalScalar & value) = 0;
	  virtual Bool readValue(String & name, NumericalComplex & value) = 0;
	  virtual Bool readValue(String & name, InterfaceObject & obj) = 0;
	  virtual Bool readValue(String & name, PersistentObject & obj) = 0;
	  virtual Bool readValue(String & name, String & st) = 0;
	  virtual Bool readValue(String & name, UnsignedLong & value) = 0;
	  virtual Bool readValue(String & name, Bool & value) = 0;
	  virtual Bool readValue(InterfaceObject & obj) = 0;
	  virtual Bool readValue(InterfaceObject & obj, AttributeName attribute, const String & attributeValue) = 0;
	  virtual Bool readValue(PersistentObject & obj) = 0;
	  virtual Bool readValue(PersistentObject & obj, AttributeName attribute, const String & attributeValue) = 0;
	  
	}; /* end class List */


	class List
	{
	  Pointer<ListImplementation> p_listimplementation_;

	public:
	  List(const ListImplementation & impl) : p_listimplementation_(impl.clone()) {}

	  inline UnsignedLong getSize() const { return p_listimplementation_->getSize(); }
	  inline void firstValueToRead() { p_listimplementation_->firstValueToRead(); }
	  inline Bool moreValuesToRead() { return p_listimplementation_->moreValuesToRead(); }
	  inline void nextValueToRead() { p_listimplementation_->nextValueToRead(); }
	  // template <class T> Bool readValue(UnsignedLong & index, T & value) const { return false; }
	  inline Bool readValue(UnsignedLong & index, NumericalScalar & value) { return p_listimplementation_->readValue(index, value); }
	  inline Bool readValue(UnsignedLong & index, NumericalComplex & value) { return p_listimplementation_->readValue(index, value); }
	  inline Bool readValue(UnsignedLong & index, InterfaceObject & obj) { return p_listimplementation_->readValue(index, obj); }
	  inline Bool readValue(UnsignedLong & index, PersistentObject & obj) { return p_listimplementation_->readValue(index, obj); }
	  inline Bool readValue(UnsignedLong & index, String & st) { return p_listimplementation_->readValue(index, st); }
	  inline Bool readValue(UnsignedLong & index, UnsignedLong & value) { return p_listimplementation_->readValue(index, value); }
	  inline Bool readValue(UnsignedLong & index, Bool & value) { return p_listimplementation_->readValue(index, value); }
	  inline Bool readValue(String & name, NumericalScalar & value) { return p_listimplementation_->readValue(name, value); }
	  inline Bool readValue(String & name, NumericalComplex & value) { return p_listimplementation_->readValue(name, value); }
	  inline Bool readValue(String & name, InterfaceObject & obj) { return p_listimplementation_->readValue(name, obj); }
	  inline Bool readValue(String & name, PersistentObject & obj) { return p_listimplementation_->readValue(name, obj); }
	  inline Bool readValue(String & name, String & st) { return p_listimplementation_->readValue(name, st); }
	  inline Bool readValue(String & name, UnsignedLong & value) { return p_listimplementation_->readValue(name, value); }
	  inline Bool readValue(String & name, Bool & value) { return p_listimplementation_->readValue(name, value); }
	  inline Bool readValue(InterfaceObject & obj) { return p_listimplementation_->readValue(obj); }
	  inline Bool readValue(InterfaceObject & obj, AttributeName attribute, const String & attributeValue) { return p_listimplementation_->readValue(obj, attribute, attributeValue); }
	  inline Bool readValue(PersistentObject & obj) { return p_listimplementation_->readValue(obj); }
	  inline Bool readValue(PersistentObject & obj, AttributeName attribute, const String & attributeValue) { return p_listimplementation_->readValue(obj, attribute, attributeValue); }
	  
	}; /* end class List */


	class AdvocateImplementation
	{
	  friend class StorageManager;

	  /** The manager we talk with */
	  StorageManager * p_manager_;

	protected:

	  AdvocateImplementation(StorageManager * p_mgr);

	public:
	  AdvocateImplementation(const AdvocateImplementation & other);
	  virtual AdvocateImplementation * clone() const;
	  virtual ~AdvocateImplementation();

	  StorageManager * getManager() const;

	  /** List methods */
	  List getList(EntityName entity) const;

	  /** Methods to write out class members */
	  void writeAttribute(AttributeName attribute, const String & st) const;
	  void writeAttribute(AttributeName attribute, UnsignedLong ul) const;

	  // template <class T> void writeValue(UnsignedLong index, T value) const
	  // { return p_manager_->writeValue(index,value); }
	  void writeValue(UnsignedLong index, NumericalScalar value) const;
	  void writeValue(UnsignedLong index, NumericalComplex value) const;
	  void writeValue(UnsignedLong index, const InterfaceObject & obj) const;
	  void writeValue(UnsignedLong index, const PersistentObject & obj) const;
	  void writeValue(UnsignedLong index, const String & st) const;
	  void writeValue(UnsignedLong index, UnsignedLong value) const;
	  void writeValue(UnsignedLong index, Bool value) const;
	  void writeValue(const String & name, NumericalScalar value) const;
	  void writeValue(const String & name, NumericalComplex value) const;
	  void writeValue(const String & name, const InterfaceObject & obj) const;
	  void writeValue(const String & name, const PersistentObject & obj) const;
	  void writeValue(const String & name, const String & st) const;
	  void writeValue(const String & name, UnsignedLong value) const;
	  void writeValue(const String & name, Bool value) const;
	  void writeValue(const InterfaceObject & obj) const;
	  void writeValue(const InterfaceObject & obj, AttributeName attribute, const String & st) const;
	  void writeValue(const PersistentObject & obj) const;
	  void writeValue(const PersistentObject & obj, AttributeName attribute, const String & st) const;

	  /** Methods to read class members */
	  void readAttribute(AttributeName attribute, String & st) const;
	  void readAttribute(AttributeName attribute, UnsignedLong & ul) const;

	  // template <class T> void readValue(UnsignedLong & index, T & value) const
	  // { return p_manager_->readValue(index,value); }
	  void readValue(UnsignedLong & index, NumericalScalar & value) const;
	  void readValue(UnsignedLong & index, NumericalComplex & value) const;
	  void readValue(UnsignedLong & index, InterfaceObject & obj) const;
	  void readValue(UnsignedLong & index, PersistentObject & obj) const;
	  void readValue(UnsignedLong & index, String & st) const;
	  void readValue(UnsignedLong & index, UnsignedLong & value) const;
	  void readValue(UnsignedLong & index, Bool & value) const;
	  void readValue(String & name, NumericalScalar & value) const;
	  void readValue(String & name, NumericalComplex & value) const;
	  void readValue(String & name, InterfaceObject & obj) const;
	  void readValue(String & name, PersistentObject & obj) const;
	  void readValue(String & name, String & st) const;
	  void readValue(String & name, UnsignedLong & value) const;
	  void readValue(String & name, Bool & value) const;
	  void readValue(InterfaceObject & obj, AttributeName attribute, const String & st) const;
	  void readValue(PersistentObject & obj, AttributeName attribute, const String & st) const;

	}; /* end class Advocate */

	friend class Advocate;



	class WriteAdvocateImplementation : public AdvocateImplementation
	{
	  friend class StorageManager;

	  const PersistentObject & obj_;

	  WriteAdvocateImplementation(StorageManager * p_mgr, const PersistentObject & obj, bool fromStudy = false);

	public:
	  virtual AdvocateImplementation * clone() const;
	  virtual ~WriteAdvocateImplementation();

	}; /* end class WriteAdvocateImplementation */

	friend class WriteAdvocateImplementation;



	class ReadAdvocateImplementation : public AdvocateImplementation
	{
	  friend class StorageManager;

	  ReadAdvocateImplementation(StorageManager * p_mgr);

	public:
	  virtual AdvocateImplementation * clone() const;
	  virtual ~ReadAdvocateImplementation();

	}; /* end class ReadAdvocateImplementation */

	friend class ReadAdvocateImplementation;

	/**
	 * This method build a new advocate to talk to the storage manager
	 */
	Pointer<WriteAdvocateImplementation> registerObject(const PersistentObject & obj, bool fromStudy = false) const;

	/**
	 * This method build a new advocate to talk to the storage manager
	 */
	Pointer<ReadAdvocateImplementation> readObject() const;




	class Advocate
	{
	public:
	  typedef Pointer<AdvocateImplementation> Implementation;

	  Advocate(const Implementation & p_implementation);
	  virtual ~Advocate();

	  StorageManager * getManager() const;

	  /** List methods */
	  List getList(EntityName entity) const;

	  /** Methods to write out class members */
	  void writeAttribute(AttributeName attribute, const String & st) const;
	  void writeAttribute(AttributeName attribute, UnsignedLong ul) const;

	  // template <class T> void writeValue(UnsignedLong index, T value) const
	  // { return p_implementation_->writeValue(index,value); }
	  void writeValue(UnsignedLong index, NumericalScalar value) const;
	  void writeValue(UnsignedLong index, NumericalComplex value) const;
	  void writeValue(UnsignedLong index, const InterfaceObject & obj) const;
	  void writeValue(UnsignedLong index, const PersistentObject & obj) const;
	  void writeValue(UnsignedLong index, const String & st) const;
	  void writeValue(UnsignedLong index, UnsignedLong value) const;
	  void writeValue(UnsignedLong index, Bool value) const;
	  void writeValue(const String & name, NumericalScalar value) const;
	  void writeValue(const String & name, NumericalComplex value) const;
	  void writeValue(const String & name, const InterfaceObject & obj) const;
	  void writeValue(const String & name, const PersistentObject & obj) const;
	  void writeValue(const String & name, const String & st) const;
	  void writeValue(const String & name, UnsignedLong value) const;
	  void writeValue(const String & name, Bool value) const;
	  void writeValue(const InterfaceObject & obj) const;
	  void writeValue(const InterfaceObject & obj, AttributeName attribute, const String & st) const;
	  void writeValue(const PersistentObject & obj) const;
	  void writeValue(const PersistentObject & obj, AttributeName attribute, const String & st) const;

	  /** Methods to read class members */
	  void readAttribute(AttributeName attribute, String & st) const;
	  void readAttribute(AttributeName attribute, UnsignedLong & ul) const;

	  // template <class T> void readValue(UnsignedLong & index, T & value) const
	  // { return p_implementation_->readValue(index,value); }
	  void readValue(UnsignedLong & index, NumericalScalar & value) const;
	  void readValue(UnsignedLong & index, NumericalComplex & value) const;
	  void readValue(UnsignedLong & index, InterfaceObject & obj) const;
	  void readValue(UnsignedLong & index, PersistentObject & obj) const;
	  void readValue(UnsignedLong & index, String & st) const;
	  void readValue(UnsignedLong & index, UnsignedLong & value) const;
	  void readValue(UnsignedLong & index, Bool & value) const;
	  void readValue(String & name, NumericalScalar & value) const;
	  void readValue(String & name, NumericalComplex & value) const;
	  void readValue(String & name, InterfaceObject & obj) const;
	  void readValue(String & name, PersistentObject & obj) const;
	  void readValue(String & name, String & st) const;
	  void readValue(String & name, UnsignedLong & value) const;
	  void readValue(String & name, Bool & value) const;
	  void readValue(InterfaceObject & obj, AttributeName attribute, const String & st) const;
	  void readValue(PersistentObject & obj, AttributeName attribute, const String & st) const;

	private:
	  Implementation p_implementation_;

	}; /* end class Advocate */
#endif

	/** 
	 * Default constructor
	 * The default constructor allows the object to be
	 * stored in STL containers like vector or map.
	 * It takes an optional argument interpreted as the
	 * name of the object in a user point of view. This
	 * name is never used in another way than for user
	 * information. This name has no meaning to the platform.
	 */
	explicit StorageManager(UnsignedLong defaultVersion = 1);

	/**
	 * Virtual constructor
	 */
	virtual StorageManager * clone() const;

	/**
	 * String converter
	 * This method shows human readable information on the
	 * internal state of an object. It is used when streaming
	 * the object or for user information.
	 */
	virtual String __repr__() const;

	/**
	 * This method saves the PersistentObject onto the medium
	 */
	virtual void save(const PersistentObject & obj, bool fromStudy = false);

	/**
	 * This method reloads the PersistentObject from the medium
	 */
	virtual void load(Study & study);

	/** Study accessor */
	Study * getStudy() const;
	void setStudy(Study * p_study);

	/** Returns the version of the study that the manager can read/write */
	UnsignedLong getDefaultStudyVersion() const;

	/** Study version accessor */
	UnsignedLong getStudyVersion() const;
	void setStudyVersion(UnsignedLong version);


	/** Do some administrative tasks before saving/reloading */
	virtual void initialize(const MethodCall caller);

	/** Do some administrative tasks after saving/reloading */
	virtual void finalize(const MethodCall caller);

	/** Read and create the internal representation */
	virtual void read();

	/** Write the internal representation */
	virtual void write();


	/** Returns the string associated with the entity */
	String getEntityString(EntityName entity) const;

	/** Returns the string associated with the attribute */
	String getAttributeString(AttributeName attribute) const;

	/** Returns true if an object is already saved */
	Bool isSavedObject(Id id) const;

	/* Tells that an object is saved */
	void markObjectAsSaved(Id id) const;

      protected:

	/** Refresh the status of saved objects */
	void cleanSavedObjects() const;

	/** Query the manager if the version is correct */
	virtual Bool canManageVersion(UnsignedLong version) const;

	/** Methods to control manager internal state */
	virtual void pushState();
	virtual void popState();

	/** List methods */
	virtual List getList(EntityName entity) const;

	/** Methods to write out objects */
	virtual void writeObjectPrologue(const PersistentObject & obj, bool fromStudy = false);
	virtual void writeObjectEpilogue(const PersistentObject & obj);
	
	/** Methods to write out class members */
	virtual void writeAttribute(AttributeName attribute, const String & st);
	virtual void writeAttribute(AttributeName attribute, UnsignedLong ul);

	// template <class T> void writeValue(UnsignedLong index, T value) const
	// { throw NotDefinedException(HERE); }
	virtual void writeValue(UnsignedLong index, NumericalScalar value);
	virtual void writeValue(UnsignedLong index, NumericalComplex value);
	virtual void writeValue(UnsignedLong index, const PersistentObject & obj);
	virtual void writeValue(UnsignedLong index, const String & st);
	virtual void writeValue(UnsignedLong index, UnsignedLong value);
	virtual void writeValue(UnsignedLong index, Bool value);
	virtual void writeValue(const String & name, NumericalScalar value);
	virtual void writeValue(const String & name, NumericalComplex value);
	virtual void writeValue(const String & name, const PersistentObject & obj);
	virtual void writeValue(const String & name, const String & st);
	virtual void writeValue(const String & name, UnsignedLong value);
	virtual void writeValue(const String & name, Bool value);
	virtual void writeValue(const PersistentObject & obj);
	virtual void writeValue(const PersistentObject & obj, AttributeName attribute, const String & st);

	/** Methods to read class members */
	virtual void readAttribute(AttributeName attribute, String & st);
	virtual void readAttribute(AttributeName attribute, UnsignedLong & ul);

	// template <class T> void readValue(UnsignedLong & index, T & value) const
	// { throw NotDefinedException(HERE); }
	virtual void readValue(UnsignedLong & index, NumericalScalar & value);
	virtual void readValue(UnsignedLong & index, NumericalComplex & value);
	virtual void readValue(UnsignedLong & index, InterfaceObject & obj);
	virtual void readValue(UnsignedLong & index, PersistentObject & obj);
	virtual void readValue(UnsignedLong & index, String & st);
	virtual void readValue(UnsignedLong & index, UnsignedLong & value);
	virtual void readValue(UnsignedLong & index, Bool & value);
	virtual void readValue(String & name, NumericalScalar & value);
	virtual void readValue(String & name, NumericalComplex & value);
	virtual void readValue(String & name, InterfaceObject & obj);
	virtual void readValue(String & name, PersistentObject & obj);
	virtual void readValue(String & name, String & st);
	virtual void readValue(String & name, UnsignedLong & value);
	virtual void readValue(String & name, Bool & value);
	virtual void readValue(InterfaceObject & obj, AttributeName attribute, const String & st);
	virtual void readValue(PersistentObject & obj, AttributeName attribute, const String & st);

      private:

	/** A reference to the current study */
	Study * p_study_;

	/** The list of entity names are listed in this vector */
	std::vector<String> entity_;

	/** The list of attribute names are listed in this vector */
	std::vector<String> attribute_;

	/** The default version of the study */
	const UnsignedLong defaultVersion_;

	/** The version of the study actually used */
	UnsignedLong version_;

	/** The map that tells if objects wera already saved */
	mutable std::map<Id, Bool> savedObjects_;

      }; /* class StorageManager */


    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_STORAGEMANAGER_HXX */
