//                                               -*- C++ -*-
/**
 *  @file  ResourceMap.cxx
 *  @brief ResourceMap defines top-most resourceMap strategies
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: ResourceMap.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include <unistd.h>                 // for sysconf
#include "OTconfig.hxx"
#include "OSS.hxx"
#include "ResourceMap.hxx"
#include "Exception.hxx"
#include "Path.hxx"
#include "Log.hxx"
#include "XMLToolbox.hxx"

#ifndef WIN32

#ifndef R_EXECUTABLE_PATH
#error "R_EXECUTABLE_PATH not defined. Check configuration."
#endif

#ifndef XML_SUPPORTED
#error "XML support is mandatory. Check configuration."
#endif

#endif

namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {

      static const char * ConfigurationFileName = "openturns.conf";
      static const char * RootElementName = "openturns-configuration";
      static const char * XMLTag_value = "value";
      static const char * XMLAttr_undefined = "undefined";

      Pointer<ResourceMap> ResourceMap::P_instance_;


      /* Since ResourceMap is a singleton, GetInstance gives access to the object */
      ResourceMap & ResourceMap::GetInstance()
      {
        if (P_instance_.isNull()) P_instance_ = new ResourceMap;
        return *P_instance_;
      }


      /* Default constructor */
      ResourceMap::ResourceMap()
        : std::map<String,String>()
      {
        loadDefaultConfiguration();
        loadConfigurationFile();
      }

      /* Method for retrieving information from the resource map */
      const String ResourceMap::get(KeyType key) const
      {
        std::map<String,String>::const_iterator it = std::map<String,String>::find(key);
        return (it != end()) ? it->second : "undefined";
      }

      UnsignedLong ResourceMap::getAsUnsignedLong(KeyType key) const
      {
	UnsignedLong value;
	String st = get( key );
	std::istringstream iss( st );
	iss >> value;
	return value;
      }

      NumericalScalar ResourceMap::getAsNumericalScalar(KeyType key) const
      {
	NumericalScalar value;
	String st = get( key );
	std::istringstream iss( st );
	iss >> value;
	return value;
      }


      /* Method for setting information into the resource map */
      void ResourceMap::set(KeyType key, MappedType value)
      {
        std::map<String,String>::operator[](key) = value;
      }

      void ResourceMap::setAsUnsignedLong(KeyType key, UnsignedLong value)
      {
	set( key, OSS() << value );
      }

      void ResourceMap::setAsNumericalScalar(KeyType key, NumericalScalar value)
      {
	set( key, OSS() << value );
      }


      /* Begin points to the first pair of the map */
      ResourceMap::iterator ResourceMap::begin()
      {
        return std::map<String,String>::begin();
      }

      ResourceMap::const_iterator ResourceMap::begin() const
      {
        return std::map<String,String>::begin();
      }


      /* End points past the last pair of the map */
      ResourceMap::iterator ResourceMap::end()
      {
        return std::map<String,String>::end();
      }

      ResourceMap::const_iterator ResourceMap::end() const
      {
        return std::map<String,String>::end();
      }

#if defined HAVE_LIBXML2

      /* Update the ResourceMap with information from the configuration file */
      void ResourceMap::readConfigurationFile(const FileName & configurationFile)
      {
        Log::Info(OSS() << "Using configuration file " << configurationFile);

        // Open the configuration file and load it in a DOM document for later analysis
        XMLDoc document( configurationFile );

        // Check it is an OpenTURNS' one
        xmlNodePtr rootElt = xmlDocGetRootElement( document );
        if (rootElt == NULL) throw ConfigurationFileParsingException(HERE) << "Wrapper file has no root element" << configurationFile;
        if (xmlStrcmp( rootElt->name, REINTERPRET_CAST(const xmlChar *, RootElementName) ))
          throw ConfigurationFileParsingException(HERE) << "Wrapper file " << configurationFile
                                                        << " has an invalid root element (" << rootElt->name << ")"
                                                        << " at line " << xmlGetLineNo(rootElt);


        // Walk through the DOM document to read the data
        if (rootElt && rootElt->children) {
          for (xmlNodePtr current = rootElt->children; current; current = current->next) {
            if (XML::IsElement(current)) {
              String key = XML::ToString(current->name);
              String value = XML::GetAttributeByName( current, XMLTag_value );
              if (value.empty()) value = XMLAttr_undefined;
              std::map<String,String>::operator[](key) = value;
            }
          }
        }

      }

#else
      /* Update the ResourceMap with information from the configuration file */
      void ResourceMap::readConfigurationFile(const FileName & configurationFile) {}

#endif

      /* Find the configuration file in specific path (see Path class for algorithm) */
      FileName ResourceMap::findConfigurationFile() const
      {
        Path::DirectoryList dirList = Path::GetConfigDirectoryList();
        return Path::FindFileByNameInDirectoryList(ConfigurationFileName, dirList);
      }

      /* Load the configuration file */
      void ResourceMap::loadConfigurationFile()
      {
	try {
	  readConfigurationFile(findConfigurationFile());
	}
	catch (FileNotFoundException & ex) {
	  Log::Warn(OSS() << "The configuration file has not been found, using default parameters.");
	}
      }

      /* Load the configuration defined at installation time */
      void ResourceMap::loadDefaultConfiguration()
      {
#ifdef R_EXECUTABLE_PATH
        std::map<String,String>::operator[]("R-executable-command")       = R_EXECUTABLE_PATH;
#endif
        std::map<String,String>::operator[]("csv-file-separator")         = ";";
#ifndef WIN32
	std::map<String,String>::operator[]("temporary-directory")       = "/tmp";
        std::map<String,String>::operator[]("parallel-threads")           = ( OSS() << sysconf(_SC_NPROCESSORS_CONF) );
#else
	std::map<String,String>::operator[]("temporary-directory")       = getenv("TEMP");
	std::map<String,String>::operator[]("parallel-threads")          = getenv("NUMBER_OF_PROCESSORS");
#endif
        std::map<String,String>::operator[]("real-regexp-shortcut")       = "\\R";
        std::map<String,String>::operator[]("integer-regexp-shortcut")    = "\\I";
        std::map<String,String>::operator[]("separator-regexp-shortcut")  = "\\S";
        std::map<String,String>::operator[]("real-regexp")                = "([+-]?[ \t]*([0-9]+(\\.[0-9]*)?|\\.[0-9]+)([Ee][+-]?[0-9]+)?)";
        std::map<String,String>::operator[]("integer-regexp")             = "([+-]?[ \t]*[0-9]+)";
        std::map<String,String>::operator[]("separator-regexp")           = "([ \t])";
        std::map<String,String>::operator[]("cache-max-size")             = "1024";

        // OT::Base::Optim::BoundConstrainedAlgorithmImplementation parameters //
        std::map<String,String>::operator[]("BoundConstrainedAlgorithmImplementation-DefaultMaximumEvaluationsNumber")  = "100";
        std::map<String,String>::operator[]("BoundConstrainedAlgorithmImplementation-DefaultMaximumAbsoluteError")      = "1.0e-5";
        std::map<String,String>::operator[]("BoundConstrainedAlgorithmImplementation-DefaultMaximumRelativeError")      = "1.0e-5";
        std::map<String,String>::operator[]("BoundConstrainedAlgorithmImplementation-DefaultMaximumObjectiveError")     = "1.0e-5";
        std::map<String,String>::operator[]("BoundConstrainedAlgorithmImplementation-DefaultMaximumConstraintError")    = "1.0e-5";

        // OT::Base::Optim::SQP::SQP parameters //
        std::map<String,String>::operator[]("SQP-DefaultTau")             = "0.5";
        std::map<String,String>::operator[]("SQP-DefaultOmega")           = "1.0e-4";
        std::map<String,String>::operator[]("SQP-DefaultSmooth")          = "1.2";

        // OT::Base::Optim::TNC::TNC parameters //
        std::map<String,String>::operator[]("TNC-DefaultMaxCGit")         = "50";
        std::map<String,String>::operator[]("TNC-DefaultEta")             = "0.25";
        std::map<String,String>::operator[]("TNC-DefaultStepmx")          = "10.0";
        std::map<String,String>::operator[]("TNC-DefaultAccuracy")        = "1.0e-4";
        std::map<String,String>::operator[]("TNC-DefaultFmin")            = "1.0";
        std::map<String,String>::operator[]("TNC-DefaultRescale")         = "1.3";

        // OT::Base::Optim::AbdoRackwitz::AbdoRackwitz parameters //
        std::map<String,String>::operator[]("AbdoRackwitz-DefaultTau")    = "0.5";
        std::map<String,String>::operator[]("AbdoRackwitz-DefaultOmega")  = "1.0e-4";
        std::map<String,String>::operator[]("AbdoRackwitz-DefaultSmooth") = "1.2";

        // OT::Base::Optim::NearestPointAlgorithmImplementation parameters //
        std::map<String,String>::operator[]("NearestPointAlgorithmImplementation-DefaultLevelValue")                = "0.0";
        std::map<String,String>::operator[]("NearestPointAlgorithmImplementation-DefaultMaximumIterationsNumber")   = "100";
        std::map<String,String>::operator[]("NearestPointAlgorithmImplementation-DefaultMaximumAbsoluteError")      = "1.0e-5";
        std::map<String,String>::operator[]("NearestPointAlgorithmImplementation-DefaultMaximumRelativeError")      = "1.0e-5";
        std::map<String,String>::operator[]("NearestPointAlgorithmImplementation-DefaultMaximumResidualError")      = "1.0e-5";
        std::map<String,String>::operator[]("NearestPointAlgorithmImplementation-DefaultMaximumConstraintError")    = "1.0e-5";

        // OT::Base::Optim::Cobyla::Cobyla parameters //
        std::map<String,String>::operator[]("Cobyla-DefaultRhoBeg")       = "0.1";

        // OT::Base::Solver::SolverImplementation parameters //
        std::map<String,String>::operator[]("SolverImplementation-DefaultAbsoluteError")             = " 1e-5";
        std::map<String,String>::operator[]("SolverImplementation-DefaultRelativeError")             = " 1e-5";
        std::map<String,String>::operator[]("SolverImplementation-DefaultMaximumFunctionEvaluation") = " 100";

        // OT::Base::Graph::Pie parameters //
        std::map<String,String>::operator[]("Pie-HorizontalMargin")       = "0.3";
        std::map<String,String>::operator[]("Pie-VerticalMargin")         = "0.1";
        std::map<String,String>::operator[]("Pie-LabelThreshold")         = "0.02";

        // OT::Base::Graph::DrawableImplementation parameters //
        std::map<String,String>::operator[]("DrawableImplementation-NoSpecifiedLabel") = "";
        std::map<String,String>::operator[]("DrawableImplementation-DefaultColor") = "blue";
        std::map<String,String>::operator[]("DrawableImplementation-DefaultSurfaceColor") = "white";
        std::map<String,String>::operator[]("DrawableImplementation-DefaultFillStyle") = "solid";
        std::map<String,String>::operator[]("DrawableImplementation-DefaultPointStyle") = "plus";
        std::map<String,String>::operator[]("DrawableImplementation-DefaultLineStyle") = "solid";
        std::map<String,String>::operator[]("DrawableImplementation-DefaultPattern") = "s";
        std::map<String,String>::operator[]("DrawableImplementation-AlternativePattern") = "S";
        std::map<String,String>::operator[]("DrawableImplementation-DefaultLineWidth") = "1";
        std::map<String,String>::operator[]("DrawableImplementation-DataThreshold")    = "2000";

        // OT::Base::Graph::GraphImplementation parameters //
        std::map<String,String>::operator[]("GraphImplementation-NoSpecifiedLabel")                 = "";
        std::map<String,String>::operator[]("GraphImplementation-DefaultWidth")                     = "640";
        std::map<String,String>::operator[]("GraphImplementation-DefaultHeight")                    = "480";
        std::map<String,String>::operator[]("GraphImplementation-DefaultLegendFontSize")            = "1.0";

        // OT::Base::Graph::Contour parameters //
        std::map<String,String>::operator[]("Contour-DefaultLevelsNumber")             = "10";

        // OT::Base::Stat::LinearModelFactory parameters //
        std::map<String,String>::operator[]("LinearModelFactory-DefaultLevelValue")    = "0.95";

        // OT::Base::Stat::Last parameters //
        std::map<String,String>::operator[]("Last-DefaultMaximumSize")    = "65536";

        // OT::Base::Stat::Compact parameters //
        std::map<String,String>::operator[]("Compact-DefaultHalfMaximumSize")          = "1024";

        // OT::Base::Stat::SobolSequence parameters //
        std::map<String,String>::operator[]("SobolSequence-InitialSeed")  = "1";

        // OT::Base::Stat::CorrelationAnalysis parameters //
        std::map<String,String>::operator[]("CorrelationAnalysis-DefaultMaximumOrder")                     = "0";
        std::map<String,String>::operator[]("CorrelationAnalysis-DefaultTotalIndiceComputation")           = "false";

        // OT::Base::Stat::SensitivityAnalysis parameters //
        std::map<String,String>::operator[]("SensitivityAnalysis-DefaultSampleSize")          = "1000";

        // OT::Base::Stat::RandomGenerator parameters //
        std::map<String,String>::operator[]("RandomGenerator-InitialSeed")                = "0";

        // OT::Base::Diff::CenteredFiniteDifferenceGradient parameters //
        std::map<String,String>::operator[]("CenteredFiniteDifferenceGradient-DefaultEpsilon")             = "1.0e-5";

        // OT::Base::Diff::CenteredFiniteDifferenceHessian parameters //
        std::map<String,String>::operator[]("CenteredFiniteDifferenceHessian-DefaultEpsilon")             = "1.0e-4";

        // OT::Base::Diff::NonCenteredFiniteDifferenceGradient parameters //
        std::map<String,String>::operator[]("NonCenteredFiniteDifferenceGradient-DefaultEpsilon")             = "1.0e-7";

        // OT::Base::Type::MatrixImplementation parameters //
        std::map<String,String>::operator[]("MatrixImplementation-DefaultSmallPivot")          = "1.0e-7";

        // OT::Uncertainty::Distribution::Normal parameters //
        std::map<String,String>::operator[]("Normal-MaximumNumberOfPoints")            = "10000000";
        std::map<String,String>::operator[]("Normal-MinimumNumberOfPoints")            = "1000000";
        std::map<String,String>::operator[]("Normal-MaximumCDFEpsilon")                = "5.0e-6";
        std::map<String,String>::operator[]("Normal-MinimumCDFEpsilon")                = "5.0e-2";

        // OT::Uncertainty::Distribution::TruncatedDistribution parameters //
        std::map<String,String>::operator[]("TruncatedDistribution-DefaultThresholdRealization")      = "0.5";

        // OT::Uncertainty::Distribution::Student parameters //
        std::map<String,String>::operator[]("Student-MaximumNumberOfPoints")            = "10000000";
        std::map<String,String>::operator[]("Student-MinimumNumberOfPoints")            = "1000000";
        std::map<String,String>::operator[]("Student-MaximumCDFEpsilon")                = "5.0e-6";
        std::map<String,String>::operator[]("Student-MinimumCDFEpsilon")                = "5.0e-2";

        // OT::Uncertainty::Distribution::FrankCopulaFactory parameters //
        std::map<String,String>::operator[]("FrankCopulaFactory-ThetaEpsilon")                     = "1e-14";

        // OT::Uncertainty::Distribution::RandomMixture parameters //
        std::map<String,String>::operator[]("RandomMixture-DefaultBlockMin")                  = "3";
        std::map<String,String>::operator[]("RandomMixture-DefaultBlockMax")                  = "16";
        std::map<String,String>::operator[]("RandomMixture-DefaultMaxSize")                   = "65536";
        std::map<String,String>::operator[]("RandomMixture-DefaultAlpha")                     = "5.0";
        std::map<String,String>::operator[]("RandomMixture-DefaultPDFEpsilon")                = "1.0e-10";
        std::map<String,String>::operator[]("RandomMixture-DefaultCDFEpsilon")                = "1.0e-10";
        std::map<String,String>::operator[]("RandomMixture-GraphPDFEpsilon")                  = "1.0e-5";
        std::map<String,String>::operator[]("RandomMixture-GraphCDFEpsilon")                  = "1.0e-5";
        std::map<String,String>::operator[]("RandomMixture-SmallSize")                        = "100";
        std::map<String,String>::operator[]("RandomMixture-DefaultSizeProjection")            = "10000";

        // OT::Uncertainty::Distribution::ExtraFunc::SpecFunc parameters //
        std::map<String,String>::operator[]("SpecFunc-Precision")                        = "1.0e-12";

        // OT::Uncertainty::Distribution::ExtraFunc::DistFunc parameters //
        std::map<String,String>::operator[]("DistFunc-Precision")                        = "1.0e-14";
        std::map<String,String>::operator[]("DistFunc-MaximumIteration")                 = "5000";

        // OT::Uncertainty::Algorithm::Simulation::DirectionalSampling::RootStrategyImplementation parameters //
        std::map<String,String>::operator[]("RootStrategyImplementation-DefaultMaximumDistance")               = "8.0";
        std::map<String,String>::operator[]("RootStrategyImplementation-DefaultStepSize")                      = "1.0";

        // OT::Uncertainty::Algorithm::Simulation::Simulation parameters //
        std::map<String,String>::operator[]("Simulation-DefaultMaximumOuterSampling")          = "1000";
        std::map<String,String>::operator[]("Simulation-DefaultMaximumCoefficientOfVariation") = "1.0e-1";
        std::map<String,String>::operator[]("Simulation-DefaultBlockSize")                     = "1";

        // OT::Uncertainty::Algorithm::Simulation::SimulationResultImplementation parameters //
        std::map<String,String>::operator[]("SimulationResultImplementation-DefaultConfidenceLevel")               = "0.95";

        // OT::Uncertainty::Algorithm::Analytical::AnalyticalResult parameters //
        std::map<String,String>::operator[]("AnalyticalResult-DefaultWidth")                         = "1.0";

        // OT::Uncertainty::Algorithm::Analytical::StrongMaximumTest parameters //
        std::map<String,String>::operator[]("StrongMaximumTest-DefaultDeltaPrecision")                = ".01e-7";
        std::map<String,String>::operator[]("StrongMaximumTest-Epsilon")                              = "1.0e-10";

        // OT::Uncertainty::Algorithm::MetaModel::FunctionalChaos::CleaningStrategy parameters //
        std::map<String,String>::operator[]("CleaningStrategy-DefaultMaximumSize")                   = "20";
        std::map<String,String>::operator[]("CleaningStrategy-DefaultSignificanceFactor")            = "1.0e-4";

        // OT::Uncertainty::Algorithm::MetaModel::FunctionalChaos::FunctionalChaosAlgorithm parameters //
        std::map<String,String>::operator[]("FunctionalChaosAlgorithm-DefaultMaximumResidual")               = "1.0e-6";

        // OT::Uncertainty::Algorithm::Experiments::WeightedExperiment parameters //
        std::map<String,String>::operator[]("WeightedExperiment-DefaultSize") = "100";

        // OT::Uncertainty::Model::DistributionImplementation parameters //
        std::map<String,String>::operator[]("DistributionImplementation-DefaultPointNumber") = "129";
        std::map<String,String>::operator[]("DistributionImplementation-DefaultQuantileEpsilon") = "1.0e-12";
        std::map<String,String>::operator[]("DistributionImplementation-DefaultPDFEpsilon") = "1.0e-14";
        std::map<String,String>::operator[]("DistributionImplementation-DefaultCDFEpsilon") = "1.0e-14";
        std::map<String,String>::operator[]("DistributionImplementation-DefaultQuantileIteration") = "10";
        std::map<String,String>::operator[]("DistributionImplementation-QMin") = " 1.5e-1";
        std::map<String,String>::operator[]("DistributionImplementation-QMax") = " 8.5e-1";
        std::map<String,String>::operator[]("DistributionImplementation-DefaultIntegrationNodesNumber") = "255";
        std::map<String,String>::operator[]("DistributionImplementation-DefaultLevelNumber") = "10";

        // OT::Uncertainty::Model::ContinuousDistribution parameters //
        std::map<String,String>::operator[]("ContinuousDistribution-DefaultIntegrationNodesNumber") = "21";

        // OT::Uncertainty::Model::DiscreteDistribution parameters //
        std::map<String,String>::operator[]("DiscreteDistribution-SupportEpsilon") = "1.0e-14";
      }

      /* String converter */
      String ResourceMap::__repr__() const
      {
        OSS oss;
        oss << "ResourceMap={";
        ResourceMap::const_iterator it;
        const char * separator = "\n  ";
        for(it=begin(); it!=end(); it++, separator=",\n  ")
          oss << separator
              << it->first << " => " << it->second;
        oss << "\n}";
        return oss;
      }



      /*
       * @fn std::ostream & operator <<(std::ostream & os, const ResourceMap & obj)
       * @brief Output stream converter
       * @param os A STL output stream resourceMap
       * @param obj The resourceMap read by \em os
       * @return A reference to \em os
       *
       * Operator << converts the ResourceMap object to an output stream
       * so it is easy to show the content of the resourceMap.
       */
      std::ostream & operator <<(std::ostream & os, const ResourceMap & obj)
      {
        return os << obj.__repr__();
      }

      OStream & operator <<(OStream & OS, const ResourceMap & obj)
      {
        return OS << obj.__repr__();
      }





    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */
