//                                               -*- C++ -*-
/**
 *  @file  Path.hxx
 *  @brief This class provides all the mechanism of path manipulation
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-06-15 16:14:58 +0200 (lun. 15 juin 2009) $
 *  Id:      $Id: Path.hxx 1272 2009-06-15 14:14:58Z dutka $
 */
#ifndef OPENTURNS_PATH_HXX
#define OPENTURNS_PATH_HXX

#include <iostream>              // for std::ostream
#include <vector>                // for std::vector
#include "OTprivate.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {

      /**
       * @class Path
       *
       * This class provides all the mechanism of path manipulation
       */

      class Path
      {
      public:

	typedef std::vector<FileName> DirectoryList;

	/**
	 * Analyse the process environment and return the directory where
	 * installed wrappers and DTD are.
	 *
	 * The search rule is: check the following paths, in that order:
	 *    + if the env var OPENTURNS_HOME exists, return ${OPENTURNS_HOME}/share/openturns/wrappers
	 *      if it exists and is a directory,
	 *    + otherwise return the installation path ${prefix}/share/openturns/wrappers, where 'prefix'
	 *      is the installation path of the platform as provided at configuration time.
	 */
	static FileName GetStandardWrapperDirectory();

	/**
	 * Analyse the process environment
	 * and return a list of directories to search in for wrappers.
	 *
	 * The search rule is :look for the file in the following directories, in that order :
	 *    + if the env var OPENTURNS_WRAPPER_PATH exists, in directories listed in
	 *      ${OPENTURNS_WRAPPER_PATH} (see openturnsWrapperPathVariableName_)
	 *    + in directory ${HOME}/openturns/wrappers (see homeSubdirectory_)
	 *    + in the standard wrapper directory, as defined by the method
	 *      GetStandardWrapperDirectory().
	 */
	static DirectoryList GetWrapperDirectoryList();

	/**
	 * Analyse the process environment
	 * and return a list of directories to search in for configuration files.
	 *
	 * The search rule is :look for the file in the following directories, in that order :
	 *    + if the env var OPENTURNS_CONFIG_PATH exists, in directories listed in
	 *      ${OPENTURNS_CONFIG_PATH} (see openturnsConfigPathVariableName_)
	 *    + in directory ${HOME}/openturns/etc (see homeSubdirectory_)
	 *    + in standard config directory (either ${OPENTURNS_HOME}/etc/openturns if OPENTURNS_HOME
	 *      is defined and if this path is a directory, or the path ${prefix}/etc/openturns
	 *      where 'prefix' is the installation path of the platform as provided at configuration time.
	 */
	static DirectoryList GetConfigDirectoryList();

	/**
	 * Find a file named 'name' located in one of the directories
	 * listed in 'dirList'. The function returns the full path (directory/name)
	 * of the first match.
	 */
	static FileName FindFileByNameInDirectoryList(const FileName & name, const DirectoryList & dirList)
	  throw(FileNotFoundException);

	/** Build a temporary file name given a pattern */
	static String BuildTemporaryFileName(const String & pattern);


      protected:

	/**
	 * @var static const String Path::OpenturnsWrapperPathVariableName_
	 *
	 * The environment variable name
	 */
	static const String OpenturnsWrapperPathVariableName_;

	/**
	 * @var static const String Path::OpenturnsConfigPathVariableName_
	 *
	 * The environment variable name
	 */
	static const String OpenturnsConfigPathVariableName_;

	/**
	 * @var static const String Path::HomeWrapperSubdirectory_
	 *
	 * The HOME subdirectory path
	 */
	static const String HomeWrapperSubdirectory_;

	/**
	 * @var static const String Path::HomeConfigSubdirectory_
	 *
	 * The HOME subdirectory path
	 */
	static const String HomeConfigSubdirectory_;

	/**
	 * @var static const String Path::PrefixWrapperSubdirectory_
	 *
	 * The 'prefix' subdirectory path
	 */
	static const String PrefixWrapperSubdirectory_;

	/**
	 * @var static const String Path::PrefixConfigSubdirectory_
	 *
	 * The 'prefix' subdirectory path
	 */
	static const String PrefixConfigSubdirectory_;


      public:

	/** Destructor */
	virtual ~Path();

      protected:

      private:

	/** Default constructor */
	Path();

      }; /* class Path */


    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_PATH_HXX */
