/* -*-c++-*- Producer - Copyright (C) 2001-2004  Don Burns
 *
 * This library is open source and may be redistributed and/or modified under
 * the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
 * (at your option) any later version.  The full license is in LICENSE file
 * included with this distribution, and on the openscenegraph.org website.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * OpenSceneGraph Public License for more details.
 */


#if defined(WIN32) && !defined(__CYGWIN__)
    #include <Io.h>
    #include <Windows.h>
    #include <Winbase.h>
    // set up for windows so acts just like unix access().
    #define F_OK 4
#else // unix
    #include <unistd.h>
#endif


#ifdef _X11_IMPLEMENTATION
#  include <X11/Xlib.h>
#endif

#include <math.h>

#include <Producer/Math>
#include <Producer/CameraConfig>

using namespace Producer;

unsigned int CameraConfig::getNumberOfScreens()
{
    return RenderSurface::getNumberOfScreens();
}

void CameraConfig::rotateCameraOffset( Matrix::value_type deg, Matrix::value_type x, Matrix::value_type y, Matrix::value_type z )
{
    Matrix m;
    m.invert(Matrix::rotate( deg2rad(deg), x,y,z)); 
    m = m * Matrix(_offset_matrix);
    memcpy( _offset_matrix, m.ptr(), sizeof( Matrix::value_type[16] ));
}

void CameraConfig::translateCameraOffset( Matrix::value_type x, Matrix::value_type y, Matrix::value_type z )
{
    Matrix m;
    m.invert(Matrix::translate( x,y,z)); 
    m = m * Matrix(_offset_matrix);
    memcpy( _offset_matrix, m.ptr(), sizeof( Matrix::value_type[16] ));
}

void CameraConfig::scaleCameraOffset( Matrix::value_type x, Matrix::value_type y, Matrix::value_type z )
{
    Matrix m = Matrix::scale( x,y,z) * Matrix(_offset_matrix);
    memcpy( _offset_matrix, m.ptr(), sizeof( Matrix::value_type[16] ));
}

bool CameraConfig::fileExists(const std::string& filename)
{
    return access( filename.c_str(), F_OK ) == 0;
}

// Order of precedence:
//
std::string CameraConfig::findFile( std::string filename )
{
    if (filename.empty()) return filename;

    std::string path;
    // Check env var
    char *ptr = getenv( "PRODUCER_CONFIG_FILE_PATH");
    if( ptr != NULL )
    {
         path = std::string(ptr) + '/' + filename;
        if( fileExists(path)) 
            return path;
    }

    // Check standard location(s)
    //path.clear();
    path = std::string( "/usr/local/share/Producer/Config/") + filename;
    if( fileExists(path) ) 
        return path;

    //path.clear();
    path = std::string( "/usr/share/Producer/Config/") + filename;
    if( fileExists(path) ) 
        return path;

    // Check local directory
    if(fileExists(filename)) 
        return filename;

    // Fail
    return std::string();
}

bool CameraConfig::defaultConfig()
{
    if( getNumberOfCameras() != 0 ) return false;

    char *env = getenv( "PRODUCER_CONFIG_FILE" );
    if( env != NULL )
    {
        std::string file = findFile(env);
        return parseFile( file.c_str() );
    }

    unsigned int numScreens =  getNumberOfScreens();
    if( numScreens == 0 )
        return false;

    float xshear = float(numScreens-1);
    float yshear = 0.0;
    float input_xMin = -1.0f;
    float input_yMin = -1.0f;
    float input_width = 2.0f/float(numScreens);
    float input_height = 2.0f;

    // use an InputArea if there is more than one screen.
    Producer::InputArea *ia = (numScreens>1) ? new Producer::InputArea : 0;
    setInputArea(ia);

    for( unsigned int i = 0; i < numScreens; i++ )
    {
      std::string name = "Screen" + i;
      std::pair<std::map<std::string, Producer::ref_ptr<Camera> >::iterator,bool> res = 
      _camera_map.insert(std::pair<std::string, Producer::ref_ptr<Camera> >(name, new Camera));
      ((res.first)->second)->getRenderSurface()->setScreenNum( i );
      ((res.first)->second)->setLensShear( xshear, yshear );

       RenderSurface *rs = ((res.first)->second)->getRenderSurface();
       rs->setWindowName( name );
       if (ia)
       {
           rs->setInputRectangle( RenderSurface::InputRectangle(input_xMin, input_xMin+input_width, input_yMin, input_yMin+input_height) );
           input_xMin += input_width;
           ia->addRenderSurface(rs);
       }

       _render_surface_map.insert(std::pair<std::string, 
            Producer::ref_ptr<RenderSurface> >( rs->getWindowName(), rs ));

      xshear -= 2.0;
    }

    return true;
}        

