/*
 * Copyright 2008 Red Hat Inc., Durham, North Carolina.
 * All Rights Reserved.
 *
 * OpenScap CPE Lang Module Test Suite Helper
 *
 * Authors:
 *      Ondrej Moris <omoris@redhat.com>
 */

#include <stdio.h>
#include <string.h>
#include <cpelang.h>
#include <cpeuri.h>
#include <oscap.h>

// Strings representing operators.
const char *CPE_OPER_STRS[] = { "", "AND", "OR", "" };

void print_usage(const char *, FILE *);

int print_expr_prefix_form(const struct cpe_testexpr *);

void print_platform(struct cpe_platform *);

int main (int argc, char *argv[]) 
{  
  struct cpe_lang_model *lang_model = NULL;
  struct cpe_platform *platform = NULL, *new_platform = NULL;
  struct cpe_testexpr *testexpr = NULL;
  struct cpe_platform_iterator *platform_it = NULL;
  struct oscap_import_source *import_source = NULL;
  struct oscap_export_target *export_target = NULL;
  struct oscap_title_iterator *title_it = NULL;
  struct oscap_title *title = NULL;
  int ret_val = 0, i;
  
  if (argc == 2 && !strcmp(argv[1], "--help")) {
    print_usage(argv[0], stdout);
    ret_val = 0;
  }

  // Print complete content.
  else if (argc == 4 && !strcmp(argv[1], "--get-all")) {        
    if ((import_source = oscap_import_source_new_file(argv[2], argv[3])) == NULL)
      return 1;
    
    if ((lang_model = cpe_lang_model_import(import_source)) == NULL)
      return 1;

    struct xml_metadata_iterator *xml_it = cpe_lang_model_get_xmlns(lang_model);
    struct xml_metadata *xml;

    while (xml_metadata_iterator_has_more(xml_it)) {
        xml = xml_metadata_iterator_next(xml_it);
        if (strcmp(cpe_lang_model_get_ns_prefix(lang_model), xml_metadata_get_nspace(xml)))
            printf("%s:", xml_metadata_get_URI(xml));
    }

    //printf("%s:", cpe_lang_model_get_ns_href(lang_model));
    printf("%s\n", cpe_lang_model_get_ns_prefix(lang_model));
    platform_it = cpe_lang_model_get_platforms(lang_model);
    while (cpe_platform_iterator_has_more(platform_it)) {
      print_platform(cpe_platform_iterator_next(platform_it));
    }
    cpe_platform_iterator_free(platform_it);
    oscap_import_source_free(import_source);
    cpe_lang_model_free(lang_model);      
  }
  
  // Print platform of given key only.
  else if (argc == 5 && !strcmp(argv[1], "--get-key")) {        
    if ((import_source = oscap_import_source_new_file(argv[2], argv[3])) == NULL)
      return 1;
    
    if ((lang_model = cpe_lang_model_import(import_source)) == NULL)
      return 1;

    if ((platform = cpe_lang_model_get_item(lang_model, argv[4])) == NULL)
      return 2;
    
    print_platform(platform);

    oscap_import_source_free(import_source);
    cpe_lang_model_free(lang_model);      
  }

  // Set ns_prefix, ns_href, add new platforms.
  else if (argc >= 6 && !strcmp(argv[1], "--set-all")) {        
    if ((import_source = oscap_import_source_new_file(argv[2], argv[3])) == NULL)
      return 1;
    if ((lang_model = cpe_lang_model_import(import_source)) == NULL)
      return 1;
    oscap_import_source_free(import_source);
    
    struct xml_metadata *xml = xml_metadata_new();

    char *tmp = malloc(sizeof(char) * (strlen(argv[4])+6));
    if (strcmp(argv[4], "-"))  {
        sprintf(tmp, "xmlns:%s", argv[4]);
        xml_metadata_set_nspace(xml, tmp);
        cpe_lang_model_set_ns_prefix(lang_model, argv[4]);
        free(tmp);
    }
    if (strcmp(argv[5], "-")) {
        xml_metadata_set_URI(xml, argv[5]);
        cpe_lang_model_add_xml(lang_model, xml);
    }

    for (i = 6; i < argc; i++) {
      if ((new_platform =  cpe_platform_new()) == NULL)
	return 1;
      cpe_platform_set_id(new_platform, argv[i]);      
      if (!cpe_lang_model_add_platform(lang_model, new_platform))
	return 2;
    }      

    if ((export_target = oscap_export_target_new_file(argv[2], argv[3])) == NULL)
      return 1;
    cpe_lang_model_export(lang_model, export_target);
    oscap_export_target_free(export_target);
    cpe_lang_model_free(lang_model);
  }

  // Set id, change titles of platform of given key.
  else if (argc >= 6 && !strcmp(argv[1], "--set-key")) {
    if ((import_source = oscap_import_source_new_file(argv[2], argv[3])) == NULL)
      return 1;
    if ((lang_model = cpe_lang_model_import(import_source)) == NULL)
      return 1;
    oscap_import_source_free(import_source);
    
    if ((platform = cpe_lang_model_get_item(lang_model, argv[4])) == NULL)
      return 2;
    
    if (strcmp(argv[5], "-"))
      cpe_platform_set_id(platform, argv[5]);

    i = 6;
    title_it = cpe_platform_get_titles(platform);
    while (i < argc && oscap_title_iterator_has_more(title_it)) {
      title = oscap_title_iterator_next(title_it);
      if (strcmp(argv[i], "-"))
	oscap_title_set_content(title, argv[i]);
      i++;
    } 

    if ((export_target = oscap_export_target_new_file(argv[2], argv[3])) == NULL) 
      return 1;
    cpe_lang_model_export(lang_model, export_target);
    oscap_export_target_free(export_target);
    cpe_lang_model_free(lang_model);
  }

  // Create new content with new platforms.
  else if (argc >= 6 && !strcmp(argv[1], "--set-new")) {        
    if ((lang_model = cpe_lang_model_new()) == NULL)
      return 1;
    
    struct xml_metadata *xml = xml_metadata_new();
    char *tmp = malloc(sizeof(char) * (strlen(argv[4])+7));

    if (strcmp(argv[4], "-"))  {
        sprintf(tmp, "xmlns:%s", argv[4]);
        xml_metadata_set_nspace(xml, tmp);
        cpe_lang_model_set_ns_prefix(lang_model, argv[4]);
        free(tmp);
    }
    if (strcmp(argv[5], "-"))
        xml_metadata_set_URI(xml, argv[5]);

    cpe_lang_model_add_xml(lang_model, xml);
    
    for (i = 6; i < argc; i++) {
      if ((new_platform =  cpe_platform_new()) == NULL)
	return 1;
      cpe_platform_set_id(new_platform, argv[i]);      
	  /*
	  struct cpe_testexpr *expr = cpe_testexpr_new();
	  cpe_testexpr_set_oper(expr, CPE_LANG_OPER_MATCH);
	  cpe_testexpr_set_name(expr, cpe_name_new("cpe:/a:nevim"));
	  cpe_platform_set_expr(new_platform, expr);
	  */
      if (!cpe_lang_model_add_platform(lang_model, new_platform))
	return 2;
    }      

    if ((export_target = oscap_export_target_new_file(argv[2], argv[3])) == NULL) 
      return 1;
    cpe_lang_model_export(lang_model, export_target);
    oscap_export_target_free(export_target);    
    cpe_lang_model_free(lang_model);
  }

  // Sanity checks.
  else if (argc == 2 && !strcmp(argv[1], "--smoke-test")) {        

    if ((lang_model = cpe_lang_model_new()) == NULL) 
      return 1;
    else
      cpe_lang_model_free(lang_model);
    
    if ((new_platform =  cpe_platform_new()) == NULL) 
      return 1;
    else
      cpe_platform_free(new_platform);
    
    if ((testexpr = cpe_testexpr_new()) == NULL) 
      return 1;
    else
      cpe_testexpr_free(testexpr);
  }
  
  else if (argc == 6 && !strcmp(argv[1], "--export-all")) {
    if ((import_source = oscap_import_source_new_file(argv[2], argv[3])) == NULL)
      return 1;

    if ((export_target = oscap_export_target_new_file(argv[4], argv[5])) == NULL)
      return 1;

    if ((lang_model = cpe_lang_model_import(import_source)) == NULL)
      return 1;

    cpe_lang_model_export(lang_model, export_target);

    oscap_import_source_free(import_source);
    oscap_export_target_free(export_target);
    cpe_lang_model_free(lang_model);
  }
  else if (argc == 6 && !strcmp(argv[1], "--match-cpe")) {
    if ((import_source = oscap_import_source_new_file(argv[2], argv[3])) == NULL)
      return 1;

    if ((lang_model = cpe_lang_model_import(import_source)) == NULL)
      return 1;

    struct cpe_name *name1 = NULL;
    struct cpe_name *name2 = NULL;
    char * uri = NULL;

    // make cpe_name
    name1 = cpe_name_new(argv[4]);
    name2 = cpe_name_new(argv[5]);
    if ( (uri = cpe_name_get_uri(name1)) == NULL ) return 1;
    if ( (uri = cpe_name_get_uri(name2)) == NULL ) return 1;
    // actually we need array of cpe_name-s
    struct cpe_name ** names = (struct cpe_name **) malloc(2*sizeof(struct cpe_name *)); // <-- just for clear what I'm doing
    names[0] = name1;
    names[1] = name2;

    // let's get platform cpe's to match
    platform_it = cpe_lang_model_get_platforms(lang_model);
    platform = cpe_platform_iterator_next(platform_it); // we just need first one (no more there)
    cpe_platform_iterator_free(platform_it);
    
    ret_val = !cpe_platform_match_cpe(names, 2, platform);

    cpe_name_free(name1);
    cpe_name_free(name2);
    free(names);

    oscap_import_source_free(import_source);
    cpe_lang_model_free(lang_model);

    return ret_val;
  }
  else {
    print_usage(argv[0], stderr);
    ret_val = 1;
  }

  oscap_cleanup();

  return ret_val;
}

// Print usage.
void print_usage(const char *program_name, FILE *out) 
{
  fprintf(out, 
	  "Usage: \n\n"
	  "  %s --export-all CPE_LANG_XML ENCODING CPE_EXPORT_XML ENCODING\n"
	  "  %s --get-all CPE_LANG_XML ENCODING\n"
	  "  %s --get-key CPE_LANG_XML ENCODING KEY\n"
	  "  %s --set-all CPE_LANG_XML ENCODING NS_PREFIX NS_HREF (PLATFORM_ID)*\n"
	  "  %s --set-key CPE_LANG_XML ENCODING KEY ID (TITLE)*\n"
	  "  %s --set-new CPE_LANG_XML ENCODING NS_PREFIX NS_HREF (PLATFORM_ID)*\n"
	  "  %s --match-cpe CPE_LANG_XML ENCODING CPE_NAME1 CPE_NAME2\n"
	  "  %s --smoke-test\n",
	  program_name, program_name, program_name, program_name, program_name,
	  program_name, program_name, program_name);
}

// Print expression in prefix form.
int print_expr_prefix_form(const struct cpe_testexpr *expr) 
{

  //const struct cpe_testexpr *sub;

  putchar('(');
  
  if (cpe_testexpr_get_oper(expr) & CPE_LANG_OPER_NOT)
    printf("!");

  switch (cpe_testexpr_get_oper(expr) & CPE_LANG_OPER_MASK) {
  case CPE_LANG_OPER_AND:
  case CPE_LANG_OPER_OR:
    printf("%s", CPE_OPER_STRS[cpe_testexpr_get_oper(expr) & CPE_LANG_OPER_MASK]);
    //for (sub = cpe_testexpr_get_meta_expr(expr); cpe_testexpr_get_oper(sub); sub=cpe_testexpr_get_next(sub))
	OSCAP_FOREACH(cpe_testexpr, sub, cpe_testexpr_get_meta_expr(expr), print_expr_prefix_form(sub););
    break;
  case CPE_LANG_OPER_MATCH:
    printf("%s", cpe_name_get_uri(cpe_testexpr_get_meta_cpe(expr)));
    break;
  default:  
    return 1;
  }

  putchar(')');

  return 0;
}

// Print platform.
void print_platform(struct cpe_platform *platform) 
{
  struct oscap_title *title = NULL;
  struct oscap_title_iterator *title_it = NULL;
  const char *remark, *id, *content, *language;

  id = cpe_platform_get_id(platform);
  printf("%s:", id == NULL ? "" : id);
  
  remark = cpe_platform_get_remark(platform);
  printf("%s:", remark == NULL ? "" : remark);
  
  title_it = cpe_platform_get_titles(platform);
  while (oscap_title_iterator_has_more(title_it)) {
    title = oscap_title_iterator_next(title_it);

    content = oscap_title_get_content(title);
    printf("%s.", content == NULL ? "" : content);
    
    language = oscap_title_get_language(title);
    printf("%s,", language == NULL ? "" : language);
  }
  putchar(':');
  oscap_title_iterator_free(title_it);
  print_expr_prefix_form(cpe_platform_get_expr(platform));
  putchar('\n');
} 
