/*      -*- OpenSAF  -*-
 *
 * (C) Copyright 2008 The OpenSAF Foundation
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. This file and program are licensed
 * under the GNU Lesser General Public License Version 2.1, February 1999.
 * The complete license can be accessed from the following location:
 * http://opensource.org/licenses/lgpl-license.php
 * See the Copying file included with the OpenSAF distribution for full
 * licensing terms.
 *
 * Author(s): Emerson Network Power
 *
 */

/*****************************************************************************

  DESCRIPTION: This file contains proxy internode & external proxy component
               handling functions.

  FUNCTIONS INCLUDED in this module:
  
****************************************************************************/
#include "avnd.h"
static uns32 avnd_int_ext_comp_val(AVND_CB *, SaNameT *, AVND_COMP **, SaAisErrorT *);
/******************************************************************************
  Name          : avnd_evt_mds_avnd_up
 
  Description   : This routine handles MDS UP event of AvNDs.
 
  Arguments     : cb  - ptr to the AvND control block.
                  evt - ptr to the AvND event.
 
  Return Values : NCSCC_RC_SUCCESS/NCSCC_RC_FAILURE
 
  Notes         : None
******************************************************************************/
uns32 avnd_evt_mds_avnd_up(AVND_CB *cb, AVND_EVT *evt)
{
	uns32 res = 0;

	if (evt->type != AVND_EVT_MDS_AVND_UP) {
		return NCSCC_RC_FAILURE;
	}

	/* Add node id to mds dest mapping in the data base. */
	res = avnd_nodeid_mdsdest_rec_add(cb, evt->info.mds.mds_dest);

	return res;

}

/******************************************************************************
  Name          : avnd_evt_mds_avnd_dn
 
  Description   : This routine handles MDS DOWN event of AvNDs.
 
  Arguments     : cb  - ptr to the AvND control block.
                  evt - ptr to the AvND event.
 
  Return Values : NCSCC_RC_SUCCESS/NCSCC_RC_FAILURE
 
  Notes         : None
******************************************************************************/
uns32 avnd_evt_mds_avnd_dn(AVND_CB *cb, AVND_EVT *evt)
{
	uns32 res = 0;

	if (evt->type != AVND_EVT_MDS_AVND_DN) {
		return NCSCC_RC_FAILURE;
	}

	/* Delete node id to mds dest mapping in the data base. */
	res = avnd_nodeid_mdsdest_rec_del(cb, evt->info.mds.mds_dest);

	return res;

}

/******************************************************************************
  Name          : avnd_evt_ava_comp_val_req
 
  Description   : This routine creates a validation req msg and sends to AvD.
 
  Arguments     : cb  - ptr to the AvND control block.
                  evt - ptr to the AvND event.
 
  Return Values : NCSCC_RC_SUCCESS/NCSCC_RC_FAILURE
 
  Notes         : None
******************************************************************************/
uns32 avnd_evt_ava_comp_val_req(AVND_CB *cb, AVND_EVT *evt)
{
	uns32 rc = NCSCC_RC_SUCCESS;
	AVND_DND_MSG_LIST *rec = 0;
	AVND_MSG msg;
	AVSV_AMF_API_INFO *api_info = &evt->info.ava.msg->info.api_info;
	AVSV_AMF_COMP_REG_PARAM *reg = &api_info->param.reg;

	m_AVND_AVND_ENTRY_LOG("avnd_evt_ava_comp_val_req():Comp,Type and Hdl are",
			      &reg->comp_name, api_info->type, reg->hdl, 0, 0);

	memset(&msg, 0, sizeof(AVND_MSG));

	/* populate the msg */
	if (0 != (msg.info.avd = calloc(1, sizeof(AVSV_DND_MSG)))) {
		msg.type = AVND_MSG_AVD;
		msg.info.avd->msg_type = AVSV_N2D_COMP_VALIDATION_MSG;
		msg.info.avd->msg_info.n2d_comp_valid_info.msg_id = ++(cb->snd_msg_id);
		msg.info.avd->msg_info.n2d_comp_valid_info.node_id = cb->clmdb.node_info.nodeId;
		msg.info.avd->msg_info.n2d_comp_valid_info.comp_name =
		    evt->info.ava.msg->info.api_info.param.reg.comp_name;

		/* add the record to the AvD msg list */
		if ((0 != (rec = avnd_diq_rec_add(cb, &msg)))) {
			/* These parameters would not be encoded or decoded so, wouldn't be sent to AvD. */
			rec->msg.info.avd->msg_info.n2d_comp_valid_info.hdl = reg->hdl;
			rec->msg.info.avd->msg_info.n2d_comp_valid_info.proxy_comp_name = reg->proxy_comp_name;
			rec->msg.info.avd->msg_info.n2d_comp_valid_info.mds_dest = api_info->dest;
			rec->msg.info.avd->msg_info.n2d_comp_valid_info.mds_ctxt = evt->mds_ctxt;
			/* send the message */
			rc = avnd_diq_rec_send(cb, rec);

			if ((NCSCC_RC_SUCCESS != rc) && rec) {
				m_AVND_AVND_ERR_LOG("avnd_diq_rec_send:failed:Comp,Type and Hdl are",
						    &reg->comp_name, api_info->type, reg->hdl, 0, 0);
				/* pop & delete */
				m_AVND_DIQ_REC_FIND_POP(cb, rec);
				avnd_diq_rec_del(cb, rec);
			}
		} else {
			rc = NCSCC_RC_FAILURE;
			m_AVND_AVND_ERR_LOG("avnd_diq_rec_add failed:Comp,Type and Hdl are",
					    &reg->comp_name, api_info->type, reg->hdl, 0, 0);
		}
	} else
		rc = NCSCC_RC_FAILURE;

	if (NCSCC_RC_FAILURE == rc) {
		m_AVND_AVND_ERR_LOG("avnd_evt_ava_comp_val_req :failure:Comp,Type and Hdl are",
				    &reg->comp_name, api_info->type, reg->hdl, 0, 0);
	}
	/* free the contents of avnd message */
	avnd_msg_content_free(cb, &msg);

	return rc;

}

/******************************************************************************
  Name          : avnd_evt_avd_comp_validation_resp_msg
 
  Description   : This routine creates a registration message and sends it to 
                  corresponding AvND, if the AvND is UP. If AvND is Down then
                  it sends resp to AvA as TRY_AGAIN.
 
  Arguments     : cb  - ptr to the AvND control block.
                  evt - ptr to the AvND event.
 
  Return Values : NCSCC_RC_SUCCESS/NCSCC_RC_FAILURE
 
  Notes         : None
******************************************************************************/
uns32 avnd_evt_avd_comp_validation_resp_msg(AVND_CB *cb, AVND_EVT *evt)
{
	uns32 rc = NCSCC_RC_SUCCESS;
	AVND_DND_MSG_LIST *rec = 0;
	AVSV_D2N_COMP_VALIDATION_RESP_INFO *info = NULL;
	SaAisErrorT amf_rc = SA_AIS_OK;
	AVND_COMP *comp = NULL, *pxy_comp = NULL;
	AVSV_N2D_COMP_VALIDATION_INFO comp_valid_info;
	AVSV_AMF_API_INFO api_info;

	/* dont process unless AvD is up */
	if (!m_AVND_CB_IS_AVD_UP(cb))
		goto done;

	info = &evt->info.avd->msg_info.d2n_comp_valid_resp_info;

	m_AVND_AVND_ENTRY_LOG("avnd_evt_avd_comp_validation_resp_msg():Comp,MsgId,NodeId and result are",
			      &info->comp_name, info->msg_id, info->node_id, info->result, 0);

	m_AVND_DIQ_REC_FIND(cb, info->msg_id, rec);

	if ((NULL == rec) || (memcmp(&info->comp_name,
				     &rec->msg.info.avd->msg_info.n2d_comp_valid_info.comp_name,
				     sizeof(SaNameT)) != 0)) {
		/* Seems the rec was deleted, some problem. */
		m_AVND_AVND_ERR_LOG("Valid Rep:Rec is NULL or Name Mismatch:Comp,MsgId,NodeId and result are",
				    &info->comp_name, info->msg_id, info->node_id, info->result, 0);
		rc = NCSCC_RC_FAILURE;
		goto done;
	}

	comp_valid_info = rec->msg.info.avd->msg_info.n2d_comp_valid_info;

	if (AVSV_VALID_FAILURE == info->result) {
		/* Component is not configured. Send registration failure to the proxy. */
		amf_rc = SA_AIS_ERR_INVALID_PARAM;
		goto send_resp;

	} else if (AVSV_VALID_SUCC_COMP_NODE_UP == info->result) {
		/* So, let us add this component in the data base. And send reg req
		   to the AvND, where proxied comp is running. */
		comp = avnd_internode_comp_add(&(cb->internode_avail_comp_db),
					       &(info->comp_name), info->node_id, &rc, FALSE, FALSE);
		if ((comp) && (SA_AIS_OK == rc)) {
			/* Fill other informations here */
			comp->reg_hdl = comp_valid_info.hdl;
			comp->reg_dest = comp_valid_info.mds_dest;
			comp->mds_ctxt = rec->msg.info.avd->msg_info.n2d_comp_valid_info.mds_ctxt;
			/* We need to update node id attribute in rec as it was node id of proxy
			   actually it should be node id of the proxied component.  */
			comp_valid_info.node_id = info->node_id;

			/* Create proxy-proxied support here */
			if (0 == (pxy_comp = m_AVND_COMPDB_REC_GET(cb->compdb, comp_valid_info.proxy_comp_name))) {
				avnd_internode_comp_del(cb, &(cb->internode_avail_comp_db), &(info->comp_name));
				rc = NCSCC_RC_FAILURE;
				amf_rc = SA_AIS_ERR_INVALID_PARAM;
				goto send_resp;
			}

			if ((NULL != pxy_comp) && (!m_AVND_COMP_IS_REG(pxy_comp))) {
				avnd_internode_comp_del(cb, &(cb->internode_avail_comp_db), &(info->comp_name));
				rc = NCSCC_RC_FAILURE;
				amf_rc = SA_AIS_ERR_NOT_EXIST;
				goto send_resp;
			}

			/* When REG RESP will come from other AvND, then we will need
			   proxy of this component, so maintain a link here,
			   though the same link will be generated after REG RESP
			   comes back in avnd_comp_proxied_add (called from 
			   avnd_comp_reg_prc function) */

			comp->pxy_comp = pxy_comp;

			/* Send a registration message to the corresponding AvND */
			memset(&api_info, 0, sizeof(AVSV_AMF_API_INFO));
			m_AVND_COMP_REG_MSG_FILL(api_info, comp->reg_dest,
						 comp->reg_hdl, &comp->name, &comp->pxy_comp->name);
			rc = avnd_avnd_msg_send(cb, (uns8 *)&(api_info), AVSV_AMF_COMP_REG,
						&comp->mds_ctxt, comp->node_id);

			if (rc != NCSCC_RC_SUCCESS) {
				m_AVND_AVND_ERR_LOG("avnd_avnd_msg_send failed:Comp,MsgId,NodeId and result are",
						    &info->comp_name, info->msg_id, info->node_id, rc, 0);

				amf_rc = SA_AIS_ERR_TRY_AGAIN;
				avnd_internode_comp_del(cb, &(cb->internode_avail_comp_db), &(info->comp_name));
				goto send_resp;
			}
			comp->reg_resp_pending = TRUE;
			m_AVND_SEND_CKPT_UPDT_ASYNC_ADD(cb, comp, AVND_CKPT_COMP_CONFIG);
			goto done;
		} else {
			amf_rc = rc;
			rc = NCSCC_RC_FAILURE;
			goto send_resp;
		}

	} else if (AVSV_VALID_SUCC_COMP_NODE_DOWN == info->result) {
		/* Component node is down. Send TRY AGAIN to AvA. 
		   We cann't send reg succ to AvA as we cann't validate
		   the registration message, it is validated by the proxied
		   component AvND and then only we send Reg Succ to AvA. */
		amf_rc = SA_AIS_ERR_TRY_AGAIN;
		goto send_resp;

	} /* else if(AVSV_VALID_SUCC_COMP_NODE_DOWN == info->result) */
	else {
		/* Wrong result */
		amf_rc = SA_AIS_ERR_TRY_AGAIN;
		rc = NCSCC_RC_FAILURE;
		goto send_resp;
	}

 send_resp:

	/* send the response back to AvA */
	rc = avnd_amf_resp_send(cb, AVSV_AMF_COMP_REG, amf_rc, 0,
				&comp_valid_info.mds_dest, &comp_valid_info.mds_ctxt, NULL, FALSE);

 done:
	if (rec) {
		m_AVND_DIQ_REC_FIND_POP(cb, rec);
		avnd_diq_rec_del(cb, rec);
	}

	if (NCSCC_RC_SUCCESS != rc) {
		m_AVND_AVND_ERR_LOG("avnd_evt_avd_comp_validation_resp_msg failed:Comp,MsgId,NodeId and result are",
				    &info->comp_name, info->msg_id, info->node_id, info->result, 0);
	}
	return rc;
}

/******************************************************************************
  Name          : avnd_avnd_msg_send
 
  Description   : This routine sends messages of type AVSV_AMF_API_TYPE to the 
                  corresponding AvND in ASYNC.
 
  Arguments     : cb  - ptr to the AvND control block.
                  info - ptr to the msg information to be sent.
                  type - Type of the AvA message.
                  ctxt - Ptr to the MDS Context.
                  node_id - Node Id of the AvND to be sent.
 
  Return Values : NCSCC_RC_SUCCESS/NCSCC_RC_FAILURE
 
  Notes         : None
******************************************************************************/
uns32 avnd_avnd_msg_send(AVND_CB *cb, uns8 *msg_info, AVSV_AMF_API_TYPE type, MDS_SYNC_SND_CTXT *ctxt, NODE_ID node_id)
{
	uns32 rc = NCSCC_RC_SUCCESS;
	AVND_MSG msg;
	AVSV_ND2ND_AVA_MSG *nd_nd_ava_msg = NULL;
	MDS_DEST i_to_dest = 0;
	AVSV_AMF_API_INFO *info = (AVSV_AMF_API_INFO *)msg_info;

	m_AVND_AVND_ENTRY_LOG("avnd_avnd_msg_send():Type and NodeID are", NULL, type, node_id, 0, 0);

	/* Create a Registration message and send to AvND */
	memset(&msg, 0, sizeof(AVND_MSG));

	/* populate the msg */
	if (0 != (msg.info.avnd = calloc(1, sizeof(AVSV_ND2ND_AVND_MSG)))) {
		msg.type = AVND_MSG_AVND;

		if (0 == (nd_nd_ava_msg = calloc(1, sizeof(AVSV_NDA_AVA_MSG)))) {
			rc = NCSCC_RC_FAILURE;
			free(msg.info.avnd);
			goto done;
		}

		msg.info.avnd->type = AVND_AVND_AVA_MSG;
		msg.info.avnd->info.msg = nd_nd_ava_msg;
		memcpy(&msg.info.avnd->mds_ctxt, ctxt, sizeof(MDS_SYNC_SND_CTXT));
	} else {
		rc = NCSCC_RC_FAILURE;
		goto done;
	}

	switch (type) {
	case AVSV_AMF_COMP_REG:
	case AVSV_AMF_FINALIZE:
	case AVSV_AMF_COMP_UNREG:
	case AVSV_AMF_PM_START:
	case AVSV_AMF_PM_STOP:
	case AVSV_AMF_HC_START:
	case AVSV_AMF_HC_STOP:
	case AVSV_AMF_HC_CONFIRM:
	case AVSV_AMF_CSI_QUIESCING_COMPLETE:
	case AVSV_AMF_HA_STATE_GET:
	case AVSV_AMF_PG_START:
	case AVSV_AMF_PG_STOP:
	case AVSV_AMF_ERR_REP:
	case AVSV_AMF_ERR_CLEAR:
	case AVSV_AMF_RESP:
		{
			nd_nd_ava_msg->type = AVSV_AVA_API_MSG;
			memcpy(&nd_nd_ava_msg->info.api_info, info, sizeof(AVSV_AMF_API_INFO));
			break;
		}		/* case AVSV_AMF_HC_START */

	default:
		goto done;
		break;

	}			/* switch(type) */

	/* Check node id value. If it is zero then it is an external component.
	   So, we need to use Vdest of Controller AvND. */
	if (node_id != 0)
		i_to_dest = avnd_get_mds_dest_from_nodeid(cb, node_id);
	else {
		i_to_dest = cb->cntlr_avnd_vdest;
	}

	if (0 != i_to_dest)
		rc = avnd_avnd_mds_send(cb, i_to_dest, &msg);
	else
		rc = NCSCC_RC_FAILURE;

 done:
	if (NCSCC_RC_SUCCESS != rc) {
		m_AVND_AVND_ERR_LOG("AvND Send Failure:Type,NodeID, Mds and rc are",
				    NULL, type, node_id, i_to_dest, rc);
	}

	/* free the contents of the message */
	avnd_msg_content_free(cb, &msg);

	return rc;
}

/******************************************************************************
  Name          : avnd_int_ext_comp_hdlr

  Description   : This routine checks for int/ext comp and forwards the req 
                  to AvND.

  Arguments     : cb  - ptr to the AvND control block.
                  api_info - ptr to the api info structure.
                  ctxt - ptr to mds context information.
                  o_amf_rc - ptr to the amf-rc (o/p).

  Return Values : NCSCC_RC_SUCCESS/NCSCC_RC_FAILURE

  Notes         : None
******************************************************************************/
uns32 avnd_int_ext_comp_hdlr(AVND_CB *cb, AVSV_AMF_API_INFO *api_info,
			     MDS_SYNC_SND_CTXT *ctxt, SaAisErrorT *o_amf_rc, NCS_BOOL *int_ext_comp)
{
	uns32 rc = NCSCC_RC_SUCCESS;
	AVND_COMP *o_comp = NULL;
	SaNameT comp_name;
	NCS_BOOL send_resp = TRUE;
	AVND_COMP_CBK *cbk_rec = 0;

	*o_amf_rc = SA_AIS_OK;
	*int_ext_comp = FALSE;

	switch (api_info->type) {
	case AVSV_AMF_COMP_UNREG:
		{
			comp_name = api_info->param.unreg.comp_name;
			break;
		}

	case AVSV_AMF_HC_START:
		{
			comp_name = api_info->param.hc_start.comp_name;
			break;
		}

	case AVSV_AMF_HC_STOP:
		{
			comp_name = api_info->param.hc_stop.comp_name;
			break;
		}

	case AVSV_AMF_HC_CONFIRM:
		{
			comp_name = api_info->param.hc_confirm.comp_name;
			break;
		}

	case AVSV_AMF_PM_START:
		{
			comp_name = api_info->param.pm_start.comp_name;
			break;
		}

	case AVSV_AMF_PM_STOP:
		{
			comp_name = api_info->param.pm_stop.comp_name;
			break;
		}

	case AVSV_AMF_CSI_QUIESCING_COMPLETE:
		{
			comp_name = api_info->param.csiq_compl.comp_name;
			send_resp = FALSE;
			break;
		}

	case AVSV_AMF_HA_STATE_GET:
		{
			comp_name = api_info->param.ha_get.comp_name;
			break;
		}

	case AVSV_AMF_ERR_REP:
		{
			comp_name = api_info->param.err_rep.err_comp;
			break;
		}

	case AVSV_AMF_ERR_CLEAR:
		{
			comp_name = api_info->param.err_clear.comp_name;
			break;
		}

	case AVSV_AMF_RESP:
		{
			comp_name = api_info->param.resp.comp_name;
			send_resp = FALSE;
			break;
		}

	default:
		rc = NCSCC_RC_FAILURE;
		*o_amf_rc = SA_AIS_ERR_INVALID_PARAM;
		m_AVND_AVND_ERR_LOG("avnd_int_ext_comp_hdlr:Wrong Type: Type and Mds Dest are",
				    NULL, api_info->type, api_info->dest, 0, 0);
		goto done;
		break;
	}

	m_AVND_AVND_ENTRY_LOG("avnd_int_ext_comp_hdlr():Comp and Type are", &comp_name, api_info->type, 0, 0, 0);

	rc = avnd_int_ext_comp_val(cb, &comp_name, &o_comp, o_amf_rc);
	if ((NCSCC_RC_SUCCESS == rc) && (SA_AIS_OK == *o_amf_rc)) {
		*int_ext_comp = TRUE;
/*****************************  Section 1 Starts  **********************/
		if ((AVSV_AMF_RESP == api_info->type) || (AVSV_AMF_CSI_QUIESCING_COMPLETE == api_info->type)) {
			AVSV_AMF_RESP_PARAM *resp = &api_info->param.resp;
/*******************************************************************************
We need to consider AVSV_AMF_RESP/AVSV_AMF_CSI_QUIESCING_COMPLETE, here. 
Since this may a response for an internode/ext component's for callbacks, 
so we might have stored the callback in the component cbk_list. 
We need to find them and remove from the cbk_list list and the forward the 
resp to originator AvND.
*******************************************************************************/
			/* get the matching entry from the cbk list. Note that if the resp
			   has come from AvA and the component name has been found in 
			   internode_avail_comp_db, then definetely the comp is a proxied 
			   component and not a proxy component, so need not to search in the
			   comp->pxied_list list for resp->inv. */
			m_AVND_COMP_CBQ_INV_GET(o_comp, resp->inv, cbk_rec);

			if (!cbk_rec) {
				m_AVND_AVND_ERR_LOG
				    ("avnd_int_ext_comp_hdlr:Couldn't get cbk_rec: Comp, Type and Mds are", &comp_name,
				     api_info->type, api_info->dest, 0, 0);
				rc = NCSCC_RC_FAILURE;
				goto done;
			}
			/* We got the callback record, so before deleting it, replace the 
			   invocation handle in the response with the original one. Check
			   function avnd_evt_avnd_avnd_cbk_msg_hdl()'s comments */
			resp->inv = cbk_rec->orig_opq_hdl;
			avnd_comp_cbq_rec_pop_and_del(cb, o_comp, cbk_rec, FALSE);

		}

		/* if(AVSV_AMF_RESP == api_info->type)  */
 /************************* Section 1 Ends *********************************/
 /************************* Section 2 Starts  ******************************/
		/* If the registration request has been sent to proxied component AvND,
		   then we should obviate the other operations on the proxied component
		   till we get the SUCC response and we finally consider this component as 
		   a valid registered component.
		 */
		if (TRUE == o_comp->reg_resp_pending) {
			/* Let at least this operation complete. */
			*o_amf_rc = SA_AIS_ERR_TRY_AGAIN;
			goto resp_send;
		}

/************************* Section 2 Ends *********************************/

		/* We need to forward this req to other AvND */
		rc = avnd_avnd_msg_send(cb, (uns8 *)api_info, api_info->type, ctxt, o_comp->node_id);
		if (NCSCC_RC_SUCCESS != rc) {
			/* We couldn't send this to other AvND, tell user to try again.  */
			*o_amf_rc = SA_AIS_ERR_TRY_AGAIN;
			m_AVND_AVND_ERR_LOG("avnd_int_ext_comp_hdlr:Msg Send Failed: Comp, Type and Mds are",
					    &comp_name, api_info->type, api_info->dest, 0, 0);
			goto resp_send;
		} else {
			/* Send SUCCESSFULLY. Return. */
			return rc;
		}
	} else {
		/* This is not an internode/ext component, so return SUCCESS. */
		*int_ext_comp = FALSE;
		return NCSCC_RC_SUCCESS;
	}

 resp_send:

	if (TRUE == send_resp) {
		rc = avnd_amf_resp_send(cb, api_info->type, *o_amf_rc, 0, &api_info->dest, ctxt, NULL, FALSE);
	}

 done:
	if (NCSCC_RC_SUCCESS != rc) {
		m_AVND_AVND_ERR_LOG("avnd_int_ext_comp_hdlr():Failure:Comp,Type and Mds Dest are",
				    &comp_name, api_info->type, api_info->dest, 0, 0);
	}
	return rc;
}

/******************************************************************************
  Name          : avnd_int_ext_comp_val

  Description   : This routine checks for int/ext comp.

  Arguments     : cb  - ptr to the AvND control block.
                  api_info - ptr to the api info structure.
                  ctxt - ptr to mds context information.
                  o_amf_rc - ptr to the amf-rc (o/p).

  Return Values : NCSCC_RC_SUCCESS/NCSCC_RC_FAILURE

  Notes         : None
******************************************************************************/
uns32 avnd_int_ext_comp_val(AVND_CB *cb, SaNameT *comp_name, AVND_COMP **o_comp, SaAisErrorT *o_amf_rc)
{
	uns32 res = NCSCC_RC_SUCCESS;
	*o_amf_rc = SA_AIS_OK;

	m_AVND_AVND_ENTRY_LOG("avnd_int_ext_comp_val(): Comp Name is", comp_name, 0, 0, 0, 0);
	if (0 == (*o_comp = m_AVND_INT_EXT_COMPDB_REC_GET(cb->internode_avail_comp_db, *comp_name))) {
		return NCSCC_RC_FAILURE;
	} else {
		/* This means that this is an internode component. But need to check wether
		   it is a proxy for external component. If it is, then we shouldn't treat
		   it as an external component though it is in internode DB. This is bz of
		   a proxy on Ctrl is proxying external component. */
		if (m_AVND_PROXY_IS_FOR_EXT_COMP(*o_comp))
			return NCSCC_RC_FAILURE;
	}
	return res;
}

/******************************************************************************
  Name          : avnd_avnd_cbk_del_send

  Description   : This routine sends Delete Callback to a particular AvND.

  Arguments     : cb  - ptr to the AvND control block.
                  comp_name - ptr to the comp name.
                  opq_hdl - ptr to handle.
                  node_id - ptr to the node_id.

  Return Values : NCSCC_RC_SUCCESS/NCSCC_RC_FAILURE

  Notes         : None
******************************************************************************/
uns32 avnd_avnd_cbk_del_send(AVND_CB *cb, SaNameT *comp_name, uns32 *opq_hdl, NODE_ID *node_id)
{
	uns32 rc = NCSCC_RC_SUCCESS;
	MDS_DEST i_to_dest = 0;
	AVND_MSG msg;

	m_AVND_AVND_ENTRY_LOG("avnd_avnd_cbk_del_send():Comp,NodeID and opq_hdl are",
			      comp_name, *node_id, *opq_hdl, 0, 0);

	/* Create a Registration message and send to AvND */
	memset(&msg, 0, sizeof(AVND_MSG));

	/* populate the msg */
	if (0 != (msg.info.avnd = calloc(1, sizeof(AVSV_ND2ND_AVND_MSG)))) {
		msg.type = AVND_MSG_AVND;
		msg.info.avnd->type = AVND_AVND_CBK_DEL;
		msg.info.avnd->info.cbk_del.comp_name = *comp_name;
		msg.info.avnd->info.cbk_del.opq_hdl = *opq_hdl;
	} else {
		rc = NCSCC_RC_FAILURE;
		goto done;
	}

	i_to_dest = avnd_get_mds_dest_from_nodeid(cb, *node_id);

	rc = avnd_avnd_mds_send(cb, i_to_dest, &msg);

 done:
	if (NCSCC_RC_SUCCESS != rc) {
		m_AVND_AVND_ERR_LOG("AvND Send Failure:Comp,NodeID,opq_hdl and MdsDest are",
				    comp_name, *node_id, *opq_hdl, i_to_dest, 0);
	}

	/* free the contents of the message */
	avnd_msg_content_free(cb, &msg);

	return rc;
}
