// -*- mode: C++; tab-width: 2; -*-
// vi: set ts=2:
//
// --------------------------------------------------------------------------
//                   OpenMS Mass Spectrometry Framework
// --------------------------------------------------------------------------
//  Copyright (C) 2003-2011 -- Oliver Kohlbacher, Knut Reinert
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// --------------------------------------------------------------------------
// $Maintainer: Andreas Bertsch $
// $Authors: Andreas Bertsch, Daniel Jameson, Chris Bielow $
// --------------------------------------------------------------------------

#include <OpenMS/FORMAT/IdXMLFile.h>
#include <OpenMS/FORMAT/MascotXMLFile.h>
#include <OpenMS/FORMAT/MascotRemoteQuery.h>
#include <OpenMS/FORMAT/MascotGenericFile.h>
#include <OpenMS/KERNEL/StandardTypes.h>
#include <OpenMS/FORMAT/FileHandler.h>
#include <OpenMS/FORMAT/FileTypes.h>
#include <OpenMS/SYSTEM/File.h>

#include <OpenMS/APPLICATIONS/TOPPBase.h>

#include <sstream>

#include <QtCore/QDir>
#include <QtCore/QFile>
#include <QtCore/QCoreApplication>
#include <QtCore/QTimer>

using namespace OpenMS;
using namespace std;

//-------------------------------------------------------------
//Doxygen docu
//-------------------------------------------------------------

/**
	@page TOPP_MascotAdapterOnline MascotAdapterOnline

	@brief Identifies peptides in MS/MS spectra via Mascot.

<CENTER>
	<table>
		<tr>
			<td ALIGN = "center" BGCOLOR="#EBEBEB"> pot. predecessor tools </td>
			<td VALIGN="middle" ROWSPAN=2> \f$ \longrightarrow \f$ MascotAdapterOnline \f$ \longrightarrow \f$</td>
			<td ALIGN = "center" BGCOLOR="#EBEBEB"> pot. successor tools </td>
		</tr>
		<tr>
			<td VALIGN="middle" ALIGN = "center" ROWSPAN=1> any signal-/preprocessing tool @n (in mzML format)</td>
			<td VALIGN="middle" ALIGN = "center" ROWSPAN=1> @ref TOPP_IDFilter or @n any protein/peptide processing tool</td>
		</tr>
	</table>
</CENTER>

	This wrapper application serves for getting peptide identifications
	for MS/MS spectra. It communicates with the Mascot server
	over the network and is not needed to be called from same machine.

	It support Mascot security features and has also proxy server
	support. This minimal version of Mascot support by this wrapper
	is version 2.2.x. Mascot 2.3 works as well, but has not been tested extensively.
  @note Some Mascot server instances seem to fail without reporting back an error message.
        Thus, try to run the search on another Mascot server or change/validate search parameters
        (e.g., using modifications in the INI file, which are unknown to Mascot, but known to OpenMS might be a problem).


  @note Be aware that Mascot returns incomplete/incorrect protein assignments for most identified peptides (why ever that is). 
        Thus we do not forward any protein assignments, only peptide sequences. You should run PeptideIndexer after this tool to get correct assignments.
        You can use the flag 'keep_protein_links' to override this behavior.

	<B>The command line parameters of this tool are:</B>
	@verbinclude TOPP_MascotAdapterOnline.cli

	For the parameters of the algorithm section see the algorithms documentation: @n
	@ref OpenMS::MascotRemoteQuery "Mascot_server" @n
	@ref OpenMS::MascotGenericFile "Mascot_parameters" @n

*/

// We do not want this class to show up in the doc:
/// @cond TOPPCLASSES


class TOPPMascotAdapterOnline
	: public TOPPBase
{
	public:
		TOPPMascotAdapterOnline()
			: TOPPBase("MascotAdapterOnline","Annotates MS/MS spectra using Mascot.")
		{
		}

	protected:

		void registerOptionsAndFlags_()
		{
			registerInputFile_("in", "<file>", "", "input file in mzML format.\n");
			setValidFormats_("in", StringList::create("mzML"));
			registerOutputFile_("out", "<file>", "", "output file in IdXML format.\n");
			setValidFormats_("out", StringList::create("idXML"));

			registerSubsection_("Mascot_server", "Mascot server details");
			registerSubsection_("Mascot_parameters", "Mascot parameters used for searching");
      registerFlag_("keep_protein_links", "The Mascot response file usually returns incomplete/wrong protein hits, so re-indexing the peptide hits is required. To avoid confusion why there"
                                          " are so few protein hits and force re-indexing, no proteins should be reported. To see the original (wrong) list, enable this flag.", true);
		}

    Param getSubsectionDefaults_(const String& section) const
    {
			if (section == "Mascot_server")
			{
				MascotRemoteQuery mascot_query;
				return mascot_query.getParameters();
			}

			if (section == "Mascot_parameters")
			{
				MascotGenericFile mascot_infile;
				return mascot_infile.getParameters();
			}

      return Param();
    }


		ExitCodes main_(int argc, const char** argv)
		{
			//-------------------------------------------------------------
      // parameter handling
      //-------------------------------------------------------------

      //input/output files
			String in(getStringOption_("in")), out(getStringOption_("out"));
			FileHandler fh;
			FileTypes::Type in_type = fh.getType(in);

      //-------------------------------------------------------------
      // loading input
      //-------------------------------------------------------------

			PeakMap exp;
			fh.loadExperiment(in, exp, in_type, log_type_);

      //-------------------------------------------------------------
      // calculations
      //-------------------------------------------------------------

			Param mascot_param = getParam_().copy("Mascot_parameters:", true);
      MascotGenericFile mascot_infile;
			mascot_infile.setParameters(mascot_param);

			// get the spectra into string stream
			writeDebug_("Writing Mascot mgf file to stringstream", 1);
			stringstream ss;
			mascot_infile.store(ss, in, exp);

			// Usage of a QCoreApplication is overkill here (and ugly too), but we just use the
			// QEventLoop to process the signals and slots and grab the results afterwards from
			// the MascotRemotQuery instance
			char** argv2 = const_cast<char**>(argv);
			QCoreApplication event_loop(argc, argv2);
			MascotRemoteQuery* mascot_query = new MascotRemoteQuery(&event_loop);
			Param mascot_query_param = getParam_().copy("Mascot_server:", true);
			writeDebug_("Setting parameters for Mascot query", 1);
			mascot_query->setParameters(mascot_query_param);
			writeDebug_("Setting spectra for Mascot query", 1);
			mascot_query->setQuerySpectra(ss.str());

			// remove unnecessary spectra
			ss.clear();

		  QObject::connect(mascot_query, SIGNAL(done()), &event_loop, SLOT(quit()));
			QTimer::singleShot(1000, mascot_query, SLOT(run()));
			writeDebug_("Fire off Mascot query", 1);
			event_loop.exec();
			writeDebug_("Mascot query finished", 1);

			if (mascot_query->hasError())
			{
				writeLog_("An error occurred during the query: " + mascot_query->getErrorMessage());
				delete mascot_query;
				return EXTERNAL_PROGRAM_ERROR;
			}

			// write Mascot response to file
			String mascot_tmp_file_name(File::getTempDirectory() + "/" + File::getUniqueName() + "_Mascot_response");
			QFile mascot_tmp_file(mascot_tmp_file_name.c_str());
			mascot_tmp_file.open(QIODevice::WriteOnly);
			mascot_tmp_file.write(mascot_query->getMascotXMLResponse());
			mascot_tmp_file.close();

			// clean up
			delete mascot_query;

			vector<PeptideIdentification> pep_ids;
			ProteinIdentification prot_id;

			// read the response
			MascotXMLFile().load(mascot_tmp_file_name, prot_id, pep_ids);
      writeDebug_("Read " + String(pep_ids.size()) + " peptide ids and " + String(prot_id.getHits().size()) + " protein identifications from Mascot", 5);

			// for debugging errors relating to unexpected response files
			if (this->debug_level_ >= 100)
			{
				writeDebug_(String("\nMascot Server Response file saved to: '") + mascot_tmp_file_name + "'. If an error occurs, send this file to the OpenMS team.\n", 100);
			}
			else
			{
				// delete file
				mascot_tmp_file.remove();
			}

      // keep or delete protein identifications?!
			vector<ProteinIdentification> prot_ids;
      if (!getFlag_("keep_protein_links"))
      {
        // remove protein links from peptides
        std::vector<String> empty;
        for (Size i=0; i<pep_ids.size(); ++i)
        {
          std::vector< PeptideHit > hits = pep_ids[i].getHits();
          for (Size h=0; h<hits.size(); ++h)
          {
            hits[h].setProteinAccessions(empty);
          }
          pep_ids[i].setHits(hits);
        }
        // remove proteins
        std::vector< ProteinHit > p_hit;
        prot_id.setHits(p_hit);
      }
      prot_ids.push_back(prot_id);

      //-------------------------------------------------------------
      // writing output
      //-------------------------------------------------------------

      IdXMLFile().store(out, prot_ids, pep_ids);

			return EXECUTION_OK;
		}

};


int main( int argc, const char** argv )
{
	TOPPMascotAdapterOnline tool;

	return tool.main(argc,argv);
}

/// @endcond
