/*
   Copyright (C) 1994-2001 Digitool, Inc
   This file is part of OpenMCL.  

   OpenMCL is licensed under the terms of the Lisp Lesser GNU Public
   License , known as the LLGPL and distributed with OpenMCL as the
   file "LICENSE".  The LLGPL consists of a preamble and the LGPL,
   which is distributed with OpenMCL as the file "LGPL".  Where these
   conflict, the preamble takes precedence.  

   OpenMCL is referenced in the preamble as the "LIBRARY."

   The LLGPL is also available online at
   http://opensource.franz.com/preamble.html
*/

#ifdef DARWIN
/*	dyld.h included here because something in "lisp.h" causes
    a conflict (actually I think the problem is in "constants.h")
*/
#include <mach-o/dyld.h>
#endif
#include "lisp.h"
#include "lisp_globals.h"
#include "gc.h"
#include "area.h"
#include <stdlib.h>
#include <string.h>
#include "lisp-exceptions.h"
#include <stdio.h>
#include <stdlib.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#ifdef LINUX
#include <mcheck.h>
#include <dirent.h>
#include <dlfcn.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <link.h>
#include <elf.h>
#include <asm/cputable.h>
#endif

#ifdef DARWIN
#include <sys/types.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <sys/resource.h>
#include <mach/mach_types.h>
#include <mach/message.h>
#include <mach/vm_region.h>
#include <sys/sysctl.h>
#endif

#ifdef VXWORKS
#include <limits.h>
#include <sys/stat.h>
#include <vxWorks.h>
#include <time.h>
#include <nfsLib.h>
#include <ioLib.h>
#include <taskLib.h>
#include <arch/ppc/vxPpcLib.h>
#endif
#include <ctype.h>
#ifndef VXWORKS
#include <sys/select.h>
#endif


LispObj lisp_nil = (LispObj) 0;


/* These are all "persistent" : they're initialized when
   subprims are first loaded and should never change. */
extern LispObj (*start_lisp)(LispObj, LispObj);
extern LispObj subprims_base;
extern LispObj ret1valn;
extern LispObj lexpr_return;
extern LispObj lexpr_return1v;
LispObj real_subprims_base = 0;
LispObj text_start = 0;

/* A pointer to some of the kernel's own data; also persistent. */

extern LispObj import_ptrs_base;


unsigned
align_to_power_of_2(unsigned n, unsigned power)
{
  unsigned align = (1<<power) -1;

  return (n+align) & ~align;
}


void
xMakeDataExecutable(void *, unsigned);

void
make_dynamic_heap_executable(LispObj *p, LispObj *q)
{
  void * cache_start = (void *) p;
  unsigned ncacheflush = (unsigned) q - (unsigned) p;

  xMakeDataExecutable(cache_start, ncacheflush);  
}
      
size_t
ensure_stack_limit(size_t stack_size)
{
  struct rlimit limits;
  rlim_t cur_stack_limit, max_stack_limit;
  
  stack_size += (CSTACK_HARDPROT+CSTACK_SOFTPROT);
  getrlimit(RLIMIT_STACK, &limits);
  cur_stack_limit = limits.rlim_cur;
  max_stack_limit = limits.rlim_max;
  if (stack_size > max_stack_limit) {
    stack_size = max_stack_limit;
  }
  if (cur_stack_limit < stack_size) {
    limits.rlim_cur = stack_size;
    errno = 0;
    if (setrlimit(RLIMIT_STACK, &limits)) {
      int e = errno;
      fprintf(stderr, "errno = %d\n", e);
      Fatal(": Stack resource limit too small", "");
    }
  }
  return stack_size - (CSTACK_HARDPROT+CSTACK_SOFTPROT);
}

#ifdef DARWIN
LispObj
darwin_remap_subprims(void *curloc)
{
  void *desired = (void *) (1<<20);

  if (curloc != desired) {
    void *ret = mmap(desired, 
                    1<<12,
                    PROT_READ|PROT_WRITE,
                    MAP_FIXED|MAP_ANON,
                    -1, 
                    0);

    if (ret != desired) {
      fprintf (stderr, "can't remap subprims \n");
      exit(1);
    } else {
      int i, disp;
      pc jtab = (pc) curloc, dest = (pc)ret;
      LispObj instr, target;
      
      for (i = 0; i < 256; i++, jtab++) {
       instr = *jtab;
       if (instr == 0) break;
       disp = instr & (~3 & ((1<<26)-1));
       target = (LispObj)jtab+disp;
       *dest++ = (BA_VAL) | target;
      }
      
      xMakeDataExecutable(ret, 1024);
    }
  }
  return (LispObj)desired;
}

#endif

/* This should write-protect the bottom of the stack.
   Doing so reliably involves ensuring that everything's unprotected on exit.
*/

static BytePtr
allocate_lisp_stack(unsigned useable,
                    unsigned softsize,
                    unsigned hardsize,
                    lisp_protection_kind softkind,
                    lisp_protection_kind hardkind,
                    Ptr *h_p,
                    BytePtr *base_p,
                    protected_area_ptr *softp,
                    protected_area_ptr *hardp)
{
  void *allocate_stack(unsigned);
  void free_stack(void *);
  unsigned size = useable+softsize+hardsize;
  unsigned overhead;
  BytePtr base, softlimit, hardlimit;
  OSErr err;
  Ptr h = allocate_stack(size+4095);
  protected_area_ptr hprotp = NULL, sprotp;

  if (h == NULL) {
    return NULL;
  }
  if (h_p) *h_p = h;
  base = (BytePtr) align_to_power_of_2((unsigned) h, 12);
  hardlimit = (BytePtr) (base+hardsize);
  softlimit = hardlimit+softsize;

  overhead = (base - (BytePtr) h);
  if (hardsize) {
    hprotp = new_protected_area((BytePtr)base,hardlimit,hardkind, hardsize, true);
    if (hprotp == NULL) {
      if (base_p) *base_p = NULL;
      if (h_p) *h_p = NULL;
      deallocate(h);
      return NULL;
    }
    if (hardp) *hardp = hprotp;
  }
  if (softsize) {
    sprotp = new_protected_area(hardlimit,softlimit, softkind, softsize, true);
    if (sprotp == NULL) {
      if (base_p) *base_p = NULL;
      if (h_p) *h_p = NULL;
      if (hardp) *hardp = NULL;
      if (hprotp) delete_protected_area(hprotp);
      free_stack(h);
      return NULL;
    }
    if (softp) *softp = sprotp;
  }
  if (base_p) *base_p = base;
  return (BytePtr) ((unsigned)(base+size));
}

/* This'll allocate a tstack or a vstack, but the thread
   mangler won't let us allocate or reliably protect
   a control stack.
*/
area *
allocate_lisp_stack_area(area_code stack_type,
                         unsigned useable, 
                         unsigned softsize, 
                         unsigned hardsize, 
                         lisp_protection_kind softkind, 
                         lisp_protection_kind hardkind)

{
  BytePtr base, bottom;
  Ptr h;
  area *a = NULL;
  protected_area_ptr soft_area=NULL, hard_area=NULL;

  bottom = allocate_lisp_stack(useable, 
                               softsize, 
                               hardsize, 
                               softkind, 
                               hardkind, 
                               &h, 
                               &base,
                               &soft_area, 
                               &hard_area);

  if (bottom) {
    a = new_area(base, bottom, stack_type);
    a->hardlimit = base+hardsize;
    a->softlimit = base+hardsize+softsize;
    a->h = h;
    a->softprot = soft_area;
    a->hardprot = hard_area;
    add_area(a);
  }
  return a;
}

area*
register_cstack(BytePtr bottom, unsigned size)
{
  BytePtr lowlimit = (BytePtr) (((((unsigned)bottom)-size)+4095)&~4095);
  area *a = new_area((BytePtr) bottom-size, bottom, AREA_CSTACK);

  a->hardlimit = lowlimit+CSTACK_HARDPROT;
  a->softlimit = a->hardlimit+CSTACK_SOFTPROT;
  add_area(a);
  return a;
}
  
area*
allocate_vstack(unsigned usable)
{
  return allocate_lisp_stack_area(AREA_VSTACK, 
				  usable > MIN_VSTACK_SIZE ?
				  usable : MIN_VSTACK_SIZE,
                                  VSTACK_SOFTPROT,
                                  VSTACK_HARDPROT,
                                  kVSPsoftguard,
                                  kVSPhardguard);
}

area *
allocate_tstack(unsigned usable)
{
  return allocate_lisp_stack_area(AREA_TSTACK, 
                                  usable > MIN_TSTACK_SIZE ?
				  usable : MIN_TSTACK_SIZE,
                                  TSTACK_SOFTPROT,
                                  TSTACK_HARDPROT,
                                  kTSPsoftguard,
                                  kTSPhardguard);
}

typedef struct LSIZ_resource LSIZ_resource, *LSIZ_ptr;

#pragma options align=mac68k
struct LSIZ_resource {
  long mac_heap_minimum;
  long mac_heap_maximum;
  short mac_heap_percentage;
  long low_memory_threshold;
  long copying_gc_threshold;
  long stack_minimum;
  long stack_maximum;
  short stack_percentage;
};
#pragma options align=reset

struct LSIZ_resource fake_LSIZ = {
  100 << 10,
  400 << 10,
  5,
  (64 << 10),
  0x7fffffff,
  (32 << 10),
  (180 << 10),
  6
};

/* It's hard to believe that max & min don't exist already */
unsigned unsigned_min(unsigned x, unsigned y)
{
  if (x <= y) {
    return x;
  } else {
    return y;
  }
}

unsigned unsigned_max(unsigned x, unsigned y)
{
  if (x >= y) {
    return x;
  } else {
    return y;
  }
}





#ifdef VXWORKS
int
lisp_partition_exists = 0;

Ptr
lisp_heap_partition = NULL;
#endif

int
reserved_area_size = (1<<30);

area *nilreg_area=NULL, *tenured_area=NULL, *g2_area=NULL, *g1_area=NULL;
area *all_areas=NULL;
int cache_block_size=32;

#ifdef VXWORKS
Ptr
create_memory_partition(unsigned totalsize)
{
  lisp_heap_partition = malloc(totalsize);
  if (lisp_heap_partition != NULL) {
    dl_malloc_init(lisp_heap_partition, totalsize);
    lisp_partition_exists = 1;
  } else {
    lisp_exit(-1);
  }
  return lisp_heap_partition;
}

#endif

#define DEFAULT_LISP_HEAP_GC_THRESHOLD (4<<20)
#define DEFAULT_INITIAL_STACK_SIZE (1<<20)

unsigned
lisp_heap_gc_threshold = DEFAULT_LISP_HEAP_GC_THRESHOLD;

unsigned 
initial_stack_size = DEFAULT_INITIAL_STACK_SIZE;


/*
  'start' should be on a segment boundary; 'len' should be
  an integral number of segments.
  remap the entire range, a segment at a time.
*/

void 
uncommit_pages(void *start, unsigned len)
{
  BytePtr p;

  /* LOCK_MMAP_LOCK(); */

  madvise(start, len, MADV_DONTNEED);
  
  for (p = start; len; len -= heap_segment_size, p += heap_segment_size) {
    munmap(p, heap_segment_size);
    if (p != mmap(p,
		  heap_segment_size,
		  PROT_NONE,
		  MAP_PRIVATE | MAP_FIXED | MAP_ANON,
		  -1,
		  0)) {
      Fatal("mmap error", "");
    }
  }
  /* UNLOCK_MMAP_LOCK(); */
}

void
commit_pages(void *start, unsigned len)
{
  BytePtr p;

  /* LOCK_MMAP_LOCK(); */

  for (p = start; len; len -= heap_segment_size, p += heap_segment_size) {
    munmap(p, heap_segment_size);
    if (p != mmap(p,
		  heap_segment_size,
		  PROT_READ | PROT_WRITE | PROT_EXEC,
		  MAP_PRIVATE | MAP_FIXED | MAP_ANON,
		  -1,
		  0)) {
      Fatal("mmap error", "");
    }
  }
  /* UNLOCK_MMAP_LOCK(); */
}

area *
extend_readonly_area(unsigned more)
{
  area *a;
  unsigned mask;
  BytePtr new_start, new_end;
  for (a = active_dynamic_area->succ; a != all_areas; a = a->succ) {
    if (a->code == AREA_READONLY) {
      if ((a->active + more) > a->high) {
	return NULL;
      }
      mask = ((unsigned)a->active) & 4095;
      if (mask) {
	UnProtectMemory(a->active-mask, 4096);
      }
      new_start = (BytePtr)(align_to_power_of_2((unsigned)a->active,12));
      new_end = (BytePtr)(align_to_power_of_2((unsigned)a->active+more,12));
      if (mmap(new_start,
	       new_end-new_start,
	       PROT_READ | PROT_WRITE | PROT_EXEC,
	       MAP_PRIVATE | MAP_ANON | MAP_FIXED,
	       -1,
	       0) != new_start) {
	return NULL;
      }
      return a;
    }
  }
  return NULL;
}

LispObj image_base=0;
BytePtr pure_space_start, pure_space_active, pure_space_limit;
BytePtr static_space_start, static_space_active, static_space_limit;

#ifdef DARWIN
/*
  Check to see if the specified address is unmapped by trying to get
  information about the mapped address at or beyond the target.  If
  the difference between the target address and the next mapped address
  is >= len, we can safely mmap len bytes at addr.
*/
Boolean
address_unmapped_p(char *addr, unsigned len)
{
  vm_address_t vm_addr = (vm_address_t)addr;
  vm_size_t vm_size;
  vm_region_basic_info_data_t vm_info;
  mach_msg_type_number_t vm_info_size = VM_REGION_BASIC_INFO_COUNT;
  port_t vm_object_name = (port_t) 0;
  kern_return_t kret;

  kret = vm_region(mach_task_self(),
		   &vm_addr,
		   &vm_size,
		   VM_REGION_BASIC_INFO,
		   (vm_region_info_t)&vm_info,
		   &vm_info_size,
		   &vm_object_name);
  if (kret != KERN_SUCCESS) {
    return false;
  }

  return vm_addr >= (vm_address_t)(addr+len);
}
#endif




area *
create_reserved_area(unsigned totalsize)
{
  OSErr err;
  Ptr h;
  unsigned base, n;
  BytePtr 
    end, 
    lastbyte, 
    start, 
    protstart, 
    p, 
    want = (BytePtr)IMAGE_BASE_ADDRESS,
    try2;
  area *reserved;
  bitvector markbits;
  Boolean fixed_map_ok = false;

  /*
    Through trial and error, we've found that IMAGE_BASE_ADDRESS is
    likely to reside near the beginning of an unmapped block of memory
    that's at least 1GB in size.  We'd like to load the heap image's
    sections relative to IMAGE_BASE_ADDRESS; if we're able to do so,
    that'd allow us to file-map those sections (and would enable us to
    avoid having to relocate references in the data sections.)

    In short, we'd like to reserve 1GB starting at IMAGE_BASE_ADDRESS
    by creating an anonymous mapping with mmap().

    If we try to insist that mmap() map a 1GB block at
    IMAGE_BASE_ADDRESS exactly (by specifying the MAP_FIXED flag),
    mmap() will gleefully clobber any mapped memory that's already
    there.  (That region's empty at this writing, but some future
    version of the OS might decide to put something there.)

    If we don't specify MAP_FIXED, mmap() is free to treat the address
    we give it as a hint; Linux seems to accept the hint if doing so
    wouldn't cause a problem.  Naturally, that behavior's too useful
    for Darwin (or perhaps too inconvenient for it): it'll often
    return another address, even if the hint would have worked fine.

    We call address_unmapped_p() to ask Mach whether using MAP_FIXED
    would conflict with anything.  Until we discover a need to do 
    otherwise, we'll assume that if Linux's mmap() fails to take the
    hint, it's because of a legitimate conflict.

    If Linux starts ignoring hints, we can parse /proc/<pid>/maps
    to implement an address_unmapped_p() for Linux.
  */

  totalsize = align_to_power_of_2(totalsize, log2_heap_segment_size);

#ifdef DARWIN
  fixed_map_ok = address_unmapped_p(want,totalsize);
#endif
  start = mmap((void *)want,
	       totalsize + heap_segment_size,
	       PROT_NONE,
	       MAP_PRIVATE | MAP_ANON | (fixed_map_ok ? MAP_FIXED : 0),
	       -1,
	       0);
  if (start == MAP_FAILED) {
    perror("Initial mmap");
    return NULL;
  }

  if (start != want) {
    munmap(start, totalsize+heap_segment_size);
    start = (void *)((((unsigned)start)+heap_segment_size-1) & ~(heap_segment_size-1));
    if(mmap(start, totalsize, PROT_NONE, MAP_PRIVATE | MAP_ANON | MAP_FIXED, -1, 0) != start) {
      return NULL;
    }
  }
  mprotect(start, totalsize, PROT_NONE);

  h = (Ptr) start;
  base = (unsigned) start;
  image_base = base;
  lastbyte = (BytePtr) (start+totalsize);
  static_space_start = static_space_active = start;
  static_space_limit = static_space_start + STATIC_RESERVE;
  pure_space_start = pure_space_active = static_space_limit;
  pure_space_limit = start + PURESPACE_RESERVE;
  start = pure_space_limit;

  /*
    Allocate mark bits here.  They need to be 1/64 the size of the
     maximum useable area of the heap (+ 3 words for the EGC.)
  */
  end = lastbyte;
  end = (BytePtr) ((unsigned)((((unsigned)end) - ((totalsize+63)>>6)) & ~4095));

  markbits = (bitvector)end;
  end = (BytePtr) ((unsigned)((((unsigned)end) - ((totalsize+63) >> 6)) & ~4095));
  global_reloctab = (LispObj *) end;
  reserved = new_area(start, end, AREA_VOID);
  /* The root of all evil is initially linked to itself. */
  reserved->pred = reserved->succ = reserved;
  all_areas = reserved;
  reserved->markbits = markbits;
  return reserved;
}

void *
allocate_from_reserved_area(unsigned size)
{
  area *reserved = reserved_area;
  BytePtr low = reserved->low, high = reserved->high;
  unsigned avail = high-low;
  size = align_to_power_of_2(size, log2_heap_segment_size);

  if (size > avail) {
    return NULL;
  }
  reserved->low += size;
  reserved->active = reserved->low;
  reserved->ndwords -= (size>>3);
  return low;
}


#define FILE_MAP_FROM_RESERVED_AREA 0

void *
file_map_reserved_pages(unsigned len, int prot, int fd, unsigned offset)
{
  void *start;
  unsigned 
    offset_of_page = offset & ~((1<<12)-1), 
    offset_in_page = offset - offset_of_page,
    segment_len = align_to_power_of_2((offset+len)-offset_of_page, 
				      log2_heap_segment_size);
  
  /* LOCK_MMAP_LOCK(); */
#if FILE_MAP_FROM_RESERVED_AREA
  start = allocate_from_reserved_area(segment_len);
  if (start == NULL) {
    return start;
  }
#endif
#if FILE_MAP_FROM_RESERVED_AREA
  if (start != mmap(start,
		    segment_len,
		    prot,
		    MAP_PRIVATE | MAP_FIXED,
		    fd,
		    offset_of_page)) {
    return NULL;
  }
#else
  if ((start = mmap(NULL,
		    segment_len,
		    prot,
		    MAP_PRIVATE,
		    fd,
		    offset_of_page)) == (void *)-1) {
    return NULL;
  }
#endif
  /* UNLOCK_MMAP_LOCK(); */
  return (void *) (((unsigned)start) + offset_in_page);
}

void
ensure_gc_structures_writable()
{
  area *a = active_dynamic_area;
  unsigned 
    ndwords = a->ndwords,
    markbits_size = 12+((a->ndwords+7)>>3),
    reloctab_size = (sizeof(LispObj)*(((ndwords+31)>>5)+1));

  UnProtectMemory(global_reloctab, reloctab_size);
  UnProtectMemory(a->markbits, markbits_size);

}

area *
allocate_dynamic_area(unsigned initsize)
{
  unsigned totalsize = align_to_power_of_2(initsize + (heap_segment_size * 2)
					   , log2_heap_segment_size);
  BytePtr start, end, p, q;
  protected_area_ptr hardp, softp;
  area *a;

  start = allocate_from_reserved_area(totalsize);
  if (start == NULL) {
    return NULL;
  }
  end = start + totalsize;
  p = end - heap_segment_size;
  a = new_area(start, p, AREA_DYNAMIC);
  a->active = start+initsize;
  add_area(a);
  a->markbits = reserved_area->markbits;
  reserved_area->markbits = NULL;
  hardp = new_protected_area(p, end, kHEAPhard, heap_segment_size, false);
  q = p- (((p - a->active)>>log2_heap_segment_size)<<log2_heap_segment_size);
  softp = new_protected_area(q, p, kHEAPsoft, heap_segment_size, true);
  UnProtectMemory(start, q-start);
  a->h = start;
  a->softprot = softp;
  a->hardprot = hardp;
  a->hardlimit = p;
  ensure_gc_structures_writable();
  return a;
}


/*
  The dynamic area's hard protected_area should be unprotected when
  this is called (it'll move.)  

  The caller should decide whether or not this is really a good idea.
*/

Boolean
grow_dynamic_area(unsigned delta)
{
  area *a = active_dynamic_area, *reserved = reserved_area;
  protected_area_ptr hardp = a->hardprot, softp = a->softprot;
  unsigned avail = reserved->high - reserved->low;
  
  delta = align_to_power_of_2(delta, log2_heap_segment_size);
  if (delta > avail) {
    delta = avail;
  }
  if (!allocate_from_reserved_area(delta)) {
    return false;
  }
  a->high += delta;
  a->ndwords = area_dword(a->high, a->low);
  hardp->start += delta;
  hardp->end += delta;
  softp->start += delta;
  softp->end += delta;
  a->hardlimit = hardp->start;
  lisp_global(HEAP_END) += delta;
  ensure_gc_structures_writable();
  return true;
}

/*
  As above.  Pages that're returned to the reserved_area are
  "condemned" (e.g, we try to convince the OS that they never
  existed ...)
*/
Boolean
shrink_dynamic_area(unsigned delta)
{
  area *a = active_dynamic_area, *reserved = reserved_area;
  protected_area_ptr hardp = a->hardprot, softp = a->softprot;
  
  delta = align_to_power_of_2(delta, log2_heap_segment_size);

  a->high -= delta;
  a->ndwords = area_dword(a->high, a->low);
  hardp->start -= delta;
  hardp->end -= delta;
  softp->start -= delta;
  softp->end -= delta;
  a->hardlimit = hardp->start;
  uncommit_pages(hardp->end, delta);
  reserved->low -= delta;
  reserved->ndwords += (delta>>3);
  lisp_global(HEAP_END) -= delta;
  return true;
}


/* 
 *interrupt-level* is >= 0 when interrupts are enabled and < 0
 during without-interrupts. Normally, it is 0. When this timer
 goes off, it sets it to 1 if it's 0, or if it's negative,
 walks up the special binding list looking for a previous
 value of 0 to set to 1. 
*/
void
preemption_handler(int level)
{
  struct lispsymbol *interrupt_level_sym = &nrs_INTERRUPT_LEVEL;
  int interrupt_level = (int) interrupt_level_sym->vcell;
  LispObj tagged_il_sym = ((LispObj) interrupt_level_sym) + fulltag_misc;

  lisp_global(TICKS) += (1<<fixnumshift); /* Screw: handle overflow ? */

  if (interrupt_level == 0) {
    nrs_INTERRUPT_LEVEL.vcell = level<<fixnumshift;
  }
  else if ((tag_of(interrupt_level) == tag_fixnum) && /* This test may not be necessary */
           (interrupt_level < 0)) {
    struct special_binding *b;
    LispObj b_value;
    for (b = (special_binding *) lisp_global(DB_LINK); b != 0; b = b->link) {
      if (((b->sym) == (lispsymbol *) tagged_il_sym)) {
        if ((b_value = (b->value)) == 0) {
          b->value = level<<fixnumshift;
          break;
        }
        else if (b_value > 0) {
          break;
        }
      }
    }
  }
}

#ifndef VXWORKS
typedef struct {
  int total_hits;
  int lisp_hits;
  int active;
  int interval;
} metering_info;

metering_info
lisp_metering =
{
  0, 
  0, 
  0, 
  0
  };

void
metering_proc(int signum, struct sigcontext *context)
{
  lisp_metering.total_hits++;
#ifndef DARWIN
  if (xpGPR(context,rnil) == lisp_nil) {
    unsigned current_lisp = lisp_metering.lisp_hits, element;
    LispObj 
      rpc = (LispObj) xpPC(context),
      rfn = xpGPR(context, fn),
      rnfn = xpGPR(context, nfn),
      reg,
      v =  nrs_ALLMETEREDFUNS.vcell;

    if (area_containing((BytePtr)rfn) == NULL) {
      rfn = (LispObj) 0;
    }
    if (area_containing((BytePtr)rnfn) == NULL) {
      rnfn = (LispObj) 0;
    }

    if (tag_of(rpc) == tag_fixnum) {
      if (register_codevector_contains_pc(rfn, rpc)) {
	reg = rfn;
      } else if (register_codevector_contains_pc(rnfn, rpc)) {
	reg = rnfn;
      } else {
	reg = rpc;
      }
      element = current_lisp % lisp_metering.active;
      lisp_metering.lisp_hits++;
      deref(v,element+1) = reg; /* NOT memoized */
    }
  }
#endif
}
#endif

#if defined(LINUX) || defined(DARWIN)
#define WHAT_ITIMER ITIMER_REAL
#define WHAT_TIMER_SIGNAL SIGALRM
#endif


void
alarm_handler (int signum, struct sigcontext *context)
{
  if (signum == WHAT_TIMER_SIGNAL) {
    preemption_handler(1);
    if (lisp_metering.active) {
      metering_proc(signum, context);
    }
  }
  if (signum == SIGINT) {
    lisp_global(INTFLAG) = (1 << fixnumshift);
  }
}




/* Start up the VBL task that frobs *interrupt-level* to cause an interrupt. */
#if defined(LINUX) || defined(DARWIN)
void
start_vbl()
{
  static struct itimerval vbl_timer = {
    {0, 0},
    {0, 0}};
  int 
    ticks_per_second = sysconf(_SC_CLK_TCK), 
    usec_per_tick = 1000000 / ticks_per_second;
  vbl_timer.it_interval.tv_usec = usec_per_tick;
  vbl_timer.it_value.tv_usec = usec_per_tick;
  install_signal_handler(WHAT_TIMER_SIGNAL, (__sighandler_t)alarm_handler);
  setitimer(WHAT_ITIMER, &vbl_timer, NULL);
  signal(SIGINT, (__sighandler_t)alarm_handler);

}
#endif

#ifdef VXWORKS
int vxworks_timer_enabled = 1;

void
vxworks_timer_proc()
{
  while(vxworks_timer_enabled) {
    taskDelay(1);
    preemption_handler(1);
  }
  exit(0);
}

void
start_vbl()
{
  WIND_TCB *tcb = taskTcb(0);
  int priority = tcb->priority - 1;
  extern int lisp_timer_task;

  if (priority < 0) {
    priority = 0;
  }
  lisp_timer_task = taskSpawn("ppccl_timer", priority, 0, 8<<10, (FUNCPTR) vxworks_timer_proc, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10);
  signal(SIGINT, alarm_handler);
}

#endif

extern BytePtr
current_stack_pointer(void);

BytePtr
initial_stack_bottom()
{
  extern char **environ;
  char *p = *environ;
  while (*p) {
    p += (1+strlen(p));
  }
  return (BytePtr)((((unsigned) p) +4095) & ~4095);
}


  
Ptr fatal_spare_ptr = NULL;

void
prepare_for_the_worst()
{
  /* I guess that CouldDialog is no more */
  /* CouldDialog(666); */
}

void
Fatal(StringPtr param0, StringPtr param1)
{

  if (fatal_spare_ptr) {
    deallocate(fatal_spare_ptr);
    fatal_spare_ptr = NULL;
  }
  fprintf(stderr, "Fatal error: %s\n%s\n", param0, param1);
  lisp_exit(-1);
}

OSErr application_load_err = noErr;
extern Boolean load_subprims(char *);
#ifdef VXWORKS
extern Boolean load_vxlow(char *);
#endif

area *
set_nil(LispObj);

#ifdef VXWORKS
extern char *
strdup(char *);
char * default_lisp_image_name = "./VXPPCCL";
#endif

#ifdef DARWIN
/* 
   The underlying file system may be case-insensitive (e.g., HFS),
   so we can't just case-invert the kernel's name.
   Tack ".image" onto the end of the kernel's name.  Much better ...
*/
char *
default_image_name(char *orig)
{
  int len = strlen(orig) + strlen(".image") + 1;
  char *copy = (char *) malloc(len);

  if (copy) {
    strcat(copy, orig);
    strcat(copy, ".image");
  }
  return copy;
}

#else
char *
default_image_name(char *orig)
{
  char *copy = strdup(orig), *base = copy, *work = copy, c;
  if (copy == NULL) {
    return NULL;
  }
  while(*work) {
    if (*work++ == '/') {
      base = work;
    }
  }
  work = base;
  while (c = *work) {
    if (islower(c)) {
      *work++ = toupper(c);
    } else {
      *work++ = tolower(c);
    }
  }
  return copy;
}
#endif

/*
  Cleanup everything so that we can run some lisp image again.
  Free everything allocated in the lisp_heap_partition and
  reset the (private) malloc that we use.
  */

void
image_memory_cleanup()
{
#ifdef VXWORKS
  KG = NULL;
  free(lisp_heap_partition);
  lisp_partition_exists = 0;
  zone_malloc_reset();
  lisp_nil = (LispObj)NULL;
#endif
}

char *program_name = NULL;

void
usage_exit(char *herald, int exit_status, char* other_args)
{
  if (herald && *herald) {
    fprintf(stderr, "%s\n", herald);
  }
  fprintf(stderr, "usage: %s <options>\n", program_name);
  fprintf(stderr, "\t or %s <image-name>\n", program_name);
  fprintf(stderr, "\t where <options> are one or more of:\n");
  if (other_args && *other_args) {
    fputs(other_args, stderr);
  }
  fprintf(stderr, "\t-R, --heap-reserve <n>: reserve <n> (default: %d)\n",
	  reserved_area_size);
  fprintf(stderr, "\t\t bytes for heap expansion\n");
  fprintf(stderr, "\t-S, --stack-size <n>: set size of initial stacks to <n> (default: %d)\n", initial_stack_size);
  fprintf(stderr, "\t-b, --batch: exit when EOF on *STANDARD-INPUT*\n");
  fprintf(stderr, "\t--no-sigtrap : obscure option for running under GDB\n");
  fprintf(stderr, "\t-I, --image-name <image-name>\n");
  fprintf(stderr, "\t and <image-name> defaults to %s\n", 
	  default_image_name(program_name));
  fprintf(stderr, "\n");
  exit(exit_status);
}

int no_sigtrap = 0;
char *image_name = NULL;
int batch_flag = 0;

#ifndef VXWORKS

unsigned
parse_numeric_option(char *arg, char *argname, unsigned default_val)
{
  char *tail;
  unsigned val = 0;

  val = strtoul(arg, &tail, 0);
  switch(*tail) {
  case '\0':
    break;
    
  case 'M':
  case 'm':
    val = val << 20;
    break;
    
  case 'K':
  case 'k':
    val = val << 10;
    break;
    
  case 'G':
  case 'g':
    val = val << 30;
    break;
    
  default:
    fprintf(stderr, "couldn't parse %s argument %s", argname, arg);
    val = default_val;
    break;
  }
  return val;
}
  


/* 
   The set of arguments recognized by the kernel is
   likely to remain pretty small and pretty simple.
   This removes everything it recognizes from argv;
   remaining args will be processed by lisp code.
*/

void
process_options(int argc, char *argv[])
{
  int i, j, k, num_elide, flag, arg_error;
  char *arg, *val;
#ifdef DARWIN
  extern int NXArgc;
#endif

  for (i = 1; i < argc;) {
    arg = argv[i];
    arg_error = 0;
    if (*arg != '-') {
      i++;
    } else {
      num_elide = 0;
      val = NULL;
      if ((flag = (strncmp(arg, "-I", 2) == 0)) ||
	  (strcmp (arg, "--image-name") == 0)) {
	if (flag && arg[2]) {
	  val = arg+2;
	  num_elide = 1;
	} else {
	  if ((i+1) < argc) {
	    val = argv[i+1];
	    num_elide = 2;
	  } else {
	    arg_error = 1;
	  }
	}
	if (val) {
	  image_name = val;
	}
      } else if ((flag = (strncmp(arg, "-R", 2) == 0)) ||
		 (strcmp(arg, "--heap-reserve") == 0)) {
	unsigned reserved_size;

	if (flag && arg[2]) {
	  val = arg+2;
	  num_elide = 1;
	} else {
	  if ((i+1) < argc) {
	    val = argv[i+1];
	    num_elide = 2;
	  } else {
	    arg_error = 1;
	  }
	}

	if (val) {
	  reserved_size = parse_numeric_option(val, 
					       "-R/--heap-reserve", 
					       reserved_area_size);
	}

	if (reserved_size <= (1<< 30)) {
	  reserved_area_size = reserved_size;
	}

      } else if ((flag = (strncmp(arg, "-S", 2) == 0)) ||
		 (strcmp(arg, "--stack-size") == 0)) {
	unsigned stack_size;

	if (flag && arg[2]) {
	  val = arg+2;
	  num_elide = 1;
	} else {
	  if ((i+1) < argc) {
	    val = argv[i+1];
	    num_elide = 2;
	  } else {
	    arg_error = 1;
	  }
	}

	if (val) {
	  stack_size = parse_numeric_option(val, 
					    "-S/--stack-size", 
					    initial_stack_size);
	  

	  if (stack_size >= MIN_CSTACK_SIZE) {
	    initial_stack_size = stack_size;
	  }
	}

      } else if (strcmp(arg, "--no-sigtrap") == 0) {
	no_sigtrap = 1;
	num_elide = 1;
      } else if ((strcmp(arg, "-b") == 0) ||
		 (strcmp(arg, "--batch") == 0)) {
	batch_flag = 1;
	num_elide = 1;
      } else {
	i++;
      }
      if (arg_error) {
	usage_exit("error in program arguments", 1, "");
      }
      if (num_elide) {
	for (j = i+num_elide, k=i; j < argc; j++, k++) {
	  argv[k] = argv[j];
	}
	argc -= num_elide;
#ifdef DARWIN
	NXArgc -= num_elide;
#endif
	argv[argc] = NULL;
      }
    }
  }
}



main(int argc, char *argv[], char *envp[], void *aux)
#endif
#ifdef VXWORKS
vxppcclMain()
#endif
{
#ifdef VXWORKS
  int argc = 1;
  char *argv[] = {"vxppccl", 0};
#endif
  extern  set_fpscr(unsigned);

  extern int altivec_present;
  extern LispObj load_image(char *);
  long resp;
  BytePtr stack_end;
  area *a;
  BytePtr stack_base, current_sp = current_stack_pointer();

#ifdef LINUX
  {
    ElfW(auxv_t) *av = aux;
    int hwcap, done = false;
    
    if (av) {
      do {
	switch (av->a_type) {
	case AT_DCACHEBSIZE:
	  cache_block_size = av->a_un.a_val;
	  break;

	case AT_HWCAP:
	  hwcap = av->a_un.a_val;
	  altivec_present = ((hwcap & PPC_FEATURE_HAS_ALTIVEC) != 0);
	  break;

	case AT_NULL:
	  done = true;
	  break;
	}
	av++;
      } while (!done);
    }
  }
#endif
#ifdef DARWIN
  {
    unsigned value = 0;
    size_t len = sizeof(value);
    int mib[2];
    
    mib[0] = CTL_HW;
    mib[1] = HW_CACHELINE;
    if (sysctl(mib,2,&value,&len, NULL, 0) != -1) {
      if (len == sizeof(value)) {
	cache_block_size = value;
      }
    }
    mib[1] = HW_VECTORUNIT;
    value = 0;
    len = sizeof(value);
    if (sysctl(mib,2,&value,&len, NULL, 0) != -1) {
      if (len == sizeof(value)) {
	altivec_present = value;
      }
    }
  }
#endif

#if defined(LINUX) || defined(DARWIN)

  program_name = argv[0];
  if ((argc == 2) && (*argv[1] != '-')) {
#ifdef DARWIN
    extern int NXArgc;
    NXArgc = 1;
#endif
    image_name = argv[1];
    argv[1] = NULL;
  } else {
    process_options(argc,argv);
  }
  initial_stack_size = ensure_stack_limit(initial_stack_size);
  if (image_name == NULL) {
    image_name = default_image_name(argv[0]);
  }

#endif
#ifdef VXWORKS
#if 0
    ioTaskStdSet(0, 2, ioTaskStdGet(0, 1));
#endif
    image_name = default_lisp_image_name;
#endif

#if 0
  fcntl(0, F_SETFL, O_NONBLOCK |  fcntl(0, F_GETFL));
#endif

  prepare_for_the_worst();

  if (!load_subprims("./lisp_subprims.o")) {
    fprintf(stderr, "Couldn't load subprims.\n");
    lisp_exit(-1);
  }

  real_subprims_base = subprims_base;
#ifdef DARWIN
  real_subprims_base = darwin_remap_subprims((void *)subprims_base);
#endif
#ifdef VXWORKS
  if (!load_vxlow("./vxlow.o")) {
    fprintf(stderr, "Couldn't load vxlow module.\n");
    lisp_exit(-1);
  }
#endif
  create_reserved_area(reserved_area_size);
  set_nil(load_image(image_name));
  lisp_global(SUBPRIMS_BASE) = subprims_base;
  lisp_global(RET1VALN) = ret1valn;
  lisp_global(LEXPR_RETURN) = lexpr_return;
  lisp_global(LEXPR_RETURN1V) = lexpr_return1v;
  lisp_global(ALL_AREAS) = (LispObj) (all_areas);

  exception_init();
#ifdef PROXY_SCHEDULER
  puppet_init();
#endif

  if (lisp_global(SUBPRIMS_BASE) == 0) {
    Fatal(": Couldn't load subprims library.", "");
  }
  
  set_fpscr(0xD0);

  lisp_global(IMAGE_NAME) = (LispObj) image_name;
  lisp_global(ARGV) = (LispObj) argv;
  lisp_global(KERNEL_IMPORTS) = (LispObj) import_ptrs_base;
  a = active_dynamic_area;
  lisp_global(SAVE_FREEPTR) = (LispObj) (a->active);

  lisp_global(METERING_INFO) = (LispObj) &lisp_metering;

  *(double *) &(lisp_global(DOUBLE_FLOAT_ONE)) = (double) 1.0;
  lisp_global(EXCEPTION_SAVED_REGISTERS) = (LispObj) 0;

  lisp_global(ARGV) = (LispObj) argv;
  lisp_global(ERRNO) = (LispObj) &errno;
  lisp_global(HOST_PLATFORM) = (LispObj)
#ifdef LINUX
    1
#endif
#ifdef VXWORKS
    2
#endif
#ifdef DARWIN
    3
#endif
    /* We'll get a syntax error here if nothing's defined. */
    << fixnumshift;


  lisp_global(BATCH_FLAG) = (batch_flag << fixnumshift);

  if (nilreg_area != NULL) {
    BytePtr lowptr = (BytePtr) a->low;

    a = active_dynamic_area;
    /* Create these areas as AREA_STATIC, change them to AREA_DYNAMIC */
    g1_area = new_area(lowptr, lowptr, AREA_STATIC);
    g2_area = new_area(lowptr, lowptr, AREA_STATIC);
    tenured_area = new_area(lowptr, lowptr, AREA_STATIC);
    add_area(tenured_area);
    add_area(g2_area);
    add_area(g1_area);

    g1_area->code = AREA_DYNAMIC;
    g2_area->code = AREA_DYNAMIC;
    tenured_area->code = AREA_DYNAMIC;

/*    a->older = g1_area; */ /* Not yet: this is what "enabling the EGC" does. */
    g1_area->younger = a;
    g1_area->older = g2_area;
    g2_area->younger = g1_area;
    g2_area->older = tenured_area;
    tenured_area->younger = g2_area;
    tenured_area->refbits = a->markbits;
    lisp_global(TENURED_AREA) = (LispObj)(tenured_area);
    g2_area->threshold = (1<<18); /* 256K */
    g1_area->threshold = (1<<17); /* 128K */
    a->threshold = (1<<16);     /* 64K */
  }

  memo_base =
    (BytePtr) allocate_lisp_stack(1<<15, 1<<12, 0, kMEMOprotect, kNotProtected, NULL, NULL, NULL, NULL);
  lisp_global(SAVE_MEMO) = (LispObj) memo_base;
      
  a = allocate_vstack(initial_stack_size);
  stack_end = a->high;
  lisp_global(SAVE_VSP) = (LispObj) stack_end;
  lisp_global(CURRENT_VS) = (LispObj) a;

  a = allocate_tstack(MIN_TSTACK_SIZE);
  stack_end = a->high;
  lisp_global(SAVE_TSP) = (LispObj) stack_end;
  lisp_global(CURRENT_TS) = (LispObj) a;

  stack_base = initial_stack_bottom()-xStackSpace();
  a = register_cstack(current_sp, current_sp-stack_base);
#ifndef MACOS
  init_mac_threads((void *)(stack_base));
#endif
  lisp_global(CURRENT_CS) = (LispObj) a;
  lisp_global(CS_OVERFLOW_LIMIT) = (LispObj) (a->softlimit);

  start_vbl();

  lisp_global(ALTIVEC_PRESENT) = altivec_present << fixnumshift;


  start_lisp(lisp_nil, 0);
  lisp_exit(0);
}

int
lisp_exit(int status)
{
#ifdef VXWORKS
  vxworks_timer_enabled = 0;
#endif
  exception_cleanup();
  image_memory_cleanup();
  exit(status);
  return status;
}


mutable_data_section_header **next_libP = NULL;
unsigned next_libnum = 0;


area *
set_nil(LispObj r)
{

  if (lisp_nil == (LispObj)NULL) {

    lisp_nil = r;
  }
  return NULL;
}






#ifdef VXWORKS
char *strdup(char *src)
{
  char *dest = (char *) allocate(strlen(src)+1);
  if (dest != NULL) {
    strcpy(dest, src);
  }
  return dest;
}
#endif

      
#ifndef MACOS
void
xMakeDataExecutable(void *start, unsigned nbytes)
{
  extern void flush_cache_lines();
  unsigned ustart = (unsigned) start, base, end;
  
  base = (ustart) & ~(cache_block_size-1);
  end = (ustart + nbytes + cache_block_size - 1) & ~(cache_block_size-1);
  flush_cache_lines(base, (end-base)/cache_block_size, cache_block_size);
}

int
xStackSpace()
{
#if defined(LINUX) || defined(DARWIN)
  return initial_stack_size+CSTACK_HARDPROT+CSTACK_SOFTPROT;
#endif
#ifdef VXWORKS
  WIND_TCB *tcb = taskTcb(0);
  char here;
  return &here - tcb->pStackLimit;
#endif
}

#ifndef DARWIN
void *
xGetSharedLibrary(char *path, int mode)
{
#ifdef VXWORKS
  return NULL;
#else
  return dlopen(path, mode);
#endif
}
#else
void *
xGetSharedLibrary(char *path, int mode)
{
  NSObjectFileImageReturnCode code;
  NSObjectFileImage	         moduleImage;
  NSModule		         module;
  const struct mach_header *     header;
  const char *                   error;
  int *                          resultType;
  void *                         result;
  /* not thread safe */
  /*
  static struct {
    const struct mach_header  *header;
    NSModule	              *module;
    const char                *error;
  } results;	
  */
  resultType = (int *)mode;
  result = NULL;
  error = NULL;

  /* first try to open this as a bundle */
  code = NSCreateObjectFileImageFromFile(path,&moduleImage);
  if (code != NSObjectFileImageSuccess &&
      code != NSObjectFileImageInappropriateFile &&
      code != NSObjectFileImageAccess)
    {
      /* compute error strings */
      switch (code)
	{
	case NSObjectFileImageFailure:
	  error = "NSObjectFileImageFailure";
	  break;
	case NSObjectFileImageArch:
	  error = "NSObjectFileImageArch";
	  break;
	case NSObjectFileImageFormat:
	  error = "NSObjectFileImageFormat";
	  break;
	case NSObjectFileImageAccess:
	  /* can't find the file */
	  error = "NSObjectFileImageAccess";
	  break;
	default:
	  error = "unknown error";
	}
      *resultType = 0;
      return (void *)error;
    }
  if (code == NSObjectFileImageInappropriateFile ||
      code == NSObjectFileImageAccess ) {
    /* the pathname might be a partial pathane (hence the access error)
       or it might be something other than a bundle, if so perhaps
       it is a .dylib so now try to open it as a .dylib */

    /* protect against redundant loads, Gary Byers noticed possible
       heap corruption if this isn't done */
    header = NSAddImage(path, NSADDIMAGE_OPTION_RETURN_ON_ERROR |
			NSADDIMAGE_OPTION_WITH_SEARCHING |
			NSADDIMAGE_OPTION_RETURN_ONLY_IF_LOADED);
    if (!header)
      header = NSAddImage(path, NSADDIMAGE_OPTION_RETURN_ON_ERROR |
			  NSADDIMAGE_OPTION_WITH_SEARCHING);
    result = (void *)header;
    *resultType = 1;
  }
  else if (code == NSObjectFileImageSuccess) {
    /* we have a sucessful module image
       try to link it, don't bind symbols privately */

    module = NSLinkModule(moduleImage, path,
			  NSLINKMODULE_OPTION_RETURN_ON_ERROR | NSLINKMODULE_OPTION_BINDNOW);
    NSDestroyObjectFileImage(moduleImage);	
    result = (void *)module;
    *resultType = 2;
  }
  if (!result)
    {
      /* compute error string */
      NSLinkEditErrors ler;
      int lerno;
      const char* file;
      NSLinkEditError(&ler,&lerno,&file,&error);
      if (error) {
	result = (void *)error;
	*resultType = 0;
      }
    }
  return result;
}
#endif




int
metering_control(int interval)
{
#ifdef DARWIN
  return -1;
#else
  if (interval) {
    if (! lisp_metering.active) {
      LispObj amf = nrs_ALLMETEREDFUNS.vcell;
      if (fulltag_of(amf) == fulltag_misc) {
        unsigned header = header_of(amf);

        if (header_subtag(header) == subtag_simple_vector) {

          lisp_metering.interval = interval;
          lisp_metering.total_hits = 0;
          lisp_metering.lisp_hits = 0;
          lisp_metering.active = header_element_count(header);
          return 0;
        }
      }
    }
    return -1;
  }  else {
    if (lisp_metering.active) {
      lisp_metering.active = 0;
      return 0;
    } else {
      return -1;
    }
  }
#endif
}


#ifdef VXWORKS
#include <sys/stat.h>

struct linux_stat {
  int st_dev_high;
  int st_dev;
  unsigned __pad1;
  unsigned st_ino;
  unsigned st_mode;
  unsigned st_nlink;
  unsigned st_uid;
  unsigned st_gid;
  unsigned st_rdev_high;
  unsigned st_rdev;
  unsigned __pad2;
  long st_size;
  unsigned st_blksize;
  long st_blocks;
  unsigned st_atime;
  unsigned __unused1;
  unsigned st_mtime;
  unsigned __unused2;
  unsigned st_ctime;
  unsigned __unused3;
  unsigned __unused4;
  unsigned __unused5;
};

void
copy_stat_results(struct linux_stat *l, struct stat *s)
{
  l->st_dev = s->st_dev;
  l->st_ino = s->st_dev;
  l->st_mode = s->st_mode;
  l->st_nlink = s->st_nlink;
  l->st_uid = s->st_uid;
  l->st_gid = s->st_gid;
  l->st_rdev = s->st_rdev;
  l->st_size = s->st_size;
  l->st_atime = s->st_atime;
  l->st_mtime = s->st_mtime;
  l->st_ctime = s->st_ctime;
  l->st_blksize = s->st_blksize;
  l->st_blocks = s->st_blocks;
}
    
int
__xstat(unsigned long version, char *name, struct linux_stat *l)
{
  struct stat s_, *s = &s_;
  int result = stat(name, s);
  if (result == 0) {
    copy_stat_results(l,s);
  }
  return result;
}

int
__lxstat(unsigned long version, char *name, struct linux_stat *l)
{
  /* VxWorks doesn't seem to have an "lstat". */
  return __xstat(version,name,l);
}

void
report_unimplemented(char *name)
{
  char buf[128];
  sprintf(buf, "call to unimplemented function %s\n", name);
  Bug(NULL,buf);
}

int
gettimeofday(struct timespec *tm, void *tz)
{
  clock_gettime(CLOCK_REALTIME, tm);
  tm->tv_nsec /= 1000;
  return 0;
}



struct rusage
{
  /* Total amount of user time used.  */
  struct timespec ru_utime;
  /* Total amount of system time used.  */
  struct timespec ru_stime;
  /* Maximum resident set size (in kilobytes).  */
  long int ru_maxrss;
  /* Amount of sharing of text segment memory
     with other processes (kilobyte-seconds).  */
  long int ru_ixrss;
  /* Amount of data segment memory used (kilobyte-seconds).  */
  long int ru_idrss;
  /* Amount of stack memory used (kilobyte-seconds).  */
  long int ru_isrss;
  /* Number of soft page faults (i.e. those serviced by reclaiming
     a page from the list of pages awaiting reallocation.  */
  long int ru_minflt;
  /* Number of hard page faults (i.e. those that required I/O).  */
  long int ru_majflt;
  /* Number of times a process was swapped out of physical memory.  */
  long int ru_nswap;
  /* Number of input operations via the file system.  Note: This
     and `ru_oublock' do not include operations with the cache.  */
  long int ru_inblock;
  /* Number of output operations via the file system.  */
  long int ru_oublock;
  /* Number of IPC messages sent.  */
  long int ru_msgsnd;
  /* Number of IPC messages received.  */
  long int ru_msgrcv;
  /* Number of signals delivered.  */
  long int ru_nsignals;
  /* Number of voluntary context switches, i.e. because the process
     gave up the process before it had to (usually to wait for some
     resource to be available).  */
  long int ru_nvcsw;
  /* Number of involuntary context switches, i.e. a higher priority process
     became runnable or the current process used up its time slice.  */
  long int ru_nivcsw;
};

int 
getrusage(int who, struct rusage *usage)
{
  gettimeofday(&usage->ru_utime, NULL);
  usage->ru_stime.tv_sec = 0;
  usage->ru_stime.tv_nsec = 0;
  return 0;
}

#define Undefined(name) void name() {report_unimplemented(#name);}
Undefined(__tcgetattr)
Undefined(acosh)
Undefined(atanh)
Undefined(asinh)

struct passwd {
  char    *pw_name;       /* user name */
  char    *pw_passwd;     /* user password */
  int   pw_uid;         /* user id */
  int   pw_gid;         /* group id */
  char    *pw_gecos;      /* real name */
  char    *pw_dir;        /* home directory */
  char    *pw_shell;      /* shell program */
};

struct passwd
fmh = {
  "byers",
  "no",
  4954,
  4023,
  "guess",
  "/home/byers",
  "sh"
};

struct passwd *
getpwuid(int uid)
{
  if (uid = fmh.pw_uid) {
    return &fmh;
  }
  return NULL;
}

int
getuid()
{
  char host[512];
  int uid, gid, ngids = 0, gidbuf[100];

  nfsAuthUnixGet(host, &uid, &gid, &ngids, gidbuf);
  return uid;
}




static char *
canonicalize (const char *name, char *resolved)
{
  char *rpath, *dest, *extra_buf = NULL;
  const char *start, *end, *rpath_limit;
  long int path_max;
  int num_links = 0;

  if (name == NULL)
    {
      /* As per Single Unix Specification V2 we must return an error if
	 either parameter is a null pointer.  We extend this to allow
	 the RESOLVED parameter be NULL in case the we are expected to
	 allocate the room for the return value.  */
      errnoSet (EINVAL);
      return NULL;
    }

  if (name[0] == '\0')
    {
      /* As per Single Unix Specification V2 we must return an error if
	 the name argument points to an empty string.  */
      errnoSet (ENOENT);
      return NULL;
    }

  path_max = PATH_MAX;

  rpath = resolved ? alloca (path_max) : malloc (path_max);
  rpath_limit = rpath + path_max;

  if (name[0] != '/')
    {
      if (!xgetcwd (rpath, path_max))
	goto error;
      dest = strchr (rpath, '\0');
    }
  else
    {
      rpath[0] = '/';
      dest = rpath + 1;
    }

  for (start = end = name; *start; start = end)
    {
      struct stat st;
      int n;

      /* Skip sequence of multiple path-separators.  */
      while (*start == '/')
	++start;

      /* Find end of path component.  */
      for (end = start; *end && *end != '/'; ++end)
	/* Nothing.  */;

      if (end - start == 0)
	break;
      else if (end - start == 1 && start[0] == '.')
	/* nothing */;
      else if (end - start == 2 && start[0] == '.' && start[1] == '.')
	{
	  /* Back up to previous component, ignore if at root already.  */
	  if (dest > rpath + 1)
	    while ((--dest)[-1] != '/');
	}
      else
	{
	  size_t new_size;

	  if (dest[-1] != '/')
	    *dest++ = '/';

	  if (dest + (end - start) >= rpath_limit)
	    {
	      ptrdiff_t dest_offset = dest - rpath;

	      if (resolved)
		{
		  errnoSet (ENAMETOOLONG);
		  goto error;
		}
	      new_size = rpath_limit - rpath;
	      if (end - start + 1 > path_max)
		new_size += end - start + 1;
	      else
		new_size += path_max;
	      rpath = realloc (rpath, new_size);
	      rpath_limit = rpath + new_size;
	      if (rpath == NULL)
		return NULL;

	      dest = rpath + dest_offset;
	    }

	  dest = memcpy (dest, start, end - start);
	  dest += (end-start);
	  *dest = '\0';

	  if (stat (rpath, &st) < 0)
	    goto error;

#ifdef VXWORKS_RESOLVES_LINKS
	  if (S_ISLNK (st.st_mode))
	    {
	      char *buf = __alloca (path_max);
	      size_t len;

	      if (++num_links > MAXSYMLINKS)
		{
		  errnoSet (ELOOP);
		  goto error;
		}

	      n = __readlink (rpath, buf, path_max);
	      if (n < 0)
		goto error;
	      buf[n] = '\0';

	      if (!extra_buf)
		extra_buf = __alloca (path_max);

	      len = strlen (end);
	      if ((long int) (n + len) >= path_max)
		{
		  errnoSet (ENAMETOOLONG);
		  goto error;
		}

	      /* Careful here, end may be a pointer into extra_buf... */
	      memmove (&extra_buf[n], end, len + 1);
	      name = end = memcpy (extra_buf, buf, n);

	      if (buf[0] == '/')
		dest = rpath + 1;	/* It's an absolute symlink */
	      else
		/* Back up to previous component, ignore if at root already: */
		if (dest > rpath + 1)
		  while ((--dest)[-1] != '/');
	    }
#endif
	}
    }
  if (dest > rpath + 1 && dest[-1] == '/')
    --dest;
  *dest = '\0';

  return resolved ? memcpy (resolved, rpath, dest - rpath + 1) : rpath;

error:
  if (resolved)
    strcpy (resolved, rpath);
  else
    free (rpath);
  return NULL;
}

/* We don't seem to have any way of reading links */
char *
realpath(char *pathname, char *resolvedname)
{
  if (resolvedname == NULL)
    {
      errnoSet (EINVAL);
      return NULL;
    }

  return canonicalize (pathname, resolvedname);
}
#endif
#endif



int
fd_setsize_bytes()
{
  return FD_SETSIZE/8;
}

void
do_fd_set(int fd, fd_set *fdsetp)
{
  FD_SET(fd, fdsetp);
}

void
do_fd_clr(int fd, fd_set *fdsetp)
{
  FD_CLR(fd, fdsetp);
}

int
do_fd_is_set(int fd, fd_set *fdsetp)
{
  return FD_ISSET(fd,fdsetp);
}

void
do_fd_zero(fd_set *fdsetp)
{
  FD_ZERO(fdsetp);
}

#include "image.h"

LispObj
load_image(char *path)
{
  int fd = open(path, O_RDONLY, 0666);
  LispObj image_nil = 0;
  if (fd > 0) {
    openmcl_image_file_header ih;
    image_nil = load_openmcl_image(fd, &ih);
    /* We -were- using a duplicate fd to map the file; that
       seems to confuse Darwin (doesn't everything ?), so
       we'll instead keep the original file open.
    */
    if (!image_nil) {
      close(fd);
    }
  }
  if (image_nil == 0) {
    fprintf(stderr, "Couldn't load lisp heap image from %s\n", path);
    lisp_exit(-1);
  }
  return image_nil;
}

int
set_errno(int val)
{
  errno = val;
  return -1;
}
