;;;-*- Mode: LISP; Package: CCL -*-

(in-package "CCL")

(eval-when (:compile-toplevel :load-toplevel :execute)
  (require "COCOA-EDITOR")
  (require "PTY"))

(eval-when (:compile-toplevel :execute)
  (setq *readtable* *objc-readtable*))

;;; Setup the server end of a pty pair.
(defun setup-server-pty (pty)
  pty)

;;; Setup the client end of a pty pair.
(defun setup-client-pty (pty)
  ;; Since the same (Unix) process will be reading from and writing
  ;; to the pty, it's critical that we make the pty non-blocking.
  (fd-set-flag pty #$O_NONBLOCK)
  (disable-tty-local-modes pty (logior #$ECHO #$ECHOCTL #$ISIG))
  pty)

(declaim (special *open-editor-documents*)
	 (type list *open-editor-documents*))

(defun new-listener-process (procname input-fd output-fd)
  (make-mcl-listener-process
   procname
   (make-fd-stream
		   input-fd
		   :elements-per-buffer (#_fpathconf
					 input-fd
					 #$_PC_MAX_INPUT))
   (make-fd-stream output-fd :direction :output
				   :elements-per-buffer
				   (#_fpathconf
				    output-fd
				    #$_PC_MAX_INPUT))
   #'(lambda ()
       (let* ((info (find *current-process* *open-editor-documents*
			  :key #'cocoa-editor-info-listener)))
	 (when info
	   (setf (cocoa-editor-info-listener info) nil))))))

(defloadvar *NSFileHandleNotificationDataItem*
    (%get-ptr (foreign-symbol-address "_NSFileHandleNotificationDataItem")))

(defloadvar *NSFileHandleReadCompletionNotification*
    (%get-ptr (foreign-symbol-address "_NSFileHandleReadCompletionNotification")))



(def-objc-class "lisplistenerwindowcontroller" "lispeditorwindowcontroller"
  filehandle				;Filehandle for I/O
  (clientfd :int)			;Client (listener)'s side of pty
  (outpos :unsigned)			;Position in textview buffer
  userta				;Typing attributes for user input
  systa					;Typing attributes for system output
  usercolor				;Text foreground color for user input
  )

(define-objc-method ("windowDidLoad" "lisplistenerwindowcontroller")
    (:void)
  ;(#_NSLog #@"windowDidLoad (controller)")
  (multiple-value-bind (server client) (ignore-errors (open-pty-pair))
    (when server
      (let* ((fh [[(@class "NSFileHandle") "alloc"]
		  "initWithFileDescriptor:closeOnDealloc:"
		  :unsigned-fullword (setup-server-pty server)
		  :unsigned-byte #$YES]))
	(setq filehandle fh)
	(setq clientfd (setup-client-pty client))

	[[(@class "NSNotificationCenter") "defaultCenter"]
	 "addObserver:selector:name:object:"
	 :id self
	 :<SEL> (@selector "gotData")
	 (* :char) *NSFileHandleReadCompletionNotification*
	 :id fh]
	[fh "readInBackgroundAndNotify"]))))

(define-objc-method ("gotData" "lisplistenerwindowcontroller")
    (:id notification :void)
  (let* ((data [[notification "userInfo"]
		"objectForKey:" :address *NSFileHandleNotificationDataItem*])
	 (tv textview)
	 (fh filehandle))
    ;(#_NSLog #@"Gotdata: tv = %@, fh = %@" :address tv :address fh)
    (unless (%null-ptr-p tv)
      (let* ((buffer-text [tv "textStorage"])
	     (s [[(@class "NSString") "alloc"]
	       "initWithData:encoding:"
	       :id data :unsigned #$NSASCIIStringEncoding])
	     (str [[(@class "NSAttributedString") "alloc"]
		   "initWithString:attributes:"
		   :address s
		   :address (get-objc-instance-variable self "systa")]))
	[buffer-text "appendAttributedString:" :address str]
	;; Again, with the NSRange already ...
	(let* ((textlen [buffer-text "length" :unsigned-fullword]))
	  [tv "scrollRangeToVisible:"
	      :unsigned-fullword textlen
	      :unsigned-fullword 0
	      :void]
	  (setq outpos textlen))
	[str "release"]))
    [self "updatePackageName"]
    [fh "readInBackgroundAndNotify"]))


	     
(define-objc-method ("updatePackageName" "lisplistenerwindowcontroller")
     (:void)
  (let* ((info (info-from-controller self))
	 (proc (if info (cocoa-editor-info-listener info)))
	 (package (if proc (ignore-errors (symbol-value-in-process
					   '*package*
					   proc))))
	 (name (if (typep package 'package)
		 (shortest-package-name package)
		 "")))
    (with-cstrs ((name name))
      (let* ((string [(@class "NSString")
		      "stringWithCString:" (* :char) name]))
	[self "displayPackageName:" :id string]))))
      

    
;;; The lisplistenerwindowcontroller is the textview's "delegate": it
;;; gets consulted before certain actions are performed, and can
;;; perform actions on behalf of the textview.

(define-objc-method ("textView:shouldChangeTextInRange:replacementString:"
		     "lisplistenerwindowcontroller")
     (:id tv :<NSR>ange range  :id replacement-string :<BOOL>)
  (declare (ignorable replacement-string))
  (if (< (pref range :<NSR>ange.location) outpos)
    (progn
      (#_NSBeep)			;Overkill, maybe.
      #$NO)
    (progn
      [tv "setTypingAttributes:" :address userta]
      #$YES)))


;;; Action methods implemented by the controller (in its role as the
;;; textview's delegate).


(define-objc-method ("insertNewline:" "lisplistenerwindowcontroller")
    (:id tv)
  (let* ((textbuf [tv "textStorage"])
	 (textlen [textbuf "length" :unsigned-fullword])
	 (textstring [tv "string"]))
    (rlet ((r :<NSR>ange))
      [tv "selectedRange" (:-> r)]
      (let* ((curpos (pref r :<NSR>ange.location))
	     (curlen (pref r :<NSR>ange.length)))
	(cond ((>= curpos outpos)
	       ;; Insert the newline at the end of any selection.
	       (incf curpos (pref r :<NSR>ange.length))
	       (rlet ((newsel :<NSR>ange :location curpos :length 0))
		 [tv "setSelectedRange:" :<NSR>ange newsel])
	       [tv "insertNewline:" :address self]
	       (incf curpos)
	       (incf textlen)
	       (when (= curpos textlen)
		 (let* ((sendlen (- textlen outpos))
			(sendstring (rlet ((subrange :<NSR>ange
					     :location outpos
					     :length sendlen))
				      [textstring
				       "substringWithRange:"
				       :<NSR>ange subrange])))
		   (setf (pref r :<NSR>ange.location) 0
			 (pref r :<NSR>ange.length) sendlen)
		   (multiple-value-bind (ok second-value)
		       (balanced-expressions-in-range-forward r sendstring)
		     (if ok
		       (if second-value
			 (progn
			   [self "sendString:"
			    :address sendstring]
			   (setq outPos textlen)))
		       (if second-value
			 (#_NSBeep)))))))
	      ;; If there's a selection, copy it to the end of the
	      ;; buffer, then move to the end of the buffer.
	      ((> curlen 0)
	       (rlet ((endrange :<NSR>ange :location textlen :length 0))
		 [tv "setSelectedRange:" :<NSR>ange endrange]
		 [tv "insertText:"
		     :address
		     [textstring "substringWithRange:" :<NSR>ange r]]
		 (setf (pref endrange :<NSR>ange.location)
		       [textbuf "length" :unsigned-fullword])
		 [tv "scrollRangeToVisible:" :<NSR>ange endrange]))
	      ;; No selection, insertion point is before outpos (in
	      ;; history or in output.  If in history, copy history
	      ;; item to end of buffer, otherwise, do nothing.
	      (t
	       (rlet ((lr :<NSR>ange)
		      (fullrange :<NSR>ange :location 0 :length textlen))
		 (let* ((attr
			 [textbuf
			    "attribute:atIndex:longestEffectiveRange:inRange:"
			    :address #@"NSColor"
			    :unsigned-fullword curpos
			    :address lr
			    :<NSR>ange fullrange]))
		   (when (eql [attr "isEqual:" :address usercolor :unsigned-byte]
			      #$YES)
		     (let* ((history-start (pref lr :<NSR>ange.location))
			    (history-len (pref lr :<NSR>ange.length)))
		       (when (eql [textstring "characterAtIndex:"
					      :unsigned-fullword
					      (+ history-start
						 (1- history-len))
					      :unsigned-halfword]
				  (char-code #\NewLine))
			 (decf (pref lr :<NSR>ange.length)))
		       (unless (eql 0 history-len)
			 (setf (pref fullrange :<NSR>ange.location)
			       textlen
			       (pref fullrange :<NSR>ange.length)
			       0)
			 [tv "setSelectedRange:" :<NSR>ange fullrange]
			 [tv "insertText:"
			     :address
			     [textstring "substringWithRange:"
					 :<NSR>ange lr]]
			 (setf (pref fullrange :<NSR>ange.location)
			       [textbuf "length" :unsigned-fullword])
			 [tv "scrollRangeToVisible:" :<NSR>ange fullrange]))))))))))
    self)

;;; Force a break in the listener process.
(define-objc-method ("interrupt:" "lisplistenerwindowcontroller")
    (:id tv)
  (declare (ignore tv))
  (let* ((info (info-from-controller self))
	 (proc (if info (cocoa-editor-info-listener info))))
    (when proc (force-break-in-listener proc))
    self))

;;; This exists solely for debugging.
(define-objc-method ("logAttrs:" "lisplistenerwindowcontroller") (:id tv)
  (rlet ((lr :<NSR>ange)
	 (selection :<NSR>ange))
    [tv "selectedRange" (:-> selection)]
    (let* ((textbuf [tv "textStorage"])
	   (length [textbuf "length" :unsigned-fullword])
	   (attr
	    (rlet ((fullrange :<NSR>ange :location 0 :length length))
	    [textbuf "attributesAtIndex:longestEffectiveRange:inRange:"
		     :unsigned-fullword (pref selection :<NSR>ange.location)
		     :address lr
		     :<NSR>ange fullrange])))
      (#_NSLog #@"Attr = %@, range = [%d,%d]"
	       :address attr
	       :unsigned-fullword (pref lr :<NSR>ange.location)
	       :unsigned-fullword (pref lr :<NSR>ange.length))
      self)))

;;; If we're at the end of the buffer and at the start of a line (either
;;; at outpos or after a newline), send an EOF (0 bytes of data) to the
;;; listener.  Otherwise, have the textview do a "deleteForward:"
(define-objc-method ("deleteForward:" "lisplistenerwindowcontroller") (:id tv)
  ;(#_NSLog #@"In deleteForwardOrSendEOF:")
  (rlet ((selection :<NSR>ange))
    [tv "selectedRange" (:-> selection)]
    (let* ((textbuf [tv "textStorage"])
	   (length [textbuf "length" :unsigned-fullword]))
      (if (and (eql length (pref selection :<NSR>ange.location))
	       (or (eql outpos length)
		   (and (> length 1)
			(= [textbuf "charAtIndex:"
				    :unsigned-fullword (1- length)
				    :unsigned-halfword]
			   (char-code #\NewLine)))))
	(%stack-block ((buf 1))
	  (setf (%get-byte buf 0) (logand (char-code #\d) #x1f))
	  [filehandle "writeData:"
		      :address 
		      [(@class "NSData") "dataWithBytes:length:"
		       :address buf
		       :unsigned-fullword 1]]
	  [filehandle "synchronizeFile"]
	  ;(#_NSLog #@"wrote ctrl-d packet")
	  )
	[tv "deleteForward:" :address self])
      self)))

(define-objc-method ("addModeline:" "lisplistenerwindowcontroller")
    (:id tv)
  (declare (ignore tv))
  self
  )

(define-objc-method ("reparseModeline:" "lisplistenerwindowcontroller")
    (:id tv)
  (declare (ignore tv))
  self
  )

(define-objc-method ("dealloc" "lisplistenerwindowcontroller")  (:void)
  [[(@class "NSNotificationCenter") "defaultCenter"]
   "removeObserver:" :address self]
  [:super "dealloc"])


(define-objc-method ("sendString:" "lisplistenerwindowcontroller")
    (:id string :void)
  [filehandle
   "writeData:"
   :address
   [string
    "dataUsingEncoding:allowLossyConversion:"
    :unsigned-fullword #$NSASCIIStringEncoding
    :unsigned-byte #$YES]]
)
  
;;; The lisplistenerdocument class.


(def-objc-class "lisplistenerdocument" "lispeditordocument")

(define-objc-class-method ("topListener" "lisplistenerdocument") (:id)
  (let* ((all-documents [*NSApp* "orderedDocuments"]))
    (dotimes (i [all-documents "count" :unsigned-fullword] (%null-ptr))
      (let* ((doc [all-documents "objectAtIndex:" :unsigned-fullword i]))
	(when (eql [doc "class"] self)
	  (return doc))))))

(defun symbol-value-in-top-listener-process (symbol)
  (let* ((listenerdoc [(@class "lisplistenerdocument") "topListener"])
	 (info (info-from-document listenerdoc))
	 (process (if info (cocoa-editor-info-listener info))))
     (if process
       (ignore-errors (symbol-value-in-process symbol process))
       (values nil t))))
  


(define-objc-method ("isDocumentEdited" "lisplistenerdocument")  (:<BOOL>)
  #$NO)

(define-objc-method ("makeWindowControllers" "lisplistenerdocument")   (:void)
  (let* ((nibname [self "windowNibName"])
	 (controller [[(@class "lisplistenerwindowcontroller") "alloc"]
		      "initWithWindowNibName:owner:"
		      :address nibname
		      :address self]))
    [self "addWindowController:" :address controller]
    [controller "release"]))


      
#+nil    
(define-objc-method ("dataRepresentationOfType:" "lisplistenerdocument")
    ((* :char) type)
  (declare (ignorable type))
  (#_NSLog #@"dataRepresentationOfType: %s" :address type)
  (rlet ((fullrange :<NSR>ange :location 0
		    :length [[textview "textStorage"]
			     "length" :unsigned-fullword]))
    [textview "RTFDFromRange:" :<NSR>ange fullrange]))

#+nil
(define-objc-method ("loadDataRepresentation:ofType:" "lisplistenerdocument")
    (:id data (* :char) type :<BOOL>)
  (declare (ignorable type))
  ;(#_NSLog #@"loadDataRepresentation:ofType (listener) type = %s" :address type)
  (setq filedata data)
  (if (%null-ptr-p data)
    #$NO
    #$YES))



(defloadvar *cocoa-listener-count* 0)

(define-objc-method ("windowControllerDidLoadNib:""lisplistenerdocument")
     (:id acontroller :void)
  ;;(#_NSLog #@"windowControllerDidLoadNib (listener document)")
  [:super "windowControllerDidLoadNib:" :address acontroller]
  ;; We'll use attribute-change information to distinguish user
  ;; input from system output.  Be fascist about letting the
  ;; user change anything.
  [textview "setRichText:" :unsigned-byte #$NO]
  [textview "setUsesFontPanel:" :unsigned-byte #$NO]
  (let* ((listener-name (if (eql 1 (incf *cocoa-listener-count*))
			  "Listener"
			  (format nil
				  "Listener-~d" *cocoa-listener-count*)))
	 (info (info-from-document self)))
    (setf (cocoa-editor-info-listener info)
	  (let* ((tty (%ptr-to-int (get-objc-instance-variable
				    acontroller
				    "clientfd"))))
	    (new-listener-process listener-name tty tty)))
    [self "setFileName:" :address (%make-nsstring listener-name)])
  (set-objc-instance-variable acontroller "textview" textview)
  (set-objc-instance-variable acontroller "echoarea" echoarea)
  (set-objc-instance-variable acontroller "packagename" packagename)
  (let* ((userta [[textview "typingAttributes"] "retain"])
	 (systa (create-text-attributes :color [(@class "NSColor")
						 "blueColor"])))
    (set-objc-instance-variable acontroller "userta" userta)
    (set-objc-instance-variable acontroller "usercolor"
				[userta "valueForKey:"
					:address #@"NSColor"])
    (set-objc-instance-variable acontroller "systa" systa))
  [textview "setDelegate:" :address acontroller]
  (unless (%null-ptr-p filedata)
    (rlet ((nullrange :<NSR>ange :location 0 :length 0))
	  [textview "replaceCharactersInRange:withRTFD:"
		    :<NSR>ange nullrange
		    :address filedata])))




